/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/launch/JLinRunner.java#3 $
 */

package com.sap.tc.jtools.jlint.eclipse.launch;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;

import com.sap.tc.jtools.jlint.eclipse.JlinPlugin;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jlint.eclipse.util.JLinSetupCache;
import com.sap.tc.jtools.jtci.Dispatcher;
import com.sap.tc.jtools.jtci.PerformResponse;
import com.sap.tc.jtools.jtci.interfaces.Listener;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.util.xml.XMLTool;

public class JLinRunner implements Runnable, Listener, IRunnableWithProgress {

  private IProgressMonitor monitor;
  private MetaRequestManager requestManager;
  private MetaMarkerManager markerManager;
  private List eclipseResponses = new ArrayList();
  private String requestDir;
  private String resultDir;

  public JLinRunner(MetaRequestManager requestManager) {
    this.requestManager = requestManager;
  }

  public JLinRunner(
    MetaRequestManager requestManager,
    IProgressMonitor monitor,
    String requestDir,
    String responseDir) {
    this.monitor = monitor;
    this.requestManager = requestManager;
    this.requestDir = requestDir;
    this.resultDir = responseDir;
  }

  public void run() {
    prepare();
    Dispatcher dispatcher = getDispatcher();
    EclipseRequest[] requests = requestManager.getRequests();
    try {
      for (int i = 0; i < requests.length; i++) {
        eclipseResponses.add(
          new EclipseResponse(
            requests[i],
            dispatcher.processRequest(requests[i])));
      }
    } catch (Exception e) {
      ExceptionHandler.handle(e);
      if (monitor != null) {
        monitor.done();
      }
      return;
    }
    if (monitor != null && monitor.isCanceled())
      return;
    finish();
  }
  private void finish() {
    persistResponses();

    EclipseResponse[] responses =
      (EclipseResponse[]) eclipseResponses.toArray(new EclipseResponse[0]);

    markerManager =
      new MetaMarkerManager(requestManager.getSelectedObjects(), responses);

    markerManager.createMarkers();
    if (monitor != null) {
      monitor.done();
      markerManager.showFinishDialog(requestManager.testObjectCount());
    }
    showTasksView();

  }

  private void prepare() {
    if (requestDir != null) {
      requestManager.persistRequests(new File(requestDir));
    }
    initMonitor();
  }

  public static void showTasksView() {
    final Display dsp = Display.getDefault();
    if (dsp != null) {
      Runnable runnable = new Runnable() {
        public void run() {
          IWorkbenchPage page =
            PlatformUI
              .getWorkbench()
              .getActiveWorkbenchWindow()
              .getActivePage();
          if (page != null) {
            try {
              page.showView("org.eclipse.ui.views.TaskList"); //$NON-NLS-1$
            } catch (PartInitException ex) {
              JlinPlugin.log(ex);
            }
          }
        }
      };
      dsp.asyncExec(runnable);
    }
  }

  private void persistResponses() {
    if (resultDir == null || !new File(resultDir).isDirectory())
      return;
    BufferedWriter writer = null;
    try {
      writer =
        new BufferedWriter(new FileWriter(new File(resultDir, "results.xml")));
      XMLTool.writeDocument(getMergedResponse().toStructureTree(), writer);
    } catch (IOException ioe) {
      JlinPlugin.log(ioe);
      ExceptionHandler.handle(ioe);
    } finally {
      if (writer != null) {
        try {
          writer.close();
        } catch (IOException e) {
          ExceptionHandler.handle(e);
        }
      }
    }
  }

  private PerformResponse getMergedResponse() {
    List allResults = new ArrayList();
    for (Iterator iter = eclipseResponses.iterator(); iter.hasNext();) {
      EclipseResponse reqResp = (EclipseResponse) iter.next();
      PerformResponse response = reqResp.getResponse();
      if (response != null && response.getResults() != null) {
        allResults.addAll(Arrays.asList(response.getResults()));
      }
    }
    ResultInterface[] processedResults =
      JLinSetupCache.getDispatcher().processResults(
        (ResultInterface[]) allResults.toArray(new ResultInterface[0]));
    PerformResponse mergedResponse = new PerformResponse("");
    mergedResponse.addResults(processedResults);
    return mergedResponse;
  }

  private void initMonitor() {
    if (monitor != null) {
      String currentTaskName = TextKeyLaunch.JLIN_RUNNING.getText();
      monitor.beginTask(currentTaskName, requestManager.testObjectCount());
    }

  }

  /**
   * Method getDispatcher.
   * @return Dispatcher
   */
  private Dispatcher getDispatcher() {
    Dispatcher disp = JLinSetupCache.getDispatcher();
    disp.setListener(this);
    return disp;
  }

  /**
   * callback method  from JLin's Listener interface which is called
   * after each TestObject (= java file) that has been processed with the
   * results for this TestObject.
   * used for progress indicator and result displaying.
   */

  public void processedTestObject(ResultInterface[] results) {
  }

  public boolean isCanceled() {
    if (monitor == null)
      return false;
    return monitor.isCanceled();
  }

  public void processedTestObjectComponent() {
    if (monitor != null)
      monitor.worked(1);
  }

  public void run(IProgressMonitor monitor) {
    this.monitor = monitor;
    run();
  }

  /**
   * @return
   */
  public MetaMarkerManager getMarkerManager() {
    return markerManager;
  }

  /**
   * @return
   */
  public EclipseResponse[] getResponses() {
    return (EclipseResponse[]) eclipseResponses.toArray(new EclipseResponse[0]);
  }

}
