/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/config/TestSetTool.java#2 $
 */

package com.sap.tc.jtools.jlint.eclipse.config;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import com.sap.tc.jtools.jlint.eclipse.JlinPlugin;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jlint.eclipse.util.JLinSetupCache;
import com.sap.tc.jtools.jtci.Test;
import com.sap.tc.jtools.jtci.TestSet;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.exceptions.InvalidTestSpecificationException;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.jtci.interfaces.TestDescriptionInterface;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * Offers methods to read & write variants as well to convert 
 * the Jlint description format to a variant.
 * start a parameter dialog.
 * @version   1.0, $Date: 2004/03/22 $
 * @author    BPL Tools
 */

final public class TestSetTool {

  public static final String DEFAULT_NAME = "DEFAULT";
  static final String TESTSET_EXTENSION = ".testset.xml"; //$NON-NLS-1$

  // no instanciation
  private TestSetTool() {
  }

  /**
   * Reads a variant from the default location.
   */
  public static TestSet readTestSet(String name)
    throws IOException, BadTreeException {
    if (DEFAULT_NAME.equals(name)) {
      return getDefault();
    } else {
      return readTestSet(name, JlinPlugin.getBaseDir().toFile());
    }
  }

  /**
   * Reads a variant from a free chooseable location.
   */
  public static TestSet readTestSet(String name, File pBaseDir)
    throws IOException, BadTreeException {
    if (null == pBaseDir) {
      throw new NullPointerException("baseDir"); //$NON-NLS-1$
    } else if (!pBaseDir.isDirectory()) {
      throw new FileNotFoundException(pBaseDir.toString());
    }
    FileReader reader =
      new FileReader(new File(pBaseDir, name + TESTSET_EXTENSION));
    try {
      TestSet testSet = new TestSet(XMLTool.parseReader(reader));
      return mergeWithToolDescr(testSet);
    } catch (Exception e) {
      ExceptionHandler.handle(e);
      return null;
    } finally {
      reader.close();
    }
  }

  /**
   * Method mergeWithToolDescr.
   * @param testSet
   * @return TestSet
   */
  private static TestSet mergeWithToolDescr(TestSet testSet) {
    Test[] tests = testSet.getTests();
    TestDescriptionInterface[] existingTests =
      JLinSetupCache.getDispatcher().getTests();

    // make sure all tests in testset.xml exist in tooldescr.
    for (int i = 0; i < tests.length; i++) {
      boolean found = false;
      for (int j = 0; j < existingTests.length && !found; j++) {
        if (tests[i].getName().equals(existingTests[j].getName())) {
          found = true;
          // set parameters to default values if not set 
          ParameterInterface[] defaultParams =
            existingTests[j].getInputParameters();
          List mergedParams = new ArrayList();
          ParameterInterface mergedParam = null;
          for (int k = 0; k < defaultParams.length; k++) {
            ParameterInterface correspondingParam =
              tests[i].getParameter(defaultParams[k].getName());
            if (correspondingParam == null) {
              mergedParam = defaultParams[k];
            } else if (correspondingParam.getValue() == null) {
              correspondingParam.setValue(defaultParams[k].getValue());
              mergedParam = correspondingParam;
            } else {
              mergedParam = correspondingParam;
            }
            mergedParams.add(mergedParam);
          }
          tests[i].setParameters(
            (ParameterInterface[]) mergedParams.toArray(
              new ParameterInterface[0]));
          }
      }
      try {
        if (!found) {
          throw new InvalidTestSpecificationException(
            "test " + tests[i].getName() + " doesn\'t exist!");
        }
      } catch (InvalidTestSpecificationException e) {
        testSet.removeTest(tests[i]);
        ExceptionHandler.handle(e);
      }
    }

    // add tests that are in tooldescr, but not in testset.xml

    for (int i = 0; i < existingTests.length; i++) {
      if (testSet.getTest(existingTests[i].getName()) == null) {
        // test is missing, so add it as inactive
        testSet.addTest(
          new Test(
            existingTests[i].getName(),
            existingTests[i].getInputParameters(),
            false));
      }
    }
    return testSet;
  }

  /**
   * Writes a variant to the default location.
   */

  static void writeTestSet(TestSet pTestSet) throws IOException {
    writeTestSet(pTestSet, JlinPlugin.getBaseDir().toFile());
  }

  /**
   * Writes a variant to a free chooseable location.
   */
  static void writeTestSet(TestSet pTestSet, File pBaseDir)
    throws IOException {
    if (null == pBaseDir) {
      throw new NullPointerException("baseDir"); //$NON-NLS-1$
    } else if (!pBaseDir.isDirectory()) {
      throw new FileNotFoundException(pBaseDir.toString());
    }
    FileWriter writer =
      new FileWriter(
        new File(pBaseDir, pTestSet.getName() + TESTSET_EXTENSION));
    try {
      XMLTool.writeDocument(pTestSet.toStructureTree(), writer);
    } catch (IOException e) {
      ExceptionHandler.handle(e);
    } finally {
      writer.close();
    }
  }

  /**
   * Deletes a variant from the default location.
   */
  static void purgeTestSet(TestSet pTestSet) throws IOException {
    purgeTestSet(pTestSet.getName(), JlinPlugin.getBaseDir().toFile());
  }

  static void purgeTestSet(String pTestSetName) throws IOException {
    purgeTestSet(pTestSetName, JlinPlugin.getBaseDir().toFile());
  } /**
                        * Deletes a variant from a free chooseable location.
                        */
  static void purgeTestSet(String pTestSetName, File pBaseDir)
    throws IOException {
    if (null == pBaseDir) {
      throw new NullPointerException("baseDir"); //$NON-NLS-1$
    } else if (!pBaseDir.isDirectory()) {
      throw new FileNotFoundException(pBaseDir.toString());
    }

    File varFile = new File(pBaseDir, pTestSetName + TESTSET_EXTENSION);
    if (!varFile.isFile()) {
      throw new IOException("variant is not deleteable: " + varFile.toString()); //$NON-NLS-1$
    }
    varFile.delete();
  } /**
                        * Scan all testsets in the default directory.
                        */
  static String[] scanTestSetNames() throws IOException {
    return scanTestSetNames(JlinPlugin.getBaseDir().toFile());
  } /**
                        * Scan all testsets from a free chooseable location.
                        */
  static String[] scanTestSetNames(File pBaseDir) throws IOException {
    if (null == pBaseDir) {
      throw new NullPointerException("baseDir"); //$NON-NLS-1$
    } else if (!pBaseDir.isDirectory()) {
      throw new FileNotFoundException(pBaseDir.toString());
    }

    FilenameFilter filter = new FilenameFilter() {
      public boolean accept(File pPath, String pName) {
        return pName.endsWith(TESTSET_EXTENSION);
      }
    };
    File[] content = pBaseDir.listFiles(filter);
    if (null == content || 0 == content.length) {
      return null;
    }

    String[] testSetNames = new String[content.length];
    for (int i = 0; i < content.length; i++) {
      String filename = content[i].getName();
      testSetNames[i] =
        filename.substring(0, filename.length() - TESTSET_EXTENSION.length());
    }
    return testSetNames;
  }

  static TestSet getDefault() {
    File testSetFile = new File(JlinPlugin.getInstallDir(), "data/testconfig/testconfig.xml"); //$NON-NLS-1$
    try {
      FileReader reader = new FileReader(testSetFile);
      try {
        TestSet defaultTestSet = new TestSet(XMLTool.parseReader(reader));
        return mergeWithToolDescr(defaultTestSet);
      } catch (Exception e) {
        ExceptionHandler.handle(e);
      } finally {
        reader.close();
      }
    } catch (IOException e) {
      ExceptionHandler.handle(e);
    }
    // in case of exception: generate default testset from tooldescriptor
    return generateFromToolDescr(DEFAULT_NAME);
  }

  /**
   * Method getEmpty.
   * @param DEFAULT_NAME
   * @return TestSet
   */
  public static TestSet generateFromToolDescr(String name) {
    TestDescriptionInterface[] testdescr =
      JLinSetupCache.getDispatcher().getTests();
    Vector tests = new Vector();
    for (int i = 0; i < testdescr.length; i++) {
      tests.add(
        new Test(
          testdescr[i].getName(),
          testdescr[i].getInputParameters(),
          false));
    }
    return new TestSet(name, (Test[]) tests.toArray(new Test[tests.size()]));
  }

}