/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/config/TestSetDialog.java#3 $
 */

package com.sap.tc.jtools.jlint.eclipse.config;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;

import org.eclipse.swt.SWT;

import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.ISharedImages;

import com.sap.tc.jtools.jlint.eclipse.JlinPlugin;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jlint.eclipse.util.JLinSetupCache;
import com.sap.tc.jtools.jtci.Test;
import com.sap.tc.jtools.jtci.TestSet;
import com.sap.tc.jtools.jtci.TestTree;
import com.sap.tc.jtools.jtci.TestTreeNode;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.interfaces.TestDescriptionInterface;

/**
 * This dialog is the anchor for the test maintainance of a single variant. The activation 
 * state of each test is all controlled directely while another dialog can be started for
 * parameter maintainance.
 * @version   $Date: 2004/07/30 $
 * @author    BPL Tools
 */

public class TestSetDialog extends Dialog implements IMenuListener {

  private final TestSet fTestSet;
  private boolean readOnly;

  private Image fIconArgsMissing = null;
  private Image fIconArgsNone = null;
  private Image fIconArgsOkay = null;
  private Image fIconFolder = null;

  private TestTree fDescriptorTree =
    JLinSetupCache.getDispatcher().getTestTree();
  private Tree fTree = null;

  private Shell fShell = null;

  static private final int STATE_CHECKED_NONE = 1;
  static private final int STATE_CHECKED_ALL = 2;
  static private final int STATE_CHECKED_MIXED = 3;

  /**
   * Constructor for TestSetDialog.
   * @param pParentShell maybe null
   * @param pVariant
   */
  public TestSetDialog(
    Shell pParentShell,
    TestSet pTestSet,
    boolean readOnly) {
    super(pParentShell);
    fTestSet = pTestSet;
    this.readOnly = readOnly;
  }

  /**
   * Called during dialog.open() with a new shell in order to create the 
   * dialog area.
   * the super method creates the cancel and ok button. The returned control 
   * is a composite so we cast it to insert our test visualization.
   */
  protected Control createDialogArea(Composite pBaseComposite) {
    fShell = pBaseComposite.getShell();
    if (readOnly) {
      fShell.setText(
        TextKeyCfg.VARIANT_SHOW_TITLE.getFormattedText(fTestSet.getName()));
    } else {
      fShell.setText(
        TextKeyCfg.VARIANT_EDIT_TITLE.getFormattedText(fTestSet.getName()));
    }

    Composite composite = new Composite(pBaseComposite, SWT.NONE);
    {
      {
        GridLayout layout = new GridLayout();
        layout.marginHeight =
          convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_MARGIN);
        layout.marginWidth =
          convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_MARGIN);
        layout.verticalSpacing =
          convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_SPACING);
        layout.horizontalSpacing =
          convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_SPACING);
        composite.setLayout(layout);
      }
      {
        GridData gd = new GridData(GridData.FILL_BOTH);
        Point size = computeSize();
        gd.widthHint = size.x;
        gd.heightHint = size.y;
        composite.setLayoutData(gd);
      }
      composite.setFont(pBaseComposite.getFont());
    }
    createIcons(composite);
    createTree(composite);
    return composite;
  }

  private Point computeSize() {
    Rectangle r = Display.getCurrent().getClientArea();
    Point size = new Point(r.width, r.height);

    if (1000 > r.height) {
      if (400 > r.height) {
        size.y = 150;
      } else {
        size.y = 150 + r.height / 3;
      }
    } else {
      size.y = 350 + r.height / 5;
    }

    if (1000 > r.width) {
      if (400 > r.width) {
        size.x = 150;
      } else {
        size.x = 150 + r.width / 5;
      }
    } else {
      size.x = 300 + r.width / 15;
    }
    return size;
  }

  /**
   * Allocates the icons used and ensure automatic disposal.
   */
  private void createIcons(Composite pParentComposite) {
    Display display = fShell.getDisplay();

    fIconArgsMissing = IconKeyCfg.TEST_PARAM_MISS.create(display);
    fIconArgsNone = IconKeyCfg.TEST_PARAM_NONE.create(display);
    fIconArgsOkay = IconKeyCfg.TEST_PARAM_OKAY.create(display);
    fIconFolder =
      JlinPlugin.getDefault().getWorkbench().getSharedImages().getImage(
        ISharedImages.IMG_OBJ_FOLDER);
    DisposeListener dl = new DisposeListener() {
      public void widgetDisposed(DisposeEvent e) {
        dispose();
      }
    };
    pParentComposite.addDisposeListener(dl);
  }

  /**
   * Free the icons allocated associated with this dialog.
   **/
  private void dispose() {
    if (null != fIconArgsMissing && !fIconArgsMissing.isDisposed()) {
      fIconArgsMissing.dispose();
      fIconArgsMissing = null;
    }
    if (null != fIconArgsOkay && !fIconArgsOkay.isDisposed()) {
      fIconArgsOkay.dispose();
      fIconArgsOkay = null;
    }
    if (null != fIconArgsNone && !fIconArgsNone.isDisposed()) {
      fIconArgsNone.dispose();
      fIconArgsNone = null;
    }
  }

  /**
   * Creates a tree which visualizes all tests contained in the variant. 
   * Also install a selection listener to control the behavior of the 
   * selection box versus the activation state of the test.
   */
  private void createTree(Composite pParentComposite) {

    fTree = new Tree(pParentComposite, SWT.SINGLE | SWT.CHECK);
    fTree.setLayoutData(new GridData(GridData.FILL_BOTH));

    fillTree();
    initMenu(fTree);

    // add various listeners

    if (readOnly) {
      // install Listener which reverts the checkstate modification
      Listener ls = new Listener() {
        public void handleEvent(Event event) {
          if (SWT.CHECK == event.detail && event.item instanceof TreeItem) {
            TreeItem ti = (TreeItem) event.item;
            ti.setChecked(!ti.getChecked());
          }
        }
      };
      fTree.addListener(SWT.Selection, ls);
    } else {
      // install listener which either updates root from children or vice versa
      {
        Listener ls = new SelectionChangedListener();
        fTree.addListener(SWT.Selection, ls);
      }
      {
        Listener ls = new SpaceAndMouseListener();
        fTree.addListener(SWT.MouseDoubleClick, ls);
        fTree.addListener(SWT.KeyDown, ls);
      }
    }
  }

  /**
   * Build the test tree content.
   **/
  private void fillTree() {
    TreeItem root = new TreeItem(fTree, SWT.SINGLE);
    descendTree(fDescriptorTree, root);
    setSelectionFromChildren(root);
  }

  /**
   * Method descendTree.
   * @param fDescriptorTree
   * @param root
   */
  private void descendTree(TestTreeNode pDescrNode, TreeItem pGuiNode) {
    if (pDescrNode.getNodeType() == TestTreeNode.TREE) {
      TestTree descrTree = (TestTree) pDescrNode;
      pGuiNode.setText(descrTree.getName());
      pGuiNode.setData(descrTree);
      pGuiNode.setImage(fIconFolder);
      TestTreeNode[] children = descrTree.getNodes();
      for (int i = 0; i < children.length; i++) {
        descendTree(children[i], new TreeItem(pGuiNode, SWT.SINGLE));
      }
    } else if (pDescrNode.getNodeType() == TestTreeNode.TEST) {
      // leaf node
      TestDescriptionInterface testDescr =
        (TestDescriptionInterface) pDescrNode;
      Test test = fTestSet.getTest(testDescr.getName());
      if (test != null) {
        pGuiNode.setChecked(test.isActive() && test.paramsOkay());
      } else {
        // add as inactive
        test =
          new Test(testDescr.getName(), testDescr.getInputParameters(), false);
        fTestSet.addTest(test);
        pGuiNode.setChecked(false);
      }
      pGuiNode.setText(test.getName());
      pGuiNode.setData(test);

      // set icons
      if (0 == test.getParameters().length) {
        pGuiNode.setImage(fIconArgsNone);
      } else if (test.paramsOkay()) {
        pGuiNode.setImage(fIconArgsOkay);
      } else {
        pGuiNode.setImage(fIconArgsMissing);
      }
    }
    // we can only expand if all children have been added
    pGuiNode.setExpanded(true);
  }

  void setSelectionFromChildren(TreeItem pBaseNode) {
    switch (setCheckStateFromChildren(pBaseNode)) {
      case STATE_CHECKED_ALL :
        pBaseNode.setChecked(true);
        pBaseNode.setGrayed(false);
        break;
      case STATE_CHECKED_MIXED :
        pBaseNode.setChecked(true);
        pBaseNode.setGrayed(true);
        break;
      default :
        pBaseNode.setChecked(false);
        pBaseNode.setGrayed(false);

    }
  }

  /**
   * update icons and checkboxes which indicate state of parameters of a test
   */
  private void updateTree() {
    updateAllItems(fTree.getItems()[0]); //root item
    setCheckStateFromChildren(fTree.getItems()[0]);
  }

  private void updateAllItems(TreeItem item) {
    if (item.getData() instanceof Test) { //leaf
      Test test = (Test) item.getData();
      if (0 == test.getParameters().length) {
        item.setImage(fIconArgsNone);
      } else {
        if (test.paramsOkay()) {
          item.setImage(fIconArgsOkay);
        } else {
          test.setActive(false);
          item.setImage(fIconArgsMissing);
          item.setChecked(false);
        }
      }
    } else {
      TreeItem[] children = item.getItems();
      for (int i = 0; i < children.length; i++) {
        updateAllItems(children[i]);
      }
    }
  }

  /**
   * traverse tree and set all parents' check state according to children
   */
  int setCheckStateFromChildren(TreeItem pItem) {
    if (pItem.getData() instanceof Test) {
      return pItem.getChecked() ? STATE_CHECKED_ALL : STATE_CHECKED_NONE;
    } else if (pItem.getData() instanceof TestTree) {
      TreeItem[] children = pItem.getItems();
      int status = 0;
      for (int i = 0; i < children.length; i++) {
        status = status | setCheckStateFromChildren(children[i]);
      }
      if (status == 0) { // there were no children
        status = STATE_CHECKED_NONE;
      }
      switch (status) {
        case STATE_CHECKED_ALL :
          pItem.setChecked(true);
          pItem.setGrayed(false);
          break;
        case STATE_CHECKED_NONE :
          pItem.setChecked(false);
          pItem.setGrayed(false);
          break;
        case STATE_CHECKED_MIXED :
          pItem.setChecked(true);
          pItem.setGrayed(true);
          break;
      }
      return status;
    }
    return STATE_CHECKED_NONE;
  }

  /**
   * @see IMenuListener#initMenu
   **/
  private void initMenu(Control pControl) {
    MenuManager menuMgr = new MenuManager();
    menuMgr.setRemoveAllWhenShown(true);
    menuMgr.addMenuListener(this);
    Menu menu = menuMgr.createContextMenu(pControl);
    pControl.setMenu(menu);
  }

  /**
   * @see IMenuListener#menuAboutToShow
   **/
  public void menuAboutToShow(IMenuManager pManager) {
    TreeItem items[] = fTree.getSelection();
    if (null != items && 0 < items.length) {
      for (int i = 0; i < items.length; i++) {
        Object data = items[i].getData();
        if (data instanceof Test) {
          Test test = (Test) data;
          if (fIconArgsNone != items[i].getImage()) {
            pManager.add(new ActionParamDialog(test));
          }
          pManager.add(new ShowHelpAction(fShell, test.getName()));
        }
      }
    }
  }

  /**
   * handler for change of selection state (edit mode only)
   **/
  class SelectionChangedListener implements Listener {
    public void handleEvent(Event event) {
      if (SWT.CHECK == event.detail
        && null != event.item
        && event.item instanceof TreeItem) {
        TreeItem selectedItem = (TreeItem) event.item;
        setCheckStateRecursive(selectedItem, selectedItem.getChecked());
        setCheckStateFromChildren(fTree.getItems()[0]); // root
      }
    }

    private void setCheckStateRecursive(TreeItem item, boolean checked) {
      if (item.getData() instanceof TestTree) {
        TreeItem[] children = item.getItems();
        for (int i = 0; i < children.length; i++) {
          setCheckStateRecursive(children[i], checked);
        }
      } else if (item.getData() instanceof Test) {
        Test test = (Test) item.getData();
        if (checked) {
          test.setActive(test.paramsOkay());
        } else {
          test.setActive(false);
        }
        item.setChecked(test.isActive());
      }
    }

  }

  /**
   * handler for doubleclick & key space
   **/
  class SpaceAndMouseListener implements Listener {
    public void handleEvent(Event e) {
      switch (e.type) {
        case SWT.KeyDown :
          if (e.keyCode == 0) { // space
            handle();
          }
          break;

        case SWT.MouseDoubleClick :
          handle();
          break;
      }
    }
    private void handle() {
      if (null != fTree) {
        TreeItem items[] = fTree.getSelection();
        if (null != items && 0 < items.length) {
          Object data = items[0].getData();
          if (data instanceof Test) {
            new ActionParamDialog((Test) data).run();
          }
        }
      }
    }
  }

  /**
   * Free the icons allocated associated with this dialog.
   **/
  class ActionParamDialog extends Action {
    private Test fTest;

    /**
     * Constructor for JLintActionTestShow.
     */
    public ActionParamDialog(Test pTest) {
      super(pTest.getName());
      if (readOnly) {
        setText(TextKeyCfg.VARIANT_SHOW_TEST_ACTION.getText());
      } else {
        setText(TextKeyCfg.VARIANT_EDIT_TEST_ACTION.getText());
      }
      fTest = pTest;
    }
    public void run() {
      if (null != fTest) {
        new TestDialog(fShell, fTest, readOnly).open();
        updateTree();
      }
    }

  }


}