/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/config/TestSetComposite.java#4 $
 */

package com.sap.tc.jtools.jlint.eclipse.config;

import java.io.IOException;

import org.eclipse.jdt.internal.ui.util.SWTUtil;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;

import com.sap.tc.jtools.jlint.eclipse.JlinPlugin;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jtci.Test;
import com.sap.tc.jtools.jtci.TestSet;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;

/**
 * Dialogarea which contains a list of variants.
 * 
 * @version   $Date: 2004/07/30 $
 * @author    BPL Tools
 */

public class TestSetComposite {

  public static final String RESULT_EXPORT_DIRNAME =
    "jlin.result.export.dirname";
  public static final String EXPORT_RESULT = "jlin.export.result";

  private IPreferenceStore prefStore = JlinPlugin.getDefault().getPreferenceStore();

  final private Shell fShell;

  private Label tabLabel;
  private Table fTab;

  private Composite buttonComposite;
  private Button newButton;
  private Button showButton;
  private Button editButton;
  private Button copyButton;
  private Button removeButton;

  private Button exportResultButton;
  private Text exportResultText;
  private Button browseExportResultButton;

  /**
   * Constructor for VariantComposite.
   */
  public TestSetComposite(Composite pParentComposite, String[] pVariantNames) {
    super();
    fShell = pParentComposite.getShell();
    //The main composite
    Composite composite = new Composite(pParentComposite, SWT.NULL);
    GridLayout layout = new GridLayout();
    layout.numColumns = 2;
    layout.marginHeight = 0;
    layout.marginWidth = 0;
    composite.setLayout(layout);
    GridData data = new GridData();
    data.verticalAlignment = GridData.FILL;
    data.horizontalAlignment = GridData.FILL;
    data.grabExcessVerticalSpace = true;
    data.grabExcessHorizontalSpace = true;
    composite.setLayoutData(data);

    createVariantsTab(composite, pVariantNames);
    createButtons(composite);
    createExportComposite(composite);
  }

  /**
   * Method createExportComposite.
   */
  private void createExportComposite(Composite composite) {
    exportResultButton = new Button(composite, SWT.CHECK | SWT.LEFT);
    exportResultButton.setText(TextKeyCfg.EXPORT_BUTTON.getText());
    //"export result XML files to dir");
    GridData gridData = new GridData();
    gridData.horizontalSpan = 2;
    exportResultButton.setLayoutData(gridData);
    exportResultButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        if (exportResultButton.getSelection()) {
          browseExportResultButton.setEnabled(true);
          exportResultText.setEnabled(true);
        } else {
          browseExportResultButton.setEnabled(false);
          exportResultText.setEnabled(false);
        }
      }
    });

    exportResultText = new Text(composite, SWT.READ_ONLY | SWT.BORDER);
    exportResultText.setLayoutData(
      new GridData(GridData.GRAB_HORIZONTAL | GridData.FILL_HORIZONTAL));

    browseExportResultButton = new Button(composite, SWT.PUSH | SWT.CENTER);
    browseExportResultButton.setText(TextKeyCfg.BROWSE_LABEL.getText());
    browseExportResultButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        DirectoryDialog dd = new DirectoryDialog(JlinPlugin.getActiveWorkbenchShell());
        dd.setText(TextKeyCfg.DIRDIALOG_TEXT.getText());
        String oldDir = exportResultText.getText();
        if (oldDir != null) {
          dd.setFilterPath(oldDir);
        }
        String dirName = dd.open();
        if (dirName != null) {
          exportResultText.setText(dirName);
        }
      }
    });
    setButtonGridData(browseExportResultButton);

    // set stored values
    String exportFileName = prefStore.getString(RESULT_EXPORT_DIRNAME);
    boolean exportResult = prefStore.getBoolean(EXPORT_RESULT);
    exportResultButton.setSelection(exportResult);
    exportResultText.setText(exportFileName);
    if (exportResult) {
      exportResultText.setEnabled(true);
      browseExportResultButton.setEnabled(true);
    } else {
      exportResultText.setEnabled(false);
      browseExportResultButton.setEnabled(false);
    }
  }

  private void createVariantsTab(Composite parent, String[] variantNames) {
    tabLabel = new Label(parent, SWT.LEFT);
    tabLabel.setText(TextKeyCfg.VARIANT_LABEL.getText());
    GridData gd = new GridData();
    gd.horizontalSpan = 2;
    gd.horizontalAlignment = GridData.FILL;
    tabLabel.setLayoutData(gd);
    fTab =
      new Table(parent, SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL);
    GridLayout layout = new GridLayout();
    layout.marginWidth = 0;
    layout.marginHeight = 0;
    fTab.setLayout(layout);
    GridData tgd = new GridData(GridData.FILL_BOTH);
    tgd.heightHint = fTab.getItemHeight();
    tgd.verticalAlignment = GridData.FILL;
    tgd.horizontalAlignment = GridData.FILL;
    tgd.grabExcessHorizontalSpace = true;
    tgd.grabExcessVerticalSpace = true;
    fTab.setLayoutData(tgd);
    fillTab(variantNames);
    {
      fTab.addSelectionListener(new SelectionAdapter() {
        public void widgetSelected(SelectionEvent e) {
          if (TestSetTool
            .DEFAULT_NAME
            .equals(fTab.getItem(fTab.getSelectionIndex()).getText())) {
            editButton.setEnabled(false);
            removeButton.setEnabled(false);
          } else {
            editButton.setEnabled(true);
            removeButton.setEnabled(true);
          }
        }
      });
    };
  }

  private void createButtons(Composite parent) {
    this.buttonComposite = new Composite(parent, SWT.RIGHT);
    GridLayout layout = new GridLayout();
    layout.marginHeight = 0;
    layout.marginWidth = 0;
    this.buttonComposite.setLayout(layout);
    GridData gridData = new GridData();
    gridData.verticalAlignment = GridData.VERTICAL_ALIGN_FILL;
    gridData.horizontalAlignment = GridData.HORIZONTAL_ALIGN_FILL;
    this.buttonComposite.setLayoutData(gridData);

    // "New" button
    newButton = new Button(this.buttonComposite, SWT.CENTER | SWT.PUSH);
    newButton.setText(TextKeyCfg.NEW_BUTTON.getText());
    newButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
          new ActionTestSetNew().run();
      }
    });
    setButtonGridData(newButton);

    // "Show" button
    showButton = new Button(this.buttonComposite, SWT.CENTER | SWT.PUSH);
    showButton.setText(TextKeyCfg.SHOW_BUTTON.getText());
    showButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        int selIndex = fTab.getSelectionIndex();
        if (selIndex != -1) {
          new ActionTestSetShow(fTab.getItem(selIndex)).run();
        }
      }
    });
    setButtonGridData(showButton);

    // "Edit" button
    editButton = new Button(this.buttonComposite, SWT.CENTER | SWT.PUSH);
    editButton.setText(TextKeyCfg.EDIT_BUTTON.getText());
    editButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        int selIndex = fTab.getSelectionIndex();
        if (selIndex != -1) {
          new ActionTestSetEdit(fTab.getItem(selIndex)).run();
        }
      }
    });
    setButtonGridData(editButton);

    // "Copy" button
    copyButton = new Button(this.buttonComposite, SWT.CENTER | SWT.PUSH);
    copyButton.setText(TextKeyCfg.COPY_BUTTON.getText());
    copyButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        int selIndex = fTab.getSelectionIndex();
        if (selIndex != -1) {
          new ActionTestSetCopy(fTab.getItem(selIndex)).run();
        }
      }
    });
    setButtonGridData(copyButton);

    // "Remove" button
    removeButton = new Button(this.buttonComposite, SWT.CENTER | SWT.PUSH);
    removeButton.setText(TextKeyCfg.REMOVE_BUTTON.getText());
    removeButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        int selIndex = fTab.getSelectionIndex();
        if (selIndex != -1) {
          new ActionTestSetDelete(fTab.getItem(selIndex)).run();
        }
      }
    });
    setButtonGridData(removeButton);
  }

  private void setButtonGridData(Button button) {
    GridData data = new GridData();
    data.horizontalAlignment = GridData.FILL;
    button.setLayoutData(data);
    // set default button size
    SWTUtil.setButtonDimensionHint(button);
  } /**
         * Fills the table widget with the name of the variants.
         */
  private void fillTab(String[] pVariants) {
    TableItem variant;
    for (int i = 0; i < pVariants.length; i++) {
      variant = new TableItem(fTab, SWT.SINGLE);
      variant.setText(pVariants[i]);
      variant.setData(pVariants[i]);
    }
  } /**
       * Common error handling for the actions below.
       */
  private void handleBadTreeException(BadTreeException pBT) {
    ExceptionHandler.handle(
      pBT,
      TextKeyCfg.PARSE_ERROR_TITLE.getText(),
      TextKeyCfg.PARSE_ERROR_MESSAGE.getText());
  } /**
         * Common error handling for the actions below.
         */
  private void handleIOException(IOException pIO) {
    ExceptionHandler.handle(
      pIO,
      TextKeyCfg.IO_ERROR_TITLE.getText(),
      TextKeyCfg.IO_ERROR_MESSAGE.getText());
  } /**
         * Opens variant dialog for Display.
         */
  class ActionTestSetShow extends Action {
    private String fTestSet;
    public ActionTestSetShow(TableItem pTableItem) {
      super(TextKeyCfg.BASE_SHOW_VARIANT_ACTION.getText());
      fTestSet = (String) pTableItem.getData();
    }
    public void run() {
      if (null != fTestSet) {
        try {
          TestSet set;
          if (TestSetTool.DEFAULT_NAME.equals(fTestSet)) {
            set = TestSetTool.getDefault();
          } else {
            set = TestSetTool.readTestSet(fTestSet);
          }
          new TestSetDialog(fShell, set, true).open();
        } catch (IOException io) {
          handleIOException(io);
        } catch (BadTreeException bt) {
          handleBadTreeException(bt);
        }
      }
    }
  } /**
       * Opens variant dialog for Edit.
       */
  class ActionTestSetEdit extends Action {
    private String fTestSet;
    public ActionTestSetEdit(TableItem pTableItem) {
      super(TextKeyCfg.BASE_EDIT_VARIANT_ACTION.getText());
      fTestSet = (String) pTableItem.getData();
    }
    public void run() {
      try {
        if (!TestSetTool.DEFAULT_NAME.equals(fTestSet)) {
          TestSet set = TestSetTool.readTestSet(fTestSet);
          if (Dialog.OK == new TestSetDialog(fShell, set, false).open()) {
            TestSetTool.writeTestSet(set);
          }
        } else { // never to be reached
          throw new IllegalStateException();
        }
      } catch (IOException io) {
        handleIOException(io);
      } catch (BadTreeException bt) {
        handleBadTreeException(bt);
      }
    }
  }

     /**
      * Creates a new TestSet.
      */
  class ActionTestSetNew extends Action {
    public void run() {
        NewVariantNameDialog dlg = new NewVariantNameDialog(fShell, "Variant1"); //$NON-NLS-1$
        if (Dialog.OK == dlg.open()) {
          TestSet testSet = null;
          try {
            Test [] tests = TestSetTool.getDefault().getTests();
            for (int i = 0; i < tests.length; i++) {
              tests[i].setActive(false);
            }
            testSet = new TestSet(dlg.getValue(), tests);
            TestSetTool.writeTestSet(testSet);
          } catch (IOException e) {
            ExceptionHandler.handle(e);
          }
          TableItem item = new TableItem(fTab, SWT.NONE);
          item.setText(testSet.getName());
          item.setData(testSet.getName());
        }
      }
  }

  /**
      * Copies a variant.
      */
  class ActionTestSetCopy extends Action {
    private String fTestSet;
    public ActionTestSetCopy(TableItem pTableItem) {
      super(TextKeyCfg.BASE_COPY_VARIANT_ACTION.getText());
      fTestSet = (String) pTableItem.getData();
    }
    public void run() {
      try {
        TestSet set1;
        if (TestSetTool.DEFAULT_NAME.equals(fTestSet)) {
          set1 = TestSetTool.getDefault();
        } else {
          set1 = TestSetTool.readTestSet(fTestSet);
        }
        NewVariantNameDialog dlg = new NewVariantNameDialog(fShell, "Variant1"); //$NON-NLS-1$
        if (Dialog.OK == dlg.open()) {
          TestSet set2 = new TestSet(dlg.getValue(), set1.getTests());
          TestSetTool.writeTestSet(set2);
          TableItem item = new TableItem(fTab, SWT.NONE);
          item.setText(set2.getName());
          item.setData(set2.getName());
        }
      } catch (IOException io) {
        handleIOException(io);
      } catch (BadTreeException bt) {
        handleBadTreeException(bt);
      }
    }
  } /**
       * Deletes a variant.
       */
  class ActionTestSetDelete extends Action {
    private String fTestSet;
    TableItem fItem;
    public ActionTestSetDelete(TableItem pTableItem) {
      super(TextKeyCfg.BASE_DELETE_VARIANT_ACTION.getText());
      fTestSet = (String) pTableItem.getData();
      fItem = pTableItem;
    }
    public void run() {
      try {
        if (!TestSetTool.DEFAULT_NAME.equals(fTestSet)) {
          TestSetTool.purgeTestSet(fTestSet);
          fItem.dispose();
        } else {
          // never to be reached
          throw new IllegalStateException();
        }
      } catch (IOException io) {
        handleIOException(io);
      }
    }
  }

  public boolean exportResult() {
    return exportResultButton.getSelection();
  }

  public String getExportFileName() {
    return exportResultText.getText();
  }

}
