/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/config/TestDialog.java#3 $
 */

package com.sap.tc.jtools.jlint.eclipse.config;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.io.StringReader;
import java.util.Vector;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.Test;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.util.structures.StructureTree;
import com.sap.tc.jtools.util.xml.XMLParseException;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * Displays all parameters of one single test and allows to modify them.
 * @version   $Date: 2004/07/30 $
 * @author    BPL Tools
 */

public class TestDialog extends Dialog implements IMenuListener {

  private static final String fgBlankParam = TextKeyCfg.BLANK_PARAM.getText(); //$NON-NLS-1$
  private static final String TYPE_TAG = "TYPE"; //$NON-NLS-1$

  private Test fTest;
  private Tree fTree;
  private Shell fShell;
  private boolean fReadOnly;

  private Image fIconTest;
  private Image fIconString;
  private Image fIconArray;
  private Image fIconDeep;

  private Composite treeComposite;

  /**
   * Constructor for DialogParam.
   * @param parentShell
   */
  public TestDialog(Shell pParentShell, Test pTest, boolean pReadOnly) {
    super(pParentShell);
    fTest = pTest;
    fReadOnly = pReadOnly;
  }

  /**
   * Initializes the client area during dialog.open(). 
   * @see Dialog#createDialogArea
   */
  protected Control createDialogArea(Composite pBaseComposite) {

    fShell = pBaseComposite.getShell();
    if (fReadOnly) {
      fShell.setText(
        TextKeyCfg.TEST_SHOW_TITLE.getFormattedText(fTest.getName()));
    } else {
      fShell.setText(
        TextKeyCfg.TEST_EDIT_TITLE.getFormattedText(fTest.getName()));
    }

    treeComposite = new Composite(pBaseComposite, SWT.NONE);
    {
      GridLayout layout = new GridLayout();
      layout.marginHeight =
        convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_MARGIN);
      layout.marginWidth =
        convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_MARGIN);
      layout.verticalSpacing =
        convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_SPACING);
      layout.horizontalSpacing =
        convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_SPACING);
      treeComposite.setLayout(layout);
      treeComposite.setLayoutData(new GridData(GridData.FILL_BOTH));
    }

    createIcons(treeComposite);
    createParamTree(treeComposite);
    createMenuBar();

    return treeComposite;
  }

  protected void createMenuBar() {
    // not implemented due bar resizes window !!
    //Menu baseMenu = new Menu(fShell, SWT.BAR);
    //fShell.setMenuBar(baseMenu);

  }

  private Point computeSize() {
    Rectangle r = Display.getCurrent().getClientArea();
    Point size = new Point(r.width, r.height);

    if (1000 > r.height) {
      if (400 > r.height) {
        size.y = 150;
      } else {
        size.y = 150 + r.height / 3;
      }
    } else {
      size.y = 350 + r.height / 5;
    }

    if (1000 > r.width) {
      if (400 > r.width) {
        size.x = 150;
      } else {
        size.x = 150 + r.width / 5;
      }
    } else {
      size.x = 300 + r.width / 15;
    }
    return size;
  }

  /**
   * Returns the initial location to use for the dialog.
   */
  protected Point getInitialLocation(Point initialSize) {
    Point location = super.getInitialLocation(initialSize);
    location.x += 30;
    location.y += 30;
    return location;
  }

  /**
   * Allocates the icons used and ensure automatic disposal.
   */
  private void createIcons(Composite superComposite) {
    Display d = superComposite.getDisplay();
    fIconArray = IconKeyCfg.PARAM_TYPE_ARRAY.create(d);
    fIconDeep = IconKeyCfg.PARAM_TYPE_DEEP.create(d);
    fIconString = IconKeyCfg.PARAM_TYPE_STRING.create(d);
    fIconTest = IconKeyCfg.TEST.create(d);

    DisposeListener dl = new DisposeListener() {
      public void widgetDisposed(DisposeEvent e) {
        dispose();
      }
    };
    superComposite.addDisposeListener(dl);
  }

  /**
   * frees the system memory associated with the icons
   */
  private void dispose() {
    if (null != fIconArray && !fIconArray.isDisposed()) {
      fIconArray.dispose();
      fIconArray = null;
    }
    if (null != fIconDeep && !fIconDeep.isDisposed()) {
      fIconDeep.dispose();
      fIconDeep = null;
    }
    if (null != fIconString && !fIconString.isDisposed()) {
      fIconString.dispose();
      fIconString = null;
    }
    if (null != fIconTest && !fIconTest.isDisposed()) {
      fIconTest.dispose();
      fIconTest = null;
    }
  }

  /**
   * Places a tree into the parent composite and fills it with an visualization 
   * of the parameters.
   * @see Dialog#createDialogArea
   */
  private void createParamTree(Composite pParentComposite) {
    fTree = new Tree(pParentComposite, SWT.MULTI);
    {
      GridData gd = new GridData(GridData.FILL_BOTH);
      gd.heightHint = computeSize().y;
      gd.widthHint = computeSize().x;
      fTree.setLayoutData(gd);
    }

    fillTree();
    initMenu(fTree);

    if (!fReadOnly) {
      Listener ls = new KeyAndMouseListener();
      fTree.addListener(SWT.KeyDown, ls);
      fTree.addListener(SWT.MouseDoubleClick, ls);
    }
  }

  /**
   * Creates the various items of the tree.
   */
  private void fillTree() {
    TreeItem root;

    root = new TreeItem(fTree, SWT.SINGLE);
    root.setText(fTest.getName());
    root.setData(fTest);
    root.setImage(fIconTest);
    try {
      ParameterInterface[] params = fTest.getParameters();
      if (null != params) {
        for (int j = 0; j < params.length; j++) {

          ParameterInterface curParam = params[j];

          TreeItem nameItem = new TreeItem(root, SWT.SINGLE);
          nameItem.setText(params[j].getName());
          nameItem.setData(params[j]);
          fTree.showItem(nameItem);
          String type = curParam.getType();
          if (ParameterTool.isSimpleType(type)) {
            nameItem.setImage(fIconString);
            Serializable value =curParam.getValue();
            String textValue; 
            if (value != null) {
            	textValue = curParam.valueToString();
            } else {
            	textValue = null;
            }
            TreeItem valueItem = new TreeItem(nameItem, SWT.SINGLE);

            valueItem.setText(displayValue(textValue));
            if (null == value) {
              value = ""; //$NON-NLS-1$
            }
            valueItem.setData(value);

            valueItem.setData(TYPE_TAG, type);

          } else if (ParameterTool.isArrayType(type)) {
            nameItem.setImage(fIconArray);

            String[] values = (String[]) curParam.getValue();
            if (null != values) {

              for (int k = 0; k < values.length; k++) {
                TreeItem valueItem = new TreeItem(nameItem, SWT.SINGLE);
                valueItem.setText(displayValue(values[k]));
                valueItem.setData(values[k]);
                valueItem.setData(TYPE_TAG, type);
              }
            }
          } else if (type.equals(ParameterTool.PAR_TYPE_DEEP_STRUCTURE)) {
            nameItem.setImage(fIconDeep);

            StructureTree st = (StructureTree) curParam.getValue();
            String value = XMLTool.toString(st);

            TreeItem valueItem = new TreeItem(nameItem, SWT.SINGLE);
            valueItem.setText(displayValue(value));
            valueItem.setData(value);
            valueItem.setData(TYPE_TAG, type);

          } else {
            throw new IllegalStateException("unknown Parametertype"); //$NON-NLS-1$
          }
        }
      }
    } catch (Exception e) {
      ExceptionHandler.handle(e);
    }
  }

  private String displayValue(Serializable pValue) {
    if (null == pValue) {
      return fgBlankParam;
    } else {
      return pValue.toString();
    }
  }

  /**
   * @see IMenuManager#initMenu()
   */
  private void initMenu(Control pControl) {
    MenuManager menuMgr = new MenuManager();
    menuMgr.setRemoveAllWhenShown(true);
    menuMgr.addMenuListener(this);
    Menu menu = menuMgr.createContextMenu(pControl);
    pControl.setMenu(menu);
  }

  /**
   * Installs the valid actions on the various nodes in the context menu
   * @see IMenuManager#menuAboutToShow()
   */
  public void menuAboutToShow(IMenuManager pManager) {
    if (fReadOnly) {
      TreeItem items[] = fTree.getSelection();

      if (null != items && 0 < items.length) {
        for (int i = 0; i < items.length; i++) {
          Object data = items[i].getData();
          if (data instanceof Test) {
            pManager.add(new ShowHelpAction(fShell, ((Test) data).getName()));
          }
        }
      }
      return;
    }
    TreeItem items[] = fTree.getSelection();
    if (null != items && 0 < items.length) {
      for (int i = 0; i < items.length; i++) {
        Object data = items[i].getData();
        if (data instanceof ParameterInterface) {
          ParameterInterface param = (ParameterInterface) data;
          String pt = param.getType();
          if (ParameterTool.isArrayType(pt)) {
            pManager.add(new AppendAction(items[i]));
          }
        } else if (data instanceof Test) {
          pManager.add(new SetDefaultAction());
          pManager.add(new ShowHelpAction(fShell, ((Test) data).getName()));
        } else if (
          data instanceof String
            || data instanceof Integer
            || data instanceof Float
            || data instanceof Boolean) {
          Object info = items[i].getData(TYPE_TAG);
          if (info instanceof String) {
            String paramType = (String) info;
            if (ParameterTool.isSimpleType(paramType)) {
              pManager.add(new EditAction(items[i], false));
            } else if (ParameterTool.isArrayType(paramType)) {
              pManager.add(new EditAction(items[i], false));
              pManager.add(new RemoveAction(items[i]));
            } else if (
              paramType.equals(ParameterTool.PAR_TYPE_DEEP_STRUCTURE)) {
              pManager.add(new EditAction(items[i], true));
            }
          }
        }
      }
    }
  }
  /**
   * Action to reset a test back to its defaults defined in JLIN.
   */
  class SetDefaultAction extends Action {
    public SetDefaultAction() {
      super(TextKeyCfg.TEST_SET_DEFAULT_ACTION.getText());
    }
    public void run() {

      try {
        String testName = fTest.getName();
        Test defaultTest = TestSetTool.getDefault().getTest(testName);
        if (null != defaultTest) {
          fTest.setParameters(defaultTest.getParameters());
          fTree.removeAll();
          fillTree();
        } else {
          throw new IllegalStateException("unknown test: " + testName); //$NON-NLS-1$
        }
      } catch (Exception ex) {
        ExceptionHandler.handle(ex);
      }
    }
  }
  /**
   * Action to append one entry to a string array.
   */
  class AppendAction extends Action {
    final TreeItem fItem;
    public AppendAction(TreeItem pItem) {
      super(TextKeyCfg.TEST_APPEND_VALUE_ACTION.getText());
      fItem = pItem;
    }

    public void run() {
      TreeItem valueItem = new TreeItem(fItem, SWT.SINGLE);
      valueItem.setText(fgBlankParam);
      valueItem.setData(""); //$NON-NLS-1$
      ParameterInterface param = (ParameterInterface) fItem.getData();
      String type = param.getType();
      valueItem.setData(TYPE_TAG, type);
    }
  }
  /**
   * Action to remove one entry from a string array.
   */
  class RemoveAction extends Action {
    final TreeItem fItem;
    public RemoveAction(TreeItem pItem) {
      super(TextKeyCfg.TEST_REMOVE_VALUE_ACTION.getText());
      fItem = pItem;
    }

    public void run() {
      String value = (String) fItem.getData();
      ParameterInterface param =
        (ParameterInterface) fItem.getParentItem().getData();
      String[] values = (String[]) param.getValue();
      Vector newValues = new Vector();
      for (int i = 0; i < values.length; i++) {
        // remove selected value
        if (!value.equals(values[i])) {
          newValues.add(values[i]);
        }
      }
      String[] newValueNames =
        (String[]) newValues.toArray(new String[newValues.size()]);
      if (newValueNames.length == 0) {
        // convention: not set is null
        param.setValue(null);
      } else {
        param.setValue(newValueNames);
      }
      fItem.dispose();
    }
  }
  /**
   * Action to remove one parameter value
   */
  class EditAction extends Action {
    final TreeItem fItem;
    final boolean fStructureTree;
    EditAction(TreeItem pItem, boolean pStructureTree) {
      super(TextKeyCfg.TEST_EDIT_VALUE_ACTION.getText());
      fItem = pItem;
      fStructureTree = pStructureTree;
    }

    public void run() {
      Object data = fItem.getData();
      Object typeTag = fItem.getData(TYPE_TAG);
      String paramType = null;
      Serializable value = null;
      Serializable newValue = null;
      if (typeTag != null && typeTag instanceof String) {
        paramType = (String) typeTag;
      }
      value = (Serializable) data;
      if (ParameterTool.PAR_TYPE_FILE.equals(paramType)
        || ParameterTool.PAR_TYPE_FILE_ARRAY.equals(paramType)) {
        FileDialog fd = new FileDialog(fShell);
        if (value != null) {
          String osSpecificValue = ((String)value).replace('\\', File.separatorChar).replace('/', File.separatorChar); 
          fd.setFileName(osSpecificValue);
        }
        newValue = fd.open();
      } else if (
        ParameterTool.PAR_TYPE_DIRECTORY.equals(paramType)
          || ParameterTool.PAR_TYPE_DIRECTORY_ARRAY.equals(paramType)) {
        DirectoryDialog dd = new DirectoryDialog(fShell);
        if (value != null) {
          String osSpecificValue = ((String)value).replace('\\', File.separatorChar).replace('/', File.separatorChar); 
          dd.setFilterPath(osSpecificValue);
        }
        newValue = dd.open();
      } else {
        EditParamDialog dg =
          new EditParamDialog(
            fShell,
            value,
            fStructureTree,
            paramType,
            fStructureTree);
        if (Dialog.OK == dg.open()) {
          newValue = dg.getParamValue();
        }
      }
      if (newValue == null) {
        return;
      }
      if (!value.equals(newValue)) {
        fItem.setData(newValue);
        fItem.setText(displayValue(newValue));
        TreeItem parameterItem = fItem.getParentItem();
        ParameterInterface param = (ParameterInterface) parameterItem.getData();
        String type = param.getType();
        if (ParameterTool.isSimpleType(type)) {
          param.setValue(newValue);
        } else if (ParameterTool.isArrayType(type)) {
          TreeItem[] arrayItems = parameterItem.getItems();
          Vector v = new Vector(arrayItems.length);
          for (int i = 0; i < arrayItems.length; i++) {
            String curValue = (String) arrayItems[i].getData();
            if (null != curValue && 0 < curValue.length()) {
              v.add(curValue);
            }
          }
          String[] newValues = new String[v.size()];
          v.copyInto(newValues);
          param.setValue(newValues);
        } else if (ParameterTool.PAR_TYPE_DEEP_STRUCTURE.equals(type)) {
          StringReader sr = new StringReader(newValue.toString());
          try {
            StructureTree valTree = XMLTool.parseReader(sr);
            param.setValue(valTree);
          } catch (IOException io) {
            // cant happen
            throw new IllegalStateException();
          } catch (XMLParseException e) {
            ExceptionHandler.handle(
              e,
              TextKeyCfg.PARSE_ERROR_TITLE.getText(),
              TextKeyCfg.PARSE_ERROR_MESSAGE.getText());
            return;
          }

        } else {
          throw new IllegalStateException("unknown type"); //$NON-NLS-1$
        }
      }
    }
  }
  class KeyAndMouseListener implements Listener {
    public void handleEvent(Event e) {
      switch (e.type) {
        case SWT.KeyDown :
          if (e.keyCode == 0) { // space
            handle();
          }
          break;
        case SWT.MouseDoubleClick :
          handle();
          break;
      }
    }
    private void handle() {
      if (null != fTree) {
        TreeItem items[] = fTree.getSelection();
        if (null != items && 0 < items.length) {
          Object data = items[0].getData();
          if (data instanceof String) {
            String type = (String) data;
            new EditAction(
              items[0],
              type.equals(ParameterTool.PAR_TYPE_DEEP_STRUCTURE))
              .run();
          }
        }
      }
    }
  }
}
