/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/ExtensionManager.java#2 $
 */

package com.sap.tc.jtools.jlint.eclipse;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IPluginRegistry;
import org.eclipse.core.runtime.Platform;
import com.sap.tc.jtools.jlint.eclipse.extension.EclipseModule;
import com.sap.tc.jtools.jlint.eclipse.extension.IMarkerManager;
import com.sap.tc.jtools.jlint.eclipse.extension.IRequestCreator;
import com.sap.tc.jtools.jlint.eclipse.extension.ISelectionHandler;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jlint.extension.Module;
import com.sap.tc.jtools.jlint.extension.TestProvider;
import com.sap.tc.jtools.jlint.extension.TestProviderInterface;
import com.sap.tc.jtools.jlint.extension.TestSet;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.interfaces.TestDescriptionInterface;
import com.sap.tc.jtools.util.xml.XMLParseException;

/**
 * This singleton class reads and initializes all jlin extension point
 * implementations (for jlin extension points see the schema/ subdir of this
 * plugin)
 */
public class ExtensionManager {

  private static ExtensionManager extMan;
  private static Object           lock                       = new char[]{'x'};
  /** string constants used in jlin extension points */
  private static final String     MODULE_EXT_POINT           = "module";           //$NON-NLS-1$
  private static final String     SELECTIONHANDLER_EXT_POINT = "selectionHandler"; //$NON-NLS-1$
  private static final String     TESTS_EXT_POINT            = "tests";            //$NON-NLS-1$
  private static final String     REQUESTCREATOR_ELEMENT     = "requestCreator";   //$NON-NLS-1$
  private static final String     SELECTIONHANDLER_ELEMENT   = "selectionHandler"; //$NON-NLS-1$
  private static final String     TESTOBJECTTYPE_ELEMENT     = "testObjectType";   //$NON-NLS-1$
  private static final String     TESTSET_ELEMENT            = "testSet";          //$NON-NLS-1$
  private static final String     MARKERMANAGER_ELEMENT      = "markerManager";    //$NON-NLS-1$
  private static final String     CLASS_ATTRIBUTE            = "class";            //$NON-NLS-1$
  private static final String     DESCRIPTORFILE_ATTRIBUTE   = "descriptorFile";   //$NON-NLS-1$
  private static final String     MODULE_ID_ATTRIBUTE        = "moduleId";         //$NON-NLS-1$
  private static final String     NAME_ATTRIBUTE             = "name";             //$NON-NLS-1$

  public static ExtensionManager getExtensionManager() {
    synchronized (lock) {
      if (extMan != null) {
        return extMan;
      } else {
        return extMan = new ExtensionManager();
      }
    }
  }
  /** some helper maps */
  private Map                   testName2ModuleMap = new HashMap();
  private Map                   id2ModuleMap         = new HashMap();
  private Map                   testObj2ModuleMap = new HashMap();
  private List                  selHandlerList       = new ArrayList();
  private TestProviderInterface testProvider;

  /** no instantiation from outside */
  private ExtensionManager() {
    try {
      init();
    } catch (Exception e) {
      ExceptionHandler.handle(e);
    }
  }

  public ISelectionHandler[] getSelectionHandlers() {
    return (ISelectionHandler[]) selHandlerList
        .toArray(new ISelectionHandler[0]);
  }

  public TestProviderInterface getTestProvider() {
    return testProvider;
  }

  /**
   * initialize jlin extensions
   */
  private void init() throws IOException, CoreException, BadTreeException,
      XMLParseException {
    IPluginRegistry registry = Platform.getPluginRegistry();
    initModules(registry.getExtensionPoint(JlinPlugin.getPluginId(),
        MODULE_EXT_POINT));
    initSelectionHandlers(registry.getExtensionPoint(JlinPlugin.getPluginId(),
        SELECTIONHANDLER_EXT_POINT));
    initTests(registry.getExtensionPoint(JlinPlugin.getPluginId(),
        TESTS_EXT_POINT));
  }

  private void initModules(IExtensionPoint moduleExtensionPoint)
      throws CoreException, XMLParseException {
    IExtension[] moduleExtensions = moduleExtensionPoint.getExtensions();
    for (int i = 0; i < moduleExtensions.length; i++) {
      IConfigurationElement[] elements = moduleExtensions[i]
          .getConfigurationElements();
      IMarkerManager markerMan = null;
      IRequestCreator creator = null;
      Map name2testObjClasNameMap = new HashMap();
      for (int j = 0; j < elements.length; j++) {
        if (MARKERMANAGER_ELEMENT.equals(elements[j].getName())) {
          markerMan = (IMarkerManager) elements[j]
              .createExecutableExtension(CLASS_ATTRIBUTE);
        } else if (REQUESTCREATOR_ELEMENT.equals(elements[j].getName())) {
          creator = (IRequestCreator) elements[j]
              .createExecutableExtension(CLASS_ATTRIBUTE);
        } else if (TESTOBJECTTYPE_ELEMENT.equals(elements[j].getName())) {
          String name = elements[j].getAttribute(NAME_ATTRIBUTE);
          String className = elements[j].getAttribute(CLASS_ATTRIBUTE);
          name2testObjClasNameMap.put(name, className);
        } else
          throw new XMLParseException("unknown element: "
              + elements[j].getName()); //$NON-NLS-1$
      }
      if (creator == null || markerMan == null)
        throw new XMLParseException(
            "both markerManager and requestCreator elements must be declared in plugin "
                //$NON-NLS-1$
                + moduleExtensions[i].getDeclaringPluginDescriptor()
                    .getUniqueIdentifier());
      String moduleId = moduleExtensions[i].getUniqueIdentifier();
      ClassLoader pluginLoader = moduleExtensions[i]
          .getDeclaringPluginDescriptor().getPluginClassLoader();
      EclipseModule module = new EclipseModule(moduleId, pluginLoader,
          name2testObjClasNameMap, markerMan, creator);
      id2ModuleMap.put(moduleId, module);
      Set testObjTypes = name2testObjClasNameMap.keySet();
      for (Iterator iter = testObjTypes.iterator(); iter.hasNext();) {
        String testObjType = (String) iter.next();
        testObj2ModuleMap.put(testObjType, module);
      }
    }
  }

  private void initSelectionHandlers(IExtensionPoint selHandlerExtensionPoint)
      throws CoreException {
    IExtension[] selHandlers = selHandlerExtensionPoint.getExtensions();
    for (int i = 0; i < selHandlers.length; i++) {
      IConfigurationElement[] elements = selHandlers[i]
          .getConfigurationElements();
      for (int j = 0; j < elements.length; j++) {
        if (SELECTIONHANDLER_ELEMENT.equals(elements[j].getName())) {
          ISelectionHandler handler = (ISelectionHandler) elements[j]
              .createExecutableExtension(CLASS_ATTRIBUTE);
          selHandlerList.add(handler);
        }
      }
    }
  }

  private void initTests(IExtensionPoint testsExtensionPoint)
      throws BadTreeException, FileNotFoundException, IOException,
      XMLParseException {
    IExtension[] testSets = testsExtensionPoint.getExtensions();
    for (int i = 0; i < testSets.length; i++) {
      IConfigurationElement[] elements = testSets[i].getConfigurationElements();
      for (int j = 0; j < elements.length; j++) {
        if (!TESTSET_ELEMENT.equals(elements[j].getName()))
          throw new XMLParseException("unknown element: "
              + elements[j].getName()); //$NON-NLS-1$
        String descriptorFilePath = Platform.resolve(
            testSets[i].getDeclaringPluginDescriptor().getInstallURL())
            .getPath()
            + elements[j].getAttribute(DESCRIPTORFILE_ATTRIBUTE);
        String moduleId = elements[j].getAttribute(MODULE_ID_ATTRIBUTE);
        Module module = (Module) id2ModuleMap.get(moduleId);
        if (module == null) {
          throw new XMLParseException("unknown moduleID: " //$NON-NLS-1$
              + moduleId
              + " in plugin " //$NON-NLS-1$
              + testSets[i].getDeclaringPluginDescriptor()
                  .getUniqueIdentifier());
        }
        File descriptorFile = new File(descriptorFilePath);
        ClassLoader pluginLoader = testSets[i].getDeclaringPluginDescriptor()
            .getPluginClassLoader();
        TestSet testSet = new TestSet(descriptorFile, pluginLoader);
        module.addTestSet(testSet);
        TestDescriptionInterface[] tests = testSet.getTestDescriptionSet()
            .getTestTree().getAllLeaves();
        for (int k = 0; k < tests.length; k++) {
          testName2ModuleMap.put(tests[k].getName(), id2ModuleMap.get(moduleId));
        }
      }
    }
    this.testProvider = TestProvider.initialize((EclipseModule[]) id2ModuleMap
        .values().toArray(new EclipseModule[0]));
  }

  public EclipseModule getModuleForTest(String testName) {
    return (EclipseModule) testName2ModuleMap.get(testName);
  }
  
  public EclipseModule getModuleForTestObjectType(String testObjectType) {
    return (EclipseModule) testObj2ModuleMap.get(testObjectType);
  }
  
  
}
