package com.sapmarkets.technology.util;

import java.io.*;
import java.util.Random;
import java.security.SecureRandom;
import java.util.ConcurrentModificationException;

/**
 * Implements time-based UUIDs (version 1 UUIDs) according to the IETF draft on
 * UUIDs and GUIDs from February 4, 1998 (draft-leach-uuids-guids-01.txt).
 *
 * @author       Peter Eberlein, SAPMarkets
 * @version      1.3
 */

public class UUID implements Cloneable, Serializable {
  private final static long timeDiff = 0x01B21DD213814000L;
  private final static char[] hexCharacters = {
    '0', '1', '2', '3', '4', '5', '6', '7',
    '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };
  private final static boolean nativeLibraryAvailable;

  protected final static int randomClockSequence;
  protected final static byte[] randomNodeId;

  protected static long previousTime = 0;

  protected String uuid = null;

  static {
    boolean loadSuccessful = false;
    try {
      System.loadLibrary("UUID");
      loadSuccessful = true;
    } catch (UnsatisfiedLinkError e) {
    } catch (SecurityException e) {
    }
    nativeLibraryAvailable = loadSuccessful;

    Random random = new SecureRandom();
	// significantly slower than Random but better and anyway a one time effort
    randomClockSequence = random.nextInt(0x4000);

    // The ideal solution is to obtain a 47 bit cryptographic quality
    // random number, and use it as the low 47 bits of the node ID, with
    // the most significant bit of the first octet of the node ID set
    // to 1. This bit is the unicast/multicast bit, which will never be
    // set in IEEE 802 addresses obtained from network cards; hence, there
    // can never be a conflict between UUIDs generated by machines with
    // and without network cards.
    randomNodeId = new byte[6];
    random.nextBytes(randomNodeId);
    randomNodeId[0] |= 0x80;
  }

/**
 * Creates a new UUID object. It is initalized with the nil UUID
 * 00000000-0000-0000-0000-000000000000.
 */
  public UUID() {
    uuid = "00000000-0000-0000-0000-000000000000";
  }

/**
 * Creates a UUID object from a string in UUID format.
 * This constructor has no effect on the creation of further UUIDs.
 *
 * @throws NumberFormatException if the string is not in UUID format.
 */
  public UUID(String uuid) throws NumberFormatException {
    if (uuid.length() != 36 ||
        uuid.charAt(8) != '-' ||
        uuid.charAt(13) != '-' ||
        uuid.charAt(18) != '-' ||
        uuid.charAt(23) != '-') {
      throw new NumberFormatException("'" + uuid +
        "' is not in UUID format xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx");
    }
    Long.parseLong(uuid.substring(0, 8), 16);
    Integer.parseInt(uuid.substring(9, 13), 16);
    Integer.parseInt(uuid.substring(14, 18), 16);
    Integer.parseInt(uuid.substring(19, 23), 16);
    Long.parseLong(uuid.substring(24, 36), 16);

    this.uuid = uuid;
  }

  /**
   * Compares this UUID to the specified object. The result is true if and only
   * if the argument is not null and is a UUID object that represents the same
   * UUID as this object.
   *
   * @return <code>true</code> if the UUIDs are equal; <code>false</code>
   * otherwise.
   */
  public boolean equals(Object obj) {
    return (obj != null &&
            obj.getClass() == UUID.class &&
            uuid.equals(((UUID)obj).uuid));
  }

  /**
   * Returns a hash code value for the object.
   *
   * @return a hash code value for this object.
   */
  public int hashCode() {
    try {
      return ((int)(Long.parseLong(uuid.substring(0, 8), 16) ^
                    Long.parseLong(uuid.substring(28, 36), 16))) ^
             ((Integer.parseInt(uuid.substring(9, 13), 16) ^
               Integer.parseInt(uuid.substring(19, 23), 16)) << 16) ^
             ((Integer.parseInt(uuid.substring(14, 18), 16) ^
               Integer.parseInt(uuid.substring(24, 28), 16)) & 0xffff);
    } catch (NumberFormatException e) {
      // should never happen
      return uuid.hashCode();
    }
  }


  /**
   * Static factory method that creates a newly generated time-based UUID
   * (version 1 UUID).
   */
  public static UUID createUUID() {
    UUID uuid = new UUID();
    uuid.generateUUID();
    return uuid;
  }

  /**
   * Generates a time-based UUID (version 1 UUID).
   */
  public void generateUUID() {
    synchronized (UUID.class) {
      String nativeUUID = null;
      if (nativeLibraryAvailable) {
        try {
          nativeUUID = getNativeUUID();
        } catch (ConcurrentModificationException e) {
          // nativeUUID is still null so we will create a pure Java UUID
        }
      }

      if (nativeUUID != null) {
        uuid = nativeUUID;
      } else {
        long currentTime = getCurrentTime();

        int timeLow = (int)currentTime;
        short timeMid = (short)(currentTime >> 32);
        short timeHiAndVersion =
          (short)(((currentTime >> 48) & 0xfff) | 0x1000);

        byte clockSeqHiAndReserved =
          (byte)(((randomClockSequence >> 8) & 0x3f) | 0x80);
        byte clockSeqLow = (byte)(randomClockSequence & 0xff);

        StringBuffer buf = new StringBuffer(36);
        appendHexValue(buf, timeLow, 8).append('-');
        appendHexValue(buf, timeMid, 4).append('-');
        appendHexValue(buf, timeHiAndVersion, 4).append('-');
        appendHexValue(buf, clockSeqHiAndReserved, 2);
        appendHexValue(buf, clockSeqLow, 2).append('-');
        for (int i = 0; i < 6; i++) {
          appendHexValue(buf, randomNodeId[i], 2);
        }
        uuid = buf.toString();
      }
    }
  }

  /**
   * Indicates if a UUID is strong. Strong UUIDs are more likely to be unique
   * than weak UUIDs. This is a relative qualification, strong UUIDs are still
   * not guaranteed to be indeed universally unique, however, they have a
   * higher probability than weak UUIDs.
   *
   * @return <code>true</code> if the UUID is considered to be strong;
   * <code>false</code> otherwise.
   */
  public boolean isStrong() {
    try {
      // check if it is time based, the IETF variant and a hardware node id
      return (Integer.parseInt(uuid.substring(14, 16), 16) & 0xf0) == 0x10 &&
             (Integer.parseInt(uuid.substring(19, 21), 16) & 0xc0) == 0x80 &&
             (Integer.parseInt(uuid.substring(24, 26), 16) & 0x80) == 0x00;
    } catch (NumberFormatException e) {
      // should never happen
      return false;
    }
  }

  /**
   * Gets the current time in 100ns. Makes sure that no two invokations
   * return the same value.
   *
   * @return the current time
   */
  protected static synchronized long getCurrentTime() {
    for (;;) {
      long currentTime = System.currentTimeMillis() * 10000 + timeDiff;
      if (previousTime >= currentTime) {
        if (previousTime - currentTime >= 9999) { // stall on overflow
          try {
            Thread.sleep(1);
          } catch (InterruptedException e) {
            // ignore, we will check the time again in the next loop anyway
          }
          continue;
        }
        currentTime = previousTime + 1;
      }
      previousTime = currentTime;
      return currentTime;
    }
  }

  /**
   * Appends a hex value to a <code>StringBuffer</code>.
   *
   * @param buf the string buffer.
   * @param v the value.
   * @param len the number of characters to be added.
   * @return the string buffer that was passed in as buf.
   */
  private final static StringBuffer appendHexValue(
      StringBuffer buf, int v, int len) {
    for (int i = (len - 1) * 4; i >= 0; i-= 4) {
      buf.append(hexCharacters[(v >> i) & 0xf]);
    }
    return buf;
  }

  /**
   * Returns the UUID as a string in the format
   * xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx.
   *
   * @return the string representation of the UUID.
   */
  public String toString() {
    return uuid;
  }

  /**
   * Returns a UUID created utilizing all means available from JNI.
   * This might include system wide locking to avoid parallel creation of
   * identical ids, access to the MAC address of an installed ethernet card,
   * and persisting of each generated UUID.
   *
   * @return the UUID or <code>null</code> if it could not be created.
   * throws ConcurrentModificationException if the system wide lock could not
   * be obtained.
   */
  private static native String getNativeUUID()
    throws ConcurrentModificationException;

  /**
   * Test method.
   */
  public static void main(String[] args) {
    UUID uuid1 = new UUID();
    UUID uuid2 = new UUID();
    System.out.println("Nil UUID          : " + uuid1.toString() +
      (uuid1.isStrong() ? " (strong)" : " (weak)"));
    uuid1.generateUUID();
    uuid2.generateUUID();
    System.out.println("Generated UUID 1  : " + uuid1.toString() +
      (uuid1.isStrong() ? " (strong)" : " (weak)"));
    System.out.println("Generated UUID 2  : " + uuid2.toString() +
      (uuid2.isStrong() ? " (strong)" : " (weak)"));
    UUID uuid2c = new UUID(uuid2.toString());
    System.out.println("Copied UUID 2     : " + uuid2c.toString() +
      (uuid2c.isStrong() ? " (strong)" : " (weak)"));
    uuid2.generateUUID();
    System.out.println("Regenerated UUID 2: " + uuid2.toString() +
      (uuid2.isStrong() ? " (strong)" : " (weak)"));
  }
}
