/*
 * SAPMarkets Copyright (c) 2001
 * All rights reserved
 *
 * @version $Id$
 */

package com.sapmarkets.technology.util;

import java.util.*;
import java.net.*;

/**
 * The <code>StringEncoder</code> class provides String encoding methods.
 * methods. It cannot be instantiated.
 * <p>
 * You would typically use it in the following manner (e.g. in a JSP):
 * <%= StringEncoder.HTMLEncode(s);>
 *
 * @version      0.01 04/17/2002
 * @author       Andreas Mller
 */
public class StringEncoder
{
   protected StringEncoder ()
   {
      // No instantiation of this class permitted
   }

   protected static char[] charsToEscapeHTML= { '&', '\"', '<', '>' };
   protected static String[] escapeWithHTML = { "&amp;", "&quot;", "&lt;", "&gt;" };
   protected static char[] charsToEscapeSQL= { '\'' };
   protected static String[] escapeWithSQL = { "\'\'"};


   /**
   * Does the actual encoding.
   *
   * @param      str        String to be encoded
   * @param      keys       the char keys to be encoded
   * @param      values     the string values for encoding the char keys.
   *
   * @return	the encoded String
   */
   protected static String encode(String str, char[] keys, String[] values)
   {
      String encodedString = null;
      if(str!=null)
      {
        StringBuffer sb = new StringBuffer(str);
        for(int curChar = 0; curChar < keys.length; curChar++) {
          int curCharPos = 0;
          int length = sb.length();
          while(curCharPos < length) {
            if(sb.charAt(curCharPos)==keys[curChar]) {
              String curValue = values[curChar];
              sb = sb.replace(curCharPos, curCharPos+1, curValue);
              curCharPos += (curValue.length());
              length += (curValue.length()-1);
              //System.out.println("sb: "+sb);
              //System.out.println("ccp: "+curCharPos+" / length: "+length);

            }
            else
              curCharPos++;
          }
        }
        encodedString = sb.toString();
      }
      return encodedString;
   }

   /**
   * Does the actual decoding.
   *
   * @param      str        String to be decoded
   * @param      keys       the char keys to be decoded to
   * @param      values     the string values to be decodod to the char keys.
   *
   * @return	the decoded String
   */
   protected static String decode(String str, char[] keys, String[] values)
   {
      String decodedString = null;
      if(str!=null)
      {
        decodedString = str;
        for(int curValue = 0; curValue < values.length; curValue++) {
          decodedString = StringUtil.change(decodedString, values[curValue], ""+keys[curValue]);
        }
      }
      return decodedString;
   }


   /**
   * HTML encodes a String.
   *
   * @param      str        String to be HTML encoded
   *
   * @return	the HTML encoded String
   */
   public static String HTMLEncode(String str)
   {
      return encode(str, charsToEscapeHTML, escapeWithHTML);
   }

   /**
   * HTML decodes a String.
   *
   * @param      str        String to be HTML decoded
   *
   * @return	the HTML decoded String
   */
   public static String HTMLDecode(String str)
   {
      return decode(str, charsToEscapeHTML, escapeWithHTML);
   }

   /**
   * SQL encodes a String.
   *
   * @param      str        String to be SQL encoded
   *
   * @return	the SQL encoded String
   */
   public static String SQLEncode(String str)
   {
      return encode(str, charsToEscapeSQL, escapeWithSQL);
   }

   /**
   * URL encodes a String (with unicode support).
   *
   * @param      str        String to be URL encoded
   *
   * @return	the URL encoded String
   */
   public static String URLEncode(String str)
   {
      String encodedString = null;
      if(str!=null) {
        // take care about unicode here ...
        encodedString = "";
        char[] chars = str.toCharArray();
        for(int i=0; i<chars.length; i++) {
          String curHex = Integer.toHexString(chars[i]);
          if(curHex.length()<4) {
            for(int j=curHex.length(); j<4; j++) {
              curHex = "0"+curHex;
            }
          }
          encodedString += curHex;
        }
        encodedString = URLEncoder.encode(encodedString);
      }
      return encodedString;
   }

   /**
   * URL decodes a String (with unicode support).
   *
   * @param      str        String to be URL decoded
   *
   * @return	the URL decoded String
   */
   public static String URLDecode(String str)
   {
      String decodedString = null;
      if(str!=null) {
        char[] chars = new char[str.length()/4];
        int i = 0;
        while(i+3 < str.length()) {
          chars[i/4] = (char) Integer.parseInt(str.substring(i,i+4), 16);
          i += 4;
        }
        decodedString = new String(chars);
      }
      return decodedString;
   }

  /**
   * Just for testing.
   */
  public static void main(String args[]) {

    String bla = "<asd>";//"\u1234\u9876";
    String bla1 = StringEncoder.URLEncode(bla);
    System.out.println("encoded: " + bla1);
    System.out.println("decoded: " + StringEncoder.URLDecode(bla1));

/*
    System.out.println(StringEncoder.HTMLEncode(null));
    System.out.println(StringEncoder.HTMLEncode("<b>"));
    System.out.println(StringEncoder.HTMLEncode("<b>\"a"));
    System.out.println(StringEncoder.HTMLEncode("<"));
    System.out.println(StringEncoder.SQLEncode("\'bla\'"));

    System.out.println(StringEncoder.SQLEncode("a'"));
    System.out.println(StringEncoder.SQLEncode("a'a"));
    System.out.println(StringEncoder.SQLEncode("''"));
    System.out.println(StringEncoder.SQLEncode("'a"));
    System.out.println(StringEncoder.SQLEncode("'a'"));
    System.out.println(StringEncoder.HTMLDecode("&quot;"));
*/
  }

}

