/*
 * SAPMarkets Copyright (c) 2001
 * All rights reserved
 *
 * @version $Id$
 */

package com.sapmarkets.technology.util;

import java.io.*;
import java.net.*;
import java.text.*;
import java.util.*;

/**
 * This class implements a proxy server. It may either be used by instance (multiple instances
 * possible) or as stand-alone application.
 *
 * @created   12. April 2001
 * @author    c5021826
 */
public class Proxy
{
    // Number of proxy instances
    private static long runningInstanceCounter = 0;

    // Proxy instance number
    private long thisInstanceCounter = 0;

    // Port at which this proxy should listen at
    private int internalListenerPort = 8080;

    // External redirection host and port receiving requests if no host/port is provided with the request
    private String externalRedirectHost = "localhost";
    private int externalRedirectPort = 80;

    // External proxy host and port which should be used if not null and list of destinations to be excluded from proxying
    private String externalProxyHost = null;
    private int externalProxyPort = 8080;
    private String externalProxyExcludes = null;
    private Collection externalProxyExcludesCollection = null;

    // Destination directory location for proxy protocols
    private File protocolDirectoryLocation = new File( System.getProperty( "user.dir" ) );

    // Verbose switch
    private boolean verbose = false;

    // Number of requests received and processed; bytes received and sent
    private long requestsReceived = 0;
    private long requestsProcessed = 0;
    private long bytesReceived = 0;
    private long bytesSent = 0;

    // Flags indicating the current status
    private int runningMainRequestThreads = 0;
    private boolean started = false;
    private boolean suspended = false;
    private boolean shuttingdown = false;

    // Some constants needed throughout execution of this code
    private final static String SCHEME_HTTP_ID = "http";
    private final static String SCHEME_PROTOCOL_SEPARATOR = "://";
    private final static DecimalFormat decimalFormat = new DecimalFormat( "000" );

    /**
     * Construct proxy with default settings.
     */
    public Proxy()
    {
        thisInstanceCounter = getNextInstanceCounter();
    }

    /**
     * Construct proxy with the given settings.
     *
     * @param internalListenerPort       Port at which this proxy should listen at
     * @param externalRedirectHost       External redirection host receiving requests if no
     *      host/port is provided with the request
     * @param externalRedirectPort       External redirection port receiving requests if no
     *      host/port is provided with the request
     * @param externalProxyHost          External proxy host which should be used if not null and
     *      list of destinations to be excluded from proxying
     * @param externalProxyPort          External proxy port which should be used if not null and
     *      list of destinations to be excluded from proxying
     * @param externalProxyExcludes      List of destinations to be excluded from proxying
     * @param protocolDirectoryLocation  Destination directory location for proxy protocols
     * @param verbose                    Verbose flag for System.out/err output during runtime
     */
    public Proxy( int internalListenerPort,
                  String externalRedirectHost, int externalRedirectPort,
                  String externalProxyHost, int externalProxyPort, String externalProxyExcludes,
                  File protocolDirectoryLocation, boolean verbose )
    {
        this();
        setInternalListenerPort( internalListenerPort );
        setExternalRedirectHost( externalRedirectHost );
        setExternalRedirectPort( externalRedirectPort );
        setExternalProxyHost( externalProxyHost );
        setExternalProxyPort( externalProxyPort );
        setExternalProxyExcludes( externalProxyExcludes );
        setProtocolDirectoryLocation( protocolDirectoryLocation );
        setVerbose( verbose );
    }

    /**
     * The main method parses arguments and passes them to runServer
     *
     * @param args  The command line arguments
     */
    public static void main( String[] args )
    {
        // Instantiate proxy
        Proxy proxy = new Proxy();

        // Evaluate command line parameters
        try
        {
            proxy.setExternalProxyExcludes( proxy.externalProxyExcludes );
            for( int i = 0; i < args.length; i++ )
            {
                StringTokenizer argumentTokenizer = new StringTokenizer( args[i], "=" );
                String key = argumentTokenizer.nextToken();
                String val = argumentTokenizer.nextToken();
                if( key.equalsIgnoreCase( "ILP" ) )
                {
                    proxy.setInternalListenerPort( Integer.parseInt( val ) );
                }
                if( key.equalsIgnoreCase( "ERH" ) )
                {
                    proxy.setExternalRedirectHost( val );
                }
                if( key.equalsIgnoreCase( "ERP" ) )
                {
                    proxy.setExternalRedirectPort( Integer.parseInt( val ) );
                }
                if( key.equalsIgnoreCase( "EPH" ) )
                {
                    proxy.setExternalProxyHost( val );
                }
                if( key.equalsIgnoreCase( "EPP" ) )
                {
                    proxy.setExternalProxyPort( Integer.parseInt( val ) );
                }
                if( key.equalsIgnoreCase( "EPE" ) )
                {
                    proxy.setExternalProxyExcludes( val );
                }
                if( key.equalsIgnoreCase( "PDL" ) )
                {
                    proxy.setProtocolDirectoryLocation( new File( val ) );
                    proxy.getProtocolDirectoryLocation().mkdirs();
                }
            }
        }
        catch( Exception exception )
        {
            System.err.println( "Catched exception " + exception );
            exception.printStackTrace( System.err );
            System.err.println();
            System.err.println( "Usage: java <package>.Proxy [ILP=<internal listener port>]" );
            System.err.println( "                            [ERH=<external redirect host>]" );
            System.err.println( "                            [ERP=<external redirect port>]" );
            System.err.println( "                            [EPH=<external proxy host>]" );
            System.err.println( "                            [EPP=<external proxy port>]" );
            System.err.println( "                            [EPE=<external proxy excludes separated by comma>]" );
            System.err.println( "                            [PDL=<directory location used for protocol files>]" );
            System.err.println();
            System.err.println( "This proxy will protocol incoming requests and outgoing responses to the" );
            System.err.println( "directory specified by the PDL argument. Each request and response will be" );
            System.err.println( "written to a file named <proxy instance number>.<request number>.REQ|RES." );
            System.err.println( "<client|server host name>=<client|server host adrs>@<client|server port>.txt." );
            System.err.println();
            System.err.println( "The ILP parameter specifies the port at which this proxy should listen at for" );
            System.err.println( "incoming requests. This port is your proxy port. The proxy host is the machine" );
            System.err.println( "where this proxy is started." );
            System.err.println();
            System.err.println( "All requests received without destination host and port will be redirected to" );
            System.err.println( "the host and port specfied by the arguments ERH and ERP." );
            System.err.println();
            System.err.println( "If the requests should become processed via another (real) external proxy," );
            System.err.println( "the EPH and EPP arguments will be used. With the EPE argument you may provide" );
            System.err.println( "a comma-separated list of hosts which should be contacted directly not using" );
            System.err.println( "an external proxy (even if set)." );
            System.exit( -1 );
        }

        // Output proxy arguments
        System.out.println( "Proxy configured to protocol requests and responses using the format" );
        System.out.println( "<proxy instance number>.<request number>.REQ|RES.<client|server host name>=" );
        System.out.println( "<client|server host adrs>@<client|server port>.txt into directory:" );
        System.out.println( "  " + proxy.getProtocolDirectoryLocation().getAbsolutePath() );
        System.out.println();
        System.out.println( "To access this proxy please configure this machine as your proxy host" );
        System.out.println( "and port " + proxy.getInternalListenerPort() + " as your proxy port. A simple connect to it on" );
        System.out.println( "port " + ( proxy.getInternalListenerPort() + 1 ) + " will terminate the proxy and output a summary." );
        System.out.println();
        System.out.println( "All requests received without destination host and port will be redirected to:" );
        System.out.println( "  " + proxy.getExternalRedirectHost() + ":" + proxy.getExternalRedirectPort() );
        System.out.println();
        if( proxy.getExternalProxyHost() != null )
        {
            System.out.println( "The following (real) external proxy will be used:" );
            System.out.print( "  " + proxy.getExternalProxyHost() + ":" + proxy.getExternalProxyPort() );
            if( proxy.getExternalProxyExcludes() != null )
            {
                System.out.println( " - with the following excludes (host and port starting with)" );
                for( Iterator iter = proxy.externalProxyExcludesCollection.iterator(); iter.hasNext();  )
                {
                    System.out.print( "  " + ( String )iter.next() + "*" );
                }
            }
            System.out.println();
        }
        else
        {
            System.out.println( "No (real) external proxy will be used." );
        }
        System.out.println();

        // Start proxy
        proxy.setVerbose( true );
        proxy.start();

        // Wait for shutdown request
        try
        {
            // Open the shutdown socket to listen at and wait for incoming connection
            new ServerSocket( proxy.getInternalListenerPort() + 1 ).accept();
        }
        catch( Exception exception )
        {
            proxy.printErrLn( "Failed to wait for shutdown request: ", exception );
        }
        finally
        {
            proxy.shutdown();
            System.out.println( "Proxy received " + proxy.requestsReceived + " request(s) of which " + proxy.requestsProcessed + " have been processed." );
            System.out.println( "Proxy received " + proxy.bytesReceived + " byte(s) and sent " + proxy.bytesSent + " byte(s)." );
        }
    }

    /**
     * Start proxy.
     */
    public synchronized void start()
    {
        try
        {
            // Open the working socket to listen at
            printOut( "Starting proxy..." );
            final ServerSocket listenerSocket = new ServerSocket( getInternalListenerPort() );

            // Start listener thread
            new Thread(
                new Runnable()
                {
                    /**
                     * Listen for incoming connections
                     */
                    public void run()
                    {
                        // Set started and shutting down flags
                        started = true;
                        shuttingdown = false;
                        printOutLn( "Proxy started." );

                        // Run until shutting down flag set to true
                        while( !isShuttingdown() )
                        {
                            try
                            {
                                // Wait for a connection
                                final String requestNumber = decimalFormat.format( getRequestsReceived() );
                                final Socket clientSocket = listenerSocket.accept();

                                // Check if not shutting down
                                if( !isShuttingdown() )
                                {
                                    // Increment requests received counter
                                    incRequestsReceived();
                                }

                                // Check if not shutting down and not suspended
                                if( ( !isShuttingdown() ) && ( !isSuspended() ) )
                                {
                                    // Start handler thread
                                    new Thread(
                                        new Runnable()
                                        {
                                            /**
                                             * Handle request
                                             */
                                            public void run()
                                            {
                                                // Handle request
                                                handleRequest( requestNumber, clientSocket );
                                            }
                                        } ).start();
                                }
                                else
                                {
                                    // Closing connection
                                    clientSocket.close();
                                }
                            }
                            catch( Exception exception )
                            {
                                printErrLn( "Failed to handle incoming request: ", exception );
                            }
                        }
                    }
                } ).start();
        }
        catch( Exception exception )
        {
            printErrLn( "Failed to start proxy: ", exception );
        }
    }

    /**
     * Shutting down proxy.
     */
    public void shutdown()
    {
        if( isStarted() )
        {
            // Shut down proxy
            printOut( "Stopping proxy (" + getRunningMainRequestThreads() + " request(s) open)..." );
            shuttingdown = true;
            try
            {
                new Socket( InetAddress.getLocalHost(), getInternalListenerPort() ).close();
                while( getRunningMainRequestThreads() > 0 )
                {
                    printOut( "." );
                    Thread.sleep( 100 );
                }
                printOutLn( "Proxy stopped." );
            }
            catch( Exception exception )
            {
                printErrLn( "Failed to stop proxy: ", exception );
            }
        }
    }

    /**
     * Suspend proxy; starting it if not yet started.
     */
    public synchronized void suspend()
    {
        printOut( "Suspending proxy..." );
        suspended = true;
        if( !isStarted() )
        {
            // Start proxy
            start();
        }
        printOutLn( "Proxy suspended." );
    }

    /**
     * Resume proxy; starting it if not yet started.
     */
    public synchronized void resume()
    {
        printOut( "Resuming proxy..." );
        suspended = false;
        if( !isStarted() )
        {
            // Start proxy
            start();
        }
        printOutLn( "Proxy resumed." );
    }

    /**
     * Get the next valid proxy class instance counter and raise the counter by one.
     *
     * @return   next valid instance counter
     */
    private static synchronized long getNextInstanceCounter()
    {
        return runningInstanceCounter++;
    }

    /**
     * Get the instance number.
     *
     * @return   instance number
     */
    private long getInstanceCounter()
    {
        return thisInstanceCounter;
    }

    /**
     * Get the port at which this proxy should listen at.
     *
     * @return   Port at which this proxy should listen at
     */
    public int getInternalListenerPort()
    {
        return internalListenerPort;
    }

    /**
     * Set the port at which this proxy should listen at.
     *
     * @param internalListenerPort  Port at which this proxy should listen at
     */
    public void setInternalListenerPort( int internalListenerPort )
    {
        this.internalListenerPort = internalListenerPort;
    }

    /**
     * Get the external redirection host receiving requests if no host/port is provided with the
     * request.
     *
     * @return   External redirection host receiving requests if no host/port is provided with the
     *      request
     */
    public String getExternalRedirectHost()
    {
        return externalRedirectHost;
    }

    /**
     * Set the external redirection host receiving requests if no host/port is provided with the
     * request.
     *
     * @param externalRedirectHost  External redirection host receiving requests if no host/port is
     *      provided with the request
     */
    public void setExternalRedirectHost( String externalRedirectHost )
    {
        this.externalRedirectHost = externalRedirectHost;
    }

    /**
     * Get the external redirection port receiving requests if no host/port is provided with the
     * request.
     *
     * @return   External redirection port receiving requests if no host/port is provided with the
     *      request
     */
    public int getExternalRedirectPort()
    {
        return externalRedirectPort;
    }

    /**
     * Set the external redirection port receiving requests if no host/port is provided with the
     * request.
     *
     * @param externalRedirectPort  External redirection port receiving requests if no host/port is
     *      provided with the request
     */
    public void setExternalRedirectPort( int externalRedirectPort )
    {
        this.externalRedirectPort = externalRedirectPort;
    }

    /**
     * Get the external proxy host which should be used if not null and list of destinations to be
     * excluded from proxying.
     *
     * @return   External proxy host which should be used if not null and list of destinations to be
     *      excluded from proxying
     */
    public String getExternalProxyHost()
    {
        return externalProxyHost;
    }

    /**
     * Set the external proxy host which should be used if not null and list of destinations to be
     * excluded from proxying.
     *
     * @param externalProxyHost  External proxy host which should be used if not null and list of
     *      destinations to be excluded from proxying
     */
    public void setExternalProxyHost( String externalProxyHost )
    {
        this.externalProxyHost = externalProxyHost;
    }

    /**
     * Get the external proxy port which should be used if not null and list of destinations to be
     * excluded from proxying.
     *
     * @return   External proxy port which should be used if not null and list of destinations to be
     *      excluded from proxying
     */
    public int getExternalProxyPort()
    {
        return externalProxyPort;
    }

    /**
     * Set the external proxy port which should be used if not null and list of destinations to be
     * excluded from proxying.
     *
     * @param externalProxyPort  External proxy port which should be used if not null and list of
     *      destinations to be excluded from proxying
     */
    public void setExternalProxyPort( int externalProxyPort )
    {
        this.externalProxyPort = externalProxyPort;
    }

    /**
     * Get the list of destinations to be excluded from proxying.
     *
     * @return   List of destinations to be excluded from proxying
     */
    public String getExternalProxyExcludes()
    {
        return externalProxyExcludes;
    }

    /**
     * Get the collection of destinations to be excluded from proxying.
     *
     * @return   Collection of destinations to be excluded from proxying
     */
    public Collection getExternalProxyExcludesCollection()
    {
        return externalProxyExcludesCollection;
    }

    /**
     * Set the list of destinations to be excluded from proxying.
     *
     * @param externalProxyExcludes  List of destinations to be excluded from proxying
     */
    public synchronized void setExternalProxyExcludes( String externalProxyExcludes )
    {
        this.externalProxyExcludes = externalProxyExcludes;
        if( externalProxyExcludes != null )
        {
            externalProxyExcludesCollection = new Vector();
            StringTokenizer excludeTokenizer = new StringTokenizer( externalProxyExcludes, ",;" );
            while( excludeTokenizer.hasMoreTokens() )
            {
                String exclude = excludeTokenizer.nextToken().trim();
                while( exclude.endsWith( "*" ) )
                {
                    exclude = exclude.substring( 0, exclude.length() - 1 );
                }
                while( exclude.startsWith( "**" ) )
                {
                    exclude = exclude.substring( 1 );
                }
                externalProxyExcludesCollection.add( exclude );
            }
        }
        else
        {
            externalProxyExcludesCollection = null;
        }
    }

    /**
     * Check host/port if excluded from external proxy.
     *
     * @param hostport  host/port to be checked if excluded from external proxy
     * @return          flag indicating whether or not the host/port is excluded from external proxy
     */
    public synchronized boolean isHostExcludedFromExternalProxy( String hostport )
    {
        if( externalProxyExcludesCollection != null )
        {
            for( Iterator iter = externalProxyExcludesCollection.iterator(); iter.hasNext();  )
            {
                String exclude = ( String )iter.next();
                if( exclude.startsWith( "*" ) )
                {
                    if( hostport.indexOf( exclude.substring( 1 ) ) != -1 )
                    {
                        return true;
                    }
                }
                else
                {
                    if( hostport.startsWith( exclude ) )
                    {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Get the destination directory location for proxy protocols.
     *
     * @return   Destination directory location for proxy protocols
     */
    public File getProtocolDirectoryLocation()
    {
        return protocolDirectoryLocation;
    }

    /**
     * Set the destination directory location for proxy protocols.
     *
     * @param protocolDirectoryLocation  Destination directory location for proxy protocols
     */
    public void setProtocolDirectoryLocation( File protocolDirectoryLocation )
    {
        this.protocolDirectoryLocation = protocolDirectoryLocation;
    }

    /**
     * Get the verbose state for System.out/err output during runtime.
     *
     * @return   switch for System.out/err output during runtime
     */
    public boolean isVerbose()
    {
        return verbose;
    }

    /**
     * Set the verbose switch for System.out/err output during runtime.
     *
     * @param verbose  switch for System.out/err output during runtime
     */
    public void setVerbose( boolean verbose )
    {
        this.verbose = verbose;
    }

    /**
     * Get the number of requests received.
     *
     * @return   Number of requests received
     */
    public long getRequestsReceived()
    {
        return this.requestsReceived;
    }

    /**
     * Increment the number of requests received by one.
     */
    private synchronized void incRequestsReceived()
    {
        this.requestsReceived++;
    }

    /**
     * Get the number of requests processed.
     *
     * @return   Number of requests processed
     */
    public long getRequestsProcessed()
    {
        return this.requestsProcessed;
    }

    /**
     * Increment the number of requests processed by one.
     */
    private synchronized void incRequestsProcessed()
    {
        this.requestsProcessed++;
    }

    /**
     * Get the number of bytes received in requests.
     *
     * @return   number of bytes received in requests
     */
    public long getBytesReceived()
    {
        return this.bytesReceived;
    }

    /**
     * Increment the number of bytes received in requests by the given amount.
     *
     * @param bytesReceived  additional number of bytes received
     */
    private synchronized void incBytesReceived( int bytesReceived )
    {
        this.bytesReceived += bytesReceived;
    }

    /**
     * Get the number of bytes sent with responses.
     *
     * @return   number of bytes sent with responses
     */
    public long getBytesSent()
    {
        return this.bytesSent;
    }

    /**
     * Increment the number of bytes sent with responses by the given amount.
     *
     * @param bytesSent  additional number of bytes sent
     */
    private synchronized void incBytesSent( int bytesSent )
    {
        this.bytesSent += bytesSent;
    }

    /**
     * Get the number of main request threads.
     *
     * @return   number of main request threads
     */
    public long getRunningMainRequestThreads()
    {
        return this.runningMainRequestThreads;
    }

    /**
     * Increment the number of main request threads.
     */
    private synchronized void incRunningMainRequestThreads()
    {
        this.runningMainRequestThreads++;
    }

    /**
     * Decrement the number of main request threads.
     */
    private synchronized void decRunningMainRequestThreads()
    {
        this.runningMainRequestThreads--;
    }

    /**
     * Check whether or not the proxy was started.
     *
     * @return   flag indicating whether or not the proxy was started
     */
    public boolean isStarted()
    {
        return started;
    }

    /**
     * Check whether or not the proxy was suspended.
     *
     * @return   flag indicating whether or not the proxy was suspended
     */
    public boolean isSuspended()
    {
        return suspended;
    }

    /**
     * Check whether or not the proxy is shutting down.
     *
     * @return   flag indicating whether or not the proxy is shutting down
     */
    public boolean isShuttingdown()
    {
        return shuttingdown;
    }

    /**
     * Handle request, i.e. receive request from client, redirect it to the appropriate server,
     * thereby rewrite it if necessary, redirect the response from the server in a separate thread
     * to the client and protocol each of these steps and the processed request/response buffers.
     *
     * @param clientSocket   socket of the client request connection
     * @param requestNumber  number of actual request
     */
    private void handleRequest( final String requestNumber, final Socket clientSocket )
    {
        // Increment running main threads counter
        incRunningMainRequestThreads();

        // Safeguard operations
        try
        {
            // Get protocol streams to dump request and response buffers
            final BufferedOutputStream requestProtocolStream = createRequestProtocolStream( requestNumber, clientSocket.getInetAddress(), clientSocket.getPort() );
            final BufferedOutputStream responseProtocolStream = createResponseProtocolStream( requestNumber, clientSocket.getInetAddress(), clientSocket.getPort() );

            // Safeguard operations
            try
            {
                byte[] requestBuffer = new byte[4096];
                int requestBufferRead = 0;
                String host = null;
                int port = 0;

                // Get streams from and to client
                final InputStream fromClientStream = clientSocket.getInputStream();
                final OutputStream toClientStream = clientSocket.getOutputStream();

                // Safeguard operations
                try
                {
                    // Read requestline
                    int requestBufferOffset = 0;
                    String requestCommand = null;
                    String requestLine = null;
                    int requestLinePos = 0;
                    while( requestLine == null )
                    {
                        if( ( requestBufferRead = fromClientStream.read( requestBuffer, requestBufferOffset, requestBuffer.length - requestBufferOffset ) ) != -1 )
                        {
                            printOutLn( "Request #" + requestNumber + ": Received " + requestBufferRead + " bytes of request from client " + clientSocket.getInetAddress() + ":" + clientSocket.getPort() + " and determining server the request should be sent to" );
                            incBytesReceived( requestBufferRead );
                            requestProtocolStream.write( requestBuffer, requestBufferOffset, requestBufferRead );
                            requestProtocolStream.flush();
                            for( int i = 0; i < requestBuffer.length; i++ )
                            {
                                if( requestBuffer[i] == ' ' )
                                {
                                    for( int j = i + 1; j < requestBuffer.length; j++ )
                                    {
                                        if( requestBuffer[j] == ' ' )
                                        {
                                            requestCommand = new String( requestBuffer, 0, i );
                                            if( ( !requestCommand.equalsIgnoreCase( "GET" ) ) &&
                                                ( !requestCommand.equalsIgnoreCase( "HEAD" ) ) &&
                                                ( !requestCommand.equalsIgnoreCase( "POST" ) ) )
                                            {
                                                throw new Exception( "Request #" + requestNumber + ": Failed to handle request! Request requires command \"" + requestCommand + "\", which is not supported by this proxy!" );
                                            }
                                            requestLine = new String( requestBuffer, requestLinePos = i + 1, j - i - 1 );
                                            j = requestBuffer.length;
                                        }
                                    }
                                    i = requestBuffer.length;
                                }
                            }
                            if( requestLine == null )
                            {
                                requestBufferOffset += requestBufferRead;
                                if( requestBufferOffset == requestBuffer.length )
                                {
                                    byte[] requestBufferTemp = new byte[requestBuffer.length << 1];
                                    System.arraycopy( requestBuffer, 0, requestBufferTemp, 0, requestBufferOffset );
                                    requestBuffer = requestBufferTemp;
                                }
                            }
                        }
                        else
                        {
                            throw new Exception( "Request #" + requestNumber + ": Failed to handle request! Request line incomplete!" );
                        }
                    }

                    // Determine destination
                    int pos = 0;
                    if( ( pos = requestLine.indexOf( SCHEME_PROTOCOL_SEPARATOR ) ) != -1 )
                    {
                        if( requestLine.substring( 0, pos ).equalsIgnoreCase( SCHEME_HTTP_ID ) )
                        {
                            pos += SCHEME_PROTOCOL_SEPARATOR.length();
                            int pathPos = requestLine.indexOf( "/", pos );
                            if( ( requestLine + "@" ).indexOf( "@", pos ) < pathPos )
                            {
                                throw new Exception( "Request #" + requestNumber + ": Failed to handle request! Request requires realm authentication, which is not supported by this proxy!" );
                            }
                            int portPos = requestLine.indexOf( ":", pos );
                            if( pathPos != -1 )
                            {
                                // Retrieve host and port from request line
                                if( portPos != -1 )
                                {
                                    host = requestLine.substring( pos, portPos );
                                    port = new Integer( requestLine.substring( portPos + 1, pathPos ) ).intValue();
                                }
                                else
                                {
                                    host = requestLine.substring( pos, pathPos );
                                    port = 80;
                                }

                                // Check for external proxy
                                if( getExternalProxyHost() != null )
                                {
                                    // Check if host is excluded from external proxy
                                    if( isHostExcludedFromExternalProxy( host + ":" + port ) )
                                    {
                                        // Use given host and port, so remove host/port from request
                                        System.arraycopy( requestBuffer, requestLinePos + pathPos, requestBuffer, requestLinePos, requestBufferRead - requestLinePos - pathPos );
                                        requestBufferRead -= pathPos;
                                    }
                                    else
                                    {
                                        // External proxy should handle request, so use external proxy host and port
                                        host = getExternalProxyHost();
                                        port = getExternalProxyPort();
                                    }
                                }
                                else
                                {
                                    // Use given host and port, so remove host/port from request
                                    System.arraycopy( requestBuffer, requestLinePos + pathPos, requestBuffer, requestLinePos, requestBufferRead - requestLinePos - pathPos );
                                    requestBufferRead -= pathPos;
                                }
                            }
                            else
                            {
                                throw new Exception( "Request #" + requestNumber + ": Failed to handle request! Request line misses document location!" );
                            }
                        }
                        else
                        {
                            throw new Exception( "Request #" + requestNumber + ": Failed to handle request! Request requires protocol \"" + requestLine.substring( 0, pos ) + "\", which is not supported by this proxy!" );
                        }
                    }
                    else
                    {
                        // Request contained no scheme/host/port, so use redirect host and port
                        host = getExternalRedirectHost();
                        port = getExternalRedirectPort();
                    }

                    // Safeguard operations
                    try
                    {
                        // Open connection to server
                        printOutLn( "Request #" + requestNumber + ": Opening connection to server " + host + ":" + port );
                        final Socket serverSocket = new Socket( host, port );

                        // Safeguard operations
                        try
                        {
                            // Get streams to and from server
                            final InputStream fromServerStream = serverSocket.getInputStream();
                            final OutputStream toServerStream = serverSocket.getOutputStream();

                            // Safeguard operations
                            try
                            {
                                // Redirect initial client request buffer to server in this thread
                                toServerStream.write( requestBuffer, 0, requestBufferRead );
                                toServerStream.flush();

                                // Redirect everything from server to client in separate thread
                                new Thread(
                                    new Runnable()
                                    {
                                        /**
                                         * Redirect everything from server to client in separate
                                         * thread
                                         */
                                        public void run()
                                        {
                                            // Safeguard operations
                                            try
                                            {
                                                byte[] responseBuffer = new byte[4096];
                                                int responseBufferRead = 0;
                                                while( ( responseBufferRead = fromServerStream.read( responseBuffer ) ) != -1 )
                                                {
                                                    printOutLn( "Request #" + requestNumber + ": Received " + responseBufferRead + " bytes of response from server " + serverSocket.getInetAddress() + ":" + serverSocket.getPort() + " and sending to client " + clientSocket.getInetAddress() + ":" + clientSocket.getPort() );
                                                    incBytesSent( responseBufferRead );
                                                    responseProtocolStream.write( responseBuffer, 0, responseBufferRead );
                                                    responseProtocolStream.flush();
                                                    toClientStream.write( responseBuffer, 0, responseBufferRead );
                                                    toClientStream.flush();
                                                }
                                            }
                                            catch( Exception exception )
                                            {
                                            }
                                            finally
                                            {
                                                // Server done or client terminated connection; closing streams
                                                try
                                                {
                                                    fromClientStream.close();
                                                }
                                                catch( Exception exception )
                                                {
                                                }
                                                try
                                                {
                                                    toClientStream.close();
                                                }
                                                catch( Exception exception )
                                                {
                                                }
                                                try
                                                {
                                                    fromServerStream.close();
                                                }
                                                catch( Exception exception )
                                                {
                                                }
                                                try
                                                {
                                                    toServerStream.close();
                                                }
                                                catch( Exception exception )
                                                {
                                                }
                                            }
                                        }
                                    } ).start();

                                // Redirect everything from client to server in this thread
                                while( ( requestBufferRead = fromClientStream.read( requestBuffer ) ) != -1 )
                                {
                                    printOutLn( "Request #" + requestNumber + ": Received " + requestBufferRead + " bytes of request from client " + clientSocket.getInetAddress() + ":" + clientSocket.getPort() + " and sending to server " + serverSocket.getInetAddress() + ":" + serverSocket.getPort() );
                                    incBytesReceived( requestBufferRead );
                                    requestProtocolStream.write( requestBuffer, 0, requestBufferRead );
                                    requestProtocolStream.flush();
                                    toServerStream.write( requestBuffer, 0, requestBufferRead );
                                    toServerStream.flush();
                                }
                            }
                            catch( Exception exception )
                            {
                            }
                            finally
                            {
                                // Server done or client terminated connection; close streams
                                try
                                {
                                    fromClientStream.close();
                                }
                                catch( Exception exception )
                                {
                                }
                                try
                                {
                                    toClientStream.close();
                                }
                                catch( Exception exception )
                                {
                                }
                                try
                                {
                                    fromServerStream.close();
                                }
                                catch( Exception exception )
                                {
                                }
                                try
                                {
                                    toServerStream.close();
                                }
                                catch( Exception exception )
                                {
                                }
                            }
                        }
                        catch( Exception exception )
                        {
                            printErrLn( "Request #" + requestNumber + ": Failed to open server streams: ", exception );
                        }
                        finally
                        {
                            // Close server connection
                            try
                            {
                                serverSocket.close();
                            }
                            catch( Exception exception )
                            {
                            }
                        }
                    }
                    catch( Exception exception )
                    {
                        printErrLn( "Request #" + requestNumber + ": Failed to open server connection: ", exception );
                    }
                }
                catch( Exception exception )
                {
                    printErrLn( "Request #" + requestNumber + ": Failed to read request line: ", exception );
                }
                finally
                {
                    // Close client input/output streams
                    try
                    {
                        fromClientStream.close();
                    }
                    catch( Exception exception )
                    {
                    }
                    try
                    {
                        toClientStream.close();
                    }
                    catch( Exception exception )
                    {
                    }
                }
            }
            catch( Exception exception )
            {
                printErrLn( "Request #" + requestNumber + ": Failed to open client streams: ", exception );
            }
            finally
            {
                // Close request/response protocol streams
                try
                {
                    requestProtocolStream.close();
                }
                catch( Exception exception )
                {
                    printErrLn( "Request #" + requestNumber + ": Failed to close request protocol stream: ", exception );
                }
                try
                {
                    responseProtocolStream.close();
                }
                catch( Exception exception )
                {
                    printErrLn( "Request #" + requestNumber + ": Failed to close response protocol stream: ", exception );
                }
            }
        }
        catch( Exception exception )
        {
            printErrLn( "Request #" + requestNumber + ": Failed to open protocol streams: ", exception );
        }
        finally
        {
            try
            {
                clientSocket.close();
            }
            catch( Exception exception )
            {
                printErrLn( "Request #" + requestNumber + ": Failed to close client connection: ", exception );
            }
        }

        // Decrement running main threads counter
        decRunningMainRequestThreads();

        // Increment requests processed counter
        incRequestsProcessed();

        // Done
        printErrLn( "Request #" + requestNumber + ": Done" );
    }

    /**
     * Create an output stream to protocol the actual request.
     *
     * @param host           host sending the request
     * @param port           port sending the request
     * @param requestNumber  number of actual request
     * @return               buffered output stream
     */
    private BufferedOutputStream createRequestProtocolStream( String requestNumber, InetAddress host, int port )
    {
        try
        {
            return new BufferedOutputStream( new FileOutputStream( new File( getProtocolDirectoryLocation(),
                getInstanceCounter() + "." + requestNumber + "." + "REQ" + "." +
                host.getHostName() + "=" + host.getHostAddress() + "@" + port + ".txt" ) ) );
        }
        catch( Exception exception )
        {
            printErrLn( "Failed to create request protocol output stream: ", exception );
            return null;
        }
    }

    /**
     * Create an output stream to protocol the actual response.
     *
     * @param host           host addressed for the response
     * @param port           port addressed for the response
     * @param requestNumber  number of actual request
     * @return               buffered output stream
     */
    private BufferedOutputStream createResponseProtocolStream( String requestNumber, InetAddress host, int port )
    {
        try
        {
            return new BufferedOutputStream( new FileOutputStream( new File( getProtocolDirectoryLocation(),
                getInstanceCounter() + "." + requestNumber + "." + "RES" + "." +
                host.getHostName() + "=" + host.getHostAddress() + "@" + port + ".txt" ) ) );
        }
        catch( Exception exception )
        {
            printErrLn( "Failed to create response protocol output stream: ", exception );
            return null;
        }
    }

    /**
     * Print text to STD.OUT.
     *
     * @param text  text to be printed
     */
    private void printOut( String text )
    {
        if( isVerbose() )
        {
            System.out.print( text );
        }
    }

    /**
     * Print text to STD.OUT and close line.
     *
     * @param text  text to be printed
     */
    private void printOutLn( String text )
    {
        if( isVerbose() )
        {
            System.out.println( text );
        }
    }

    /**
     * Print text to STD.ERR.
     *
     * @param text  text to be printed
     */
    private void printErr( String text )
    {
        if( isVerbose() )
        {
            System.err.print( text );
        }
    }

    /**
     * Print text to STD.ERR and close line.
     *
     * @param text  text to be printed
     */
    private void printErrLn( String text )
    {
        if( isVerbose() )
        {
            System.err.println( text );
        }
    }

    /**
     * Print text and exception to STD.ERR.
     *
     * @param text       text to be printed
     * @param exception  exception to be printed
     */
    private void printErr( String text, Exception exception )
    {
        if( isVerbose() )
        {
            System.err.print( text + exception );
            exception.printStackTrace( System.err );
        }
    }

    /**
     * Print text and exception to STD.ERR and close line.
     *
     * @param text       text to be printed
     * @param exception  exception to be printed
     */
    private void printErrLn( String text, Exception exception )
    {
        if( isVerbose() )
        {
            System.err.println( text + exception );
            exception.printStackTrace( System.err );
        }
    }
}
