/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id$
 */

package com.sapmarkets.technology.util;
import java.text.*;

import java.util.*;

/**
 * Interface describing internationalized access to messages, dates, numbers, etc. Every instance of
 * implementors of this interface is bound to a specific locale/timezone. Instances may be retrieved
 * calling the I18N management.
 *
 * @created   10. September 2001
 */
public interface II18NEnv
{
    public final static String DATE_PATTERN_DEFAULT = "yyyy-MM-dd HH:mm:ss:SSSS z";
    public final static int DATETIME_STYLE_SHORT = DateFormat.SHORT;
    public final static int DATETIME_STYLE_MEDIUM = DateFormat.MEDIUM;
    public final static int DATETIME_STYLE_LONG = DateFormat.LONG;
    public final static int DATETIME_STYLE_FULL = DateFormat.FULL;
    public final static int DATETIME_STYLE_NONE = DATETIME_STYLE_SHORT + DATETIME_STYLE_MEDIUM + DATETIME_STYLE_LONG + DATETIME_STYLE_FULL;
    public final static int CURRENCY_MINFRACTIONDIGITS_DEFAULT = 2;
    public final static int CURRENCY_MAXFRACTIONDIGITS_DEFAULT = 2;
    public final static int CURRENCY_MININTEGERDIGITS_DEFAULT = 1;
    public final static int CURRENCY_MAXINTEGERDIGITS_DEFAULT = Integer.MAX_VALUE;
    public final static int NUMBER_MINFRACTIONDIGITS_DEFAULT = 0;
    public final static int NUMBER_MAXFRACTIONDIGITS_DEFAULT = 3;
    public final static int NUMBER_MININTEGERDIGITS_DEFAULT = 1;
    public final static int NUMBER_MAXINTEGERDIGITS_DEFAULT = Integer.MAX_VALUE;
    public final static int PERCENT_MINFRACTIONDIGITS_DEFAULT = 0;
    public final static int PERCENT_MAXFRACTIONDIGITS_DEFAULT = 0;
    public final static int PERCENT_MININTEGERDIGITS_DEFAULT = 1;
    public final static int PERCENT_MAXINTEGERDIGITS_DEFAULT = Integer.MAX_VALUE;

    /**
     * Initialize i18n environment.
     *
     * @param locale    locale specific to this i18n environment
     * @param timezone  timezone specific to this i18n environment
     */
    public void init( Locale locale, TimeZone timezone );

    /**
     * Get the assigned locale.
     *
     * @return   assigned locale
     */
    public Locale getAssignedLocale();

    /**
     * Get the assigned timezone.
     *
     * @return   assigned timezone
     */
    public TimeZone getAssignedTimezone();

    /**
     * Set resource bundle by name to be used for accessing base-unspecified messages.
     *
     * @param base  base name resource bundle
     */
    public void setMessageBase( String base );

    /**
     * Get message identified by key using the set resource bundle.
     *
     * @param id  message id
     * @return    message
     */
    public String getMessage( String id );

    /**
     * Get message identified by key using the given resource bundle name.
     *
     * @param base  base name resource bundle
     * @param id    message id
     * @return      message
     */
    public String getMessage( String base, String id );

    /**
     * Get message identified by key using the set resource bundle, returning a default if
     * unavailable.
     *
     * @param id   message id
     * @param def  default message if requested message is not available
     * @return     message, evt. just default message
     */
    public String getMessageUsingDefaultIfUnavailable( String id, String def );

    /**
     * Get message identified by key using the given resource bundle name, returning a default if
     * unavailable.
     *
     * @param base  base name resource bundle
     * @param id    message id
     * @param def   default message if requested message is not available
     * @return      message, evt. just default message
     */
    public String getMessageUsingDefaultIfUnavailable( String base, String id, String def );

    /**
     * Get message identified by key using the set resource bundle and insert arguments according to
     * java.text.MessageFormat specification.
     *
     * @param id    message id
     * @param args  message arguments
     * @return      formatted message
     */
    public String formatMessage( String id, Object[] args );

    /**
     * Get message identified by key using the given resource bundle name and insert arguments
     * according to java.text.MessageFormat specification.
     *
     * @param base  base name resource bundle
     * @param id    message id
     * @param args  message arguments
     * @return      formatted message
     */
    public String formatMessage( String base, String id, Object[] args );

    /**
     * Get message identified by key using the set resource bundle, returning a default if not
     * available, and insert arguments according to java.text.MessageFormat specification.
     *
     * @param id    message id
     * @param def   default message if requested message is not available
     * @param args  message arguments
     * @return      formatted message, evt. just formatted default message
     */
    public String formatMessageUsingDefaultIfUnavailable( String id, String def, Object[] args );

    /**
     * Get message identified by key using the given resource bundle name, returning a default if
     * not available, and insert arguments according to java.text.MessageFormat specification.
     *
     * @param base  base name resource bundle
     * @param id    message id
     * @param def   default message if requested message is not available
     * @param args  message arguments
     * @return      formatted message, evt. just formatted default message
     */
    public String formatMessageUsingDefaultIfUnavailable( String base, String id, String def, Object[] args );

    /**
     * Set date format (see documentation on SimpleDateFormat) to be used for formatting
     * format-unspecified dates. Overwrites set date style.
     *
     * @param pattern  date pattern
     */
    public void setDateFormat( String pattern );

    /**
     * Set date format (see documentation on DateFormat) to be used for formatting
     * format-unspecified dates. Overwrites set date pattern.
     *
     * @param dateStyle  date formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @param timeStyle  time formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     */
    public void setDateFormat( int dateStyle, int timeStyle );

    /**
     * Format date according to locale/timezone using the set pattern/style.
     *
     * @param date  date to be formatted
     * @return      formatted date
     */
    public String formatDate( long date );

    /**
     * Format date according to locale/timezone using the set pattern/style.
     *
     * @param date  date to be formatted
     * @return      formatted date
     */
    public String formatDate( Date date );

    /**
     * Format date according to locale/timezone using the given pattern.
     *
     * @param date     date to be formatted
     * @param pattern  date pattern
     * @return         formatted date
     */
    public String formatDate( long date, String pattern );

    /**
     * Format date according to locale/timezone using the given pattern.
     *
     * @param date     date to be formatted
     * @param pattern  date pattern
     * @return         formatted date
     */
    public String formatDate( Date date, String pattern );

    /**
     * Format date according to locale/timezone using the given style.
     *
     * @param date       date to be formatted
     * @param dateStyle  date formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @param timeStyle  time formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @return           formatted date
     */
    public String formatDate( long date, int dateStyle, int timeStyle );

    /**
     * Format date according to locale/timezone using the given style.
     *
     * @param date       date to be formatted
     * @param dateStyle  date formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @param timeStyle  time formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @return           formatted date
     */
    public String formatDate( Date date, int dateStyle, int timeStyle );

    /**
     * Convert date style from string to int.
     *
     * @param style  string style
     * @return       int style
     */
    public int convertDateTimeStyle( String style );

    /**
     * Convert date style from int to string.
     *
     * @param style  int style
     * @return       string style
     */
    public String convertDateTimeStyle( int style );

    /**
     * Set currency format to be used for formatting format-unspecified currencies.
     *
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     */
    public void setCurrencyFormat( int minFractionDigits, int maxFractionDigits,
                                   int minIntegerDigits, int maxIntegerDigits );

    /**
     * Format currency according to locale using the set format.
     *
     * @param currency  currency to be formatted
     * @return          formatted currency
     */
    public String formatCurrency( long currency );

    /**
     * Format currency according to locale using the set format.
     *
     * @param currency  currency to be formatted
     * @return          formatted currency
     */
    public String formatCurrency( Number currency );

    /**
     * Format currency according to locale using the given format.
     *
     * @param currency           currency to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted currency
     */
    public String formatCurrency( long currency,
                                  int minFractionDigits, int maxFractionDigits,
                                  int minIntegerDigits, int maxIntegerDigits );

    /**
     * Format currency according to locale using the given format.
     *
     * @param currency           currency to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted currency
     */
    public String formatCurrency( Number currency,
                                  int minFractionDigits, int maxFractionDigits,
                                  int minIntegerDigits, int maxIntegerDigits );

    /**
     * Set number format to be used for formatting format-unspecified numbers.
     *
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     */
    public void setNumberFormat( int minFractionDigits, int maxFractionDigits,
                                 int minIntegerDigits, int maxIntegerDigits );

    /**
     * Format number according to locale using the set format.
     *
     * @param number  number to be formatted
     * @return        formatted number
     */
    public String formatNumber( long number );

    /**
     * Format number according to locale using the set format.
     *
     * @param number  number to be formatted
     * @return        formatted number
     */
    public String formatNumber( Number number );

    /**
     * Format number according to locale using the given format.
     *
     * @param number             number to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted number
     */
    public String formatNumber( long number,
                                int minFractionDigits, int maxFractionDigits,
                                int minIntegerDigits, int maxIntegerDigits );

    /**
     * Format number according to locale using the given format.
     *
     * @param number             number to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted number
     */
    public String formatNumber( Number number,
                                int minFractionDigits, int maxFractionDigits,
                                int minIntegerDigits, int maxIntegerDigits );

    /**
     * Set percent format to be used for formatting format-unspecified percents.
     *
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     */
    public void setPercentFormat( int minFractionDigits, int maxFractionDigits,
                                  int minIntegerDigits, int maxIntegerDigits );

    /**
     * Format percent according to locale using the set format.
     *
     * @param percent  percent to be formatted
     * @return         formatted percent
     */
    public String formatPercent( long percent );

    /**
     * Format percent according to locale using the set format.
     *
     * @param percent  percent to be formatted
     * @return         formatted percent
     */
    public String formatPercent( Number percent );

    /**
     * Format percent according to locale using the given format.
     *
     * @param percent            percent to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted percent
     */
    public String formatPercent( long percent,
                                 int minFractionDigits, int maxFractionDigits,
                                 int minIntegerDigits, int maxIntegerDigits );

    /**
     * Format percent according to locale using the given format.
     *
     * @param percent            percent to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted percent
     */
    public String formatPercent( Number percent,
                                 int minFractionDigits, int maxFractionDigits,
                                 int minIntegerDigits, int maxIntegerDigits );
}
