/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id$
 */

package com.sapmarkets.technology.util;

import java.util.*;

/**
 * Class giving internationalized access to messages, dates, numbers, etc. When calling the member
 * functions without a locale/timezone, the default context locale/timezone will be assumed
 * otherwise the explicit given locale/timezone will be used. You may retrieve an i18n environment
 * for convenience which is fixed to use the locale/timezone used during it's instantiation.
 *
 * @created   10. September 2001
 */
public class I18NMgmt implements II18NMgmt
{
    protected static II18NMgmt inst = null;

    protected HashMap envs_map = null;

    /**
     * Construct object of class I18NMgmt.
     */
    protected I18NMgmt()
    {
        init( null, null );
    }

    /**
     * Initialize i18n management.
     *
     * @param locale    locale specific to this i18n environment
     * @param timezone  timezone specific to this i18n environment
     */
    public void init( Locale locale, TimeZone timezone )
    {
        envs_map = new HashMap();
    }

    /**
     * Get instance of i18n environment bound to the default locale/timezone.
     *
     * @return   i18n environment
     */
    public II18NEnv getI18NEnvInstance()
    {
        return getI18NEnvInstance( null, null );
    }

    /**
     * Get instance of i18n environment bound to specific locale/the default timezone.
     *
     * @param locale  locale to bind i18n environment to
     * @return        i18n environment
     */
    public II18NEnv getI18NEnvInstance( Locale locale )
    {
        return getI18NEnvInstance( locale, null );
    }

    /**
     * Get instance of i18n environment bound to specific locale/timezone.
     *
     * @param locale    locale to bind i18n environment to
     * @param timezone  timezone to bind i18n environment to
     * @return          i18n environment
     */
    public II18NEnv getI18NEnvInstance( Locale locale, TimeZone timezone )
    {
        if( locale == null )
        {
            locale = getDefaultLocale();
        }
        if( timezone == null )
        {
            timezone = getDefaultTimezone();
        }
        String key = locale.getDisplayName() + "__" + timezone.getDisplayName();
        II18NEnv env = ( II18NEnv )envs_map.get( key );
        if( env == null )
        {
            env = new I18NEnv( locale, timezone );
            envs_map.put( key, env );
        }
        return env;
    }

    /**
     * Get the default locale.
     *
     * @return   default locale
     */
    public Locale getDefaultLocale()
    {
        return Locale.getDefault();
    }

    /**
     * Get the default timezone.
     *
     * @return   default timezone
     */
    public TimeZone getDefaultTimezone()
    {
        return TimeZone.getDefault();
    }

    /**
     * Get singleton instance of this class.
     *
     * @return   singleton instance
     */
    public static synchronized II18NMgmt getInstance()
    {
        // Check if singleton wasn't instantiated yet
        if( inst == null )
        {
            // Instantiate singleton instance now
            try
            {
                inst = new I18NMgmt();
            }
            catch( Exception exception )
            {
                // Nothing to do except for logging/tracing since inst wasn't assigned yet
            }
        }

        // Return singleton instance
        return inst;
    }
}
