/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id$
 */

package com.sapmarkets.technology.util;

import java.text.*;
import java.util.*;

/**
 * Class giving internationalized access to messages, dates, numbers, etc. Every instance of this
 * class is bound to a specific locale/timezone. Instances may be retrieved calling the I18N
 * management.
 *
 * @created   10. September 2001
 */
public class I18NEnv implements II18NEnv
{
    protected Locale locale = null;
    protected TimeZone timezone = null;

    protected static boolean showKey = false;
    protected ResourceBundle resourceBundleSet = null;
    protected HashMap resourceBundlesMap = new HashMap();

    protected Format dateFormatSet = null;
    protected HashMap dateFormatsMap = new HashMap();

    protected Format currencyFormatSet = null;
    protected HashMap currencyFormatsMap = new HashMap();

    protected Format numberFormatSet = null;
    protected HashMap numberFormatsMap = new HashMap();

    protected Format percentFormatSet = null;
    protected HashMap percentFormatsMap = new HashMap();

    /**
     * Construct object of class I18NEnv.
     */
    protected I18NEnv()
    {
        init( null, null );
    }

    /**
     * Construct object of class I18NEnv.
     *
     * @param locale  locale specific to this i18n environment
     */
    protected I18NEnv( Locale locale )
    {
        init( locale, null );
    }

    /**
     * Construct object of class I18NEnv.
     *
     * @param locale    locale specific to this i18n environment
     * @param timezone  timezone specific to this i18n environment
     */
    protected I18NEnv( Locale locale, TimeZone timezone )
    {
        init( locale, timezone );
    }

    /**
     * Initialize i18n environment.
     *
     * @param locale    locale specific to this i18n environment
     * @param timezone  timezone specific to this i18n environment
     */
    public void init( Locale locale, TimeZone timezone )
    {
        this.locale = locale;
        if( this.locale == null )
        {
            this.locale = I18NMgmt.getInstance().getDefaultLocale();
        }
        this.timezone = timezone;
        if( this.timezone == null )
        {
            this.timezone = I18NMgmt.getInstance().getDefaultTimezone();
        }
        dateFormatSet = getDateFormat( DATE_PATTERN_DEFAULT );
        currencyFormatSet = getCurrencyFormat(
            CURRENCY_MINFRACTIONDIGITS_DEFAULT, CURRENCY_MAXFRACTIONDIGITS_DEFAULT,
            CURRENCY_MININTEGERDIGITS_DEFAULT, CURRENCY_MAXINTEGERDIGITS_DEFAULT );
        numberFormatSet = getNumberFormat(
            NUMBER_MINFRACTIONDIGITS_DEFAULT, NUMBER_MAXFRACTIONDIGITS_DEFAULT,
            NUMBER_MININTEGERDIGITS_DEFAULT, NUMBER_MAXINTEGERDIGITS_DEFAULT );
        percentFormatSet = getPercentFormat(
            PERCENT_MINFRACTIONDIGITS_DEFAULT, PERCENT_MAXFRACTIONDIGITS_DEFAULT,
            PERCENT_MININTEGERDIGITS_DEFAULT, PERCENT_MAXINTEGERDIGITS_DEFAULT );
    }

    /**
     * Get the assigned locale.
     *
     * @return   assigned locale
     */
    public Locale getAssignedLocale()
    {
        return locale;
    }

    /**
     * Get the assigned timezone.
     *
     * @return   assigned timezone
     */
    public TimeZone getAssignedTimezone()
    {
        return timezone;
    }

    /**
     * Set resource bundle by name to be used for accessing base-unspecified messages.
     *
     * @param base  base name resource bundle
     */
    public void setMessageBase( String base )
    {
        resourceBundleSet = getMessageBase( base );
    }

    /**
     * Get instance of resource bundle from local cache bound to a specific message base (creating
     * it if not available). This method is expected to return a non-null resource bundle.
     *
     * @param base  base name resource bundle
     * @return      resource bundle
     */
    protected ResourceBundle getMessageBase( String base )
    {
        ResourceBundle resourceBundle = ( ResourceBundle )resourceBundlesMap.get( base );
        if( resourceBundle == null )
        {
            resourceBundle = ResourceBundle.getBundle( base, locale );
            resourceBundlesMap.put( base, resourceBundle );
        }
        return resourceBundle;
    }

    /**
     * Set flag whether or not the key should be displayed with the value. This method comes handy
     * during localization/translation.
     *
     * @param flag  flag whether or not the key should be displayed with the value
     */
    public static void showKey( boolean flag )
    {
        showKey = flag;
    }

    /**
     * Get message identified by key using the set resource bundle.
     *
     * @param id  message id
     * @return    message
     */
    public String getMessage( String id )
    {
        return getMessage( resourceBundleSet, id );
    }

    /**
     * Get message identified by key using the given resource bundle name.
     *
     * @param base  base name resource bundle
     * @param id    message id
     * @return      message
     */
    public String getMessage( String base, String id )
    {
        ResourceBundle resourceBundle = null;
        try
        {
            resourceBundle = getMessageBase( base );
        }
        catch( Exception exception )
        {
            // Resource bundle/ID not found
        }
        return getMessage( resourceBundle, id );
    }

    /**
     * Get message identified by key using the given resource bundle.
     *
     * @param base  base resource bundle
     * @param id    message id
     * @return      message
     */
    protected String getMessage( ResourceBundle base, String id )
    {
        String message = null;
        try
        {
            message = base.getString( id );
        }
        catch( Exception exception )
        {
            // Resource bundle/ID not found
        }
        if( showKey )
        {
            message = base + ":" + id + " (" + message + ")";
        }
        return message;
    }

    /**
     * Get message identified by key using the set resource bundle, returning a default if
     * unavailable.
     *
     * @param id   message id
     * @param def  default message if requested message is not available
     * @return     message, evt. just default message
     */
    public String getMessageUsingDefaultIfUnavailable( String id, String def )
    {
        return getMessageUsingDefaultIfUnavailable( resourceBundleSet, id, def );
    }

    /**
     * Get message identified by key using the given resource bundle name, returning a default if
     * unavailable.
     *
     * @param base  base name resource bundle
     * @param id    message id
     * @param def   default message if requested message is not available
     * @return      message, evt. just default message
     */
    public String getMessageUsingDefaultIfUnavailable( String base, String id, String def )
    {
        ResourceBundle resourceBundle = null;
        try
        {
            resourceBundle = getMessageBase( base );
        }
        catch( Exception exception )
        {
            // Resource bundle/ID not found
        }
        return getMessageUsingDefaultIfUnavailable( resourceBundle, id, def );
    }

    /**
     * Get message identified by key using the given resource bundle, returning a default if
     * unavailable.
     *
     * @param base  base resource bundle
     * @param id    message id
     * @param def   default message if requested message is not available
     * @return      message, evt. just default message
     */
    protected String getMessageUsingDefaultIfUnavailable( ResourceBundle base, String id, String def )
    {
        String message = null;
        try
        {
            message = base.getString( id );
        }
        catch( Exception exception )
        {
            // Resource bundle/ID not found
        }
        if( message == null )
        {
            message = def;
        }
        if( showKey )
        {
            message = base + ":" + id + " (" + message + ")";
        }
        return message;
    }

    /**
     * Get message identified by key using the set resource bundle and insert arguments according to
     * java.text.MessageFormat specification.
     *
     * @param id    message id
     * @param args  message arguments
     * @return      formatted message
     */
    public String formatMessage( String id, Object[] args )
    {
        return formatMessage( resourceBundleSet, id, args );
    }

    /**
     * Get message identified by key using the given resource bundle name and insert arguments
     * according to java.text.MessageFormat specification.
     *
     * @param base  base name resource bundle
     * @param id    message id
     * @param args  message arguments
     * @return      formatted message
     */
    public String formatMessage( String base, String id, Object[] args )
    {
        ResourceBundle resourceBundle = null;
        try
        {
            resourceBundle = getMessageBase( base );
        }
        catch( Exception exception )
        {
            // Resource bundle/ID not found
        }
        return formatMessage( resourceBundle, id, args );
    }

    /**
     * Get message identified by key using the given resource bundle and insert arguments according
     * to java.text.MessageFormat specification.
     *
     * @param base  base resource bundle
     * @param id    message id
     * @param args  message arguments
     * @return      formatted message
     */
    protected String formatMessage( ResourceBundle base, String id, Object[] args )
    {
        String message = null;
        try
        {
            message = base.getString( id );
        }
        catch( Exception exception )
        {
            // Resource bundle/ID not found
        }
        if( message != null )
        {
            message = MessageFormat.format( message, args );
        }
        if( showKey )
        {
            message = base + ":" + id + " (" + message + ")";
        }
        return message;
    }

    /**
     * Get message identified by key using the set resource bundle, returning a default if not
     * available, and insert arguments according to java.text.MessageFormat specification.
     *
     * @param id    message id
     * @param def   default message if requested message is not available
     * @param args  message arguments
     * @return      formatted message, evt. just formatted default message
     */
    public String formatMessageUsingDefaultIfUnavailable( String id, String def, Object[] args )
    {
        return formatMessageUsingDefaultIfUnavailable( resourceBundleSet, id, def, args );
    }

    /**
     * Get message identified by key using the given resource bundle name, returning a default if
     * not available, and insert arguments according to java.text.MessageFormat specification.
     *
     * @param base  base name resource bundle
     * @param id    message id
     * @param def   default message if requested message is not available
     * @param args  message arguments
     * @return      formatted message, evt. just formatted default message
     */
    public String formatMessageUsingDefaultIfUnavailable( String base, String id, String def, Object[] args )
    {
        ResourceBundle resourceBundle = null;
        try
        {
            resourceBundle = getMessageBase( base );
        }
        catch( Exception exception )
        {
            // Resource bundle/ID not found
        }
        return formatMessageUsingDefaultIfUnavailable( resourceBundle, id, def, args );
    }

    /**
     * Get message identified by key using the given resource bundle, returning a default if not
     * available, and insert arguments according to java.text.MessageFormat specification.
     *
     * @param base  base resource bundle
     * @param id    message id
     * @param def   default message if requested message is not available
     * @param args  message arguments
     * @return      formatted message, evt. just formatted default message
     */
    protected String formatMessageUsingDefaultIfUnavailable( ResourceBundle base, String id, String def, Object[] args )
    {
        String message = null;
        try
        {
            message = base.getString( id );
        }
        catch( Exception exception )
        {
            // Resource bundle/ID not found
        }
        if( message == null )
        {
            message = def;
        }
        if( message != null )
        {
            message = MessageFormat.format( message, args );
        }
        if( showKey )
        {
            message = base + ":" + id + " (" + message + ")";
        }
        return message;
    }

    /**
     * Set date format (see documentation on SimpleDateFormat) to be used for formatting
     * format-unspecified dates. Overwrites set date style.
     *
     * @param pattern  date pattern
     */
    public void setDateFormat( String pattern )
    {
        dateFormatSet = getDateFormat( pattern );
    }

    /**
     * Set date format (see documentation on DateFormat) to be used for formatting
     * format-unspecified dates. Overwrites set date pattern.
     *
     * @param dateStyle  date formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @param timeStyle  time formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     */
    public void setDateFormat( int dateStyle, int timeStyle )
    {
        dateFormatSet = getDateFormat( "#D:" + convertDateTimeStyle( dateStyle ) + "#" +
            "#T:" + convertDateTimeStyle( timeStyle ) + "#" );
    }

    /**
     * Get instance of date format from local cache bound to a specific date format (creating it if
     * not available). This method is expected to return a non-null date format.
     *
     * @param pattern  TBD: Description of the incoming method parameter
     * @return         date format
     */
    protected Format getDateFormat( String pattern )
    {
        String key = pattern + "__" + timezone.getDisplayName();
        Format dateFormat = ( Format )dateFormatsMap.get( key );
        if( dateFormat == null )
        {
            if( pattern.startsWith( "#" ) )
            {
                int dateStylePos = pattern.indexOf( "#D:" );
                int timeStylePos = pattern.indexOf( "#T:" );
                int dateStyle = DATETIME_STYLE_NONE;
                int timeStyle = DATETIME_STYLE_NONE;
                if( dateStylePos != -1 )
                {
                    dateStyle = convertDateTimeStyle( pattern.substring( dateStylePos + 3, pattern.indexOf( "#", dateStylePos + 3 ) ) );
                }
                if( timeStylePos != -1 )
                {
                    timeStyle = convertDateTimeStyle( pattern.substring( timeStylePos + 3, pattern.indexOf( "#", timeStylePos + 3 ) ) );
                }
                if( dateStyle == DATETIME_STYLE_NONE )
                {
                    if( timeStyle == DATETIME_STYLE_NONE )
                    {
                        dateFormat = dateFormatSet;
                    }
                    else
                    {
                        dateFormat = DateFormat.getTimeInstance( timeStyle, locale );
                        ( ( DateFormat )dateFormat ).setTimeZone( timezone );
                        dateFormatsMap.put( key, dateFormat );
                    }
                }
                else
                {
                    if( timeStyle == DATETIME_STYLE_NONE )
                    {
                        dateFormat = DateFormat.getDateInstance( dateStyle, locale );
                        ( ( DateFormat )dateFormat ).setTimeZone( timezone );
                        dateFormatsMap.put( key, dateFormat );
                    }
                    else
                    {
                        dateFormat = DateFormat.getDateTimeInstance( dateStyle, timeStyle, locale );
                        ( ( DateFormat )dateFormat ).setTimeZone( timezone );
                        dateFormatsMap.put( key, dateFormat );
                    }
                }
            }
            if( dateFormat == null )
            {
                dateFormat = new SimpleDateFormat( pattern, locale );
                ( ( DateFormat )dateFormat ).setTimeZone( timezone );
                dateFormatsMap.put( key, dateFormat );
            }
        }
        return dateFormat;
    }

    /**
     * Format date according to locale/timezone using the set pattern/style.
     *
     * @param date  date to be formatted
     * @return      formatted date
     */
    public String formatDate( long date )
    {
        return formatDate( new Date( date ) );
    }

    /**
     * Format date according to locale/timezone using the set pattern/style.
     *
     * @param date  date to be formatted
     * @return      formatted date
     */
    public String formatDate( Date date )
    {
        return dateFormatSet.format( date );
    }

    /**
     * Format date according to locale/timezone using the given pattern.
     *
     * @param date     date to be formatted
     * @param pattern  date pattern
     * @return         formatted date
     */
    public String formatDate( long date, String pattern )
    {
        return formatDate( new Date( date ), pattern );
    }

    /**
     * Format date according to locale/timezone using the given pattern.
     *
     * @param date     date to be formatted
     * @param pattern  date pattern
     * @return         formatted date
     */
    public String formatDate( Date date, String pattern )
    {
        return getDateFormat( pattern ).format( date );
    }

    /**
     * Format date according to locale/timezone using the given style.
     *
     * @param date       date to be formatted
     * @param dateStyle  date formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @param timeStyle  time formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @return           formatted date
     */
    public String formatDate( long date, int dateStyle, int timeStyle )
    {
        return formatDate( new Date( date ), dateStyle, timeStyle );
    }

    /**
     * Format date according to locale/timezone using the given style.
     *
     * @param date       date to be formatted
     * @param dateStyle  date formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @param timeStyle  time formatting style (NONE, SHORT, MEDIUM, LONG or FULL style)
     * @return           formatted date
     */
    public String formatDate( Date date, int dateStyle, int timeStyle )
    {
        return getDateFormat( "#D:" + convertDateTimeStyle( dateStyle ) + "#" +
            "#T:" + convertDateTimeStyle( timeStyle ) + "#" ).format( date );
    }

    /**
     * Convert date style from string to int.
     *
     * @param style  string style
     * @return       int style
     */
    public int convertDateTimeStyle( String style )
    {
        if( "SHORT".equalsIgnoreCase( style ) )
        {
            return DATETIME_STYLE_SHORT;
        }
        if( "MEDIUM".equalsIgnoreCase( style ) )
        {
            return DATETIME_STYLE_MEDIUM;
        }
        if( "LONG".equalsIgnoreCase( style ) )
        {
            return DATETIME_STYLE_LONG;
        }
        if( "FULL".equalsIgnoreCase( style ) )
        {
            return DATETIME_STYLE_FULL;
        }
        return DATETIME_STYLE_NONE;
    }

    /**
     * Convert date style from int to string.
     *
     * @param style  int style
     * @return       string style
     */
    public String convertDateTimeStyle( int style )
    {
        if( style == DATETIME_STYLE_SHORT )
        {
            return "SHORT";
        }
        if( style == DATETIME_STYLE_MEDIUM )
        {
            return "MEDIUM";
        }
        if( style == DATETIME_STYLE_LONG )
        {
            return "LONG";
        }
        if( style == DATETIME_STYLE_FULL )
        {
            return "FULL";
        }
        return "NONE";
    }

    /**
     * Set currency format to be used for formatting format-unspecified currencies.
     *
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     */
    public void setCurrencyFormat( int minFractionDigits, int maxFractionDigits,
                                   int minIntegerDigits, int maxIntegerDigits )
    {
        currencyFormatSet = getCurrencyFormat( minFractionDigits, maxFractionDigits,
            minIntegerDigits, maxIntegerDigits );
    }

    /**
     * Get instance of currency format from local cache bound to a specific currency format
     * (creating it if not available). This method is expected to return a non-null currency format.
     *
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   currency format
     */
    protected Format getCurrencyFormat( int minFractionDigits, int maxFractionDigits,
                                        int minIntegerDigits, int maxIntegerDigits )
    {
        String spec = "#" + minFractionDigits + "#" + maxFractionDigits + "#" + minIntegerDigits + "#" + maxIntegerDigits;
        Format currencyFormat = ( Format )currencyFormatsMap.get( spec );
        if( currencyFormat == null )
        {
            currencyFormat = NumberFormat.getCurrencyInstance( locale );
            ( ( NumberFormat )currencyFormat ).setMinimumFractionDigits( minFractionDigits );
            ( ( NumberFormat )currencyFormat ).setMaximumFractionDigits( maxFractionDigits );
            ( ( NumberFormat )currencyFormat ).setMinimumIntegerDigits( minIntegerDigits );
            ( ( NumberFormat )currencyFormat ).setMaximumIntegerDigits( maxIntegerDigits );
            currencyFormatsMap.put( spec, currencyFormat );
        }
        return currencyFormat;
    }

    /**
     * Format currency according to locale using the set format.
     *
     * @param currency  currency to be formatted
     * @return          formatted currency
     */
    public String formatCurrency( long currency )
    {
        return formatCurrency( new Long( currency ) );
    }

    /**
     * Format currency according to locale using the set format.
     *
     * @param currency  currency to be formatted
     * @return          formatted currency
     */
    public String formatCurrency( Number currency )
    {
        return currencyFormatSet.format( currency );
    }

    /**
     * Format currency according to locale using the given format.
     *
     * @param currency           currency to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted currency
     */
    public String formatCurrency( long currency,
                                  int minFractionDigits, int maxFractionDigits,
                                  int minIntegerDigits, int maxIntegerDigits )
    {
        return formatCurrency( new Long( currency ),
            minFractionDigits, maxFractionDigits,
            minIntegerDigits, maxIntegerDigits );
    }

    /**
     * Format currency according to locale using the given format.
     *
     * @param currency           currency to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted currency
     */
    public String formatCurrency( Number currency,
                                  int minFractionDigits, int maxFractionDigits,
                                  int minIntegerDigits, int maxIntegerDigits )
    {
        return getCurrencyFormat( minFractionDigits, maxFractionDigits,
            minIntegerDigits, maxIntegerDigits ).format( currency );
    }

    /**
     * Set number format to be used for formatting format-unspecified numbers.
     *
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     */
    public void setNumberFormat( int minFractionDigits, int maxFractionDigits,
                                 int minIntegerDigits, int maxIntegerDigits )
    {
        numberFormatSet = getNumberFormat( minFractionDigits, maxFractionDigits,
            minIntegerDigits, maxIntegerDigits );
    }

    /**
     * Get instance of number format from local cache bound to a specific number format (creating it
     * if not available). This method is expected to return a non-null number format.
     *
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   number format
     */
    protected Format getNumberFormat( int minFractionDigits, int maxFractionDigits,
                                      int minIntegerDigits, int maxIntegerDigits )
    {
        String spec = "#" + minFractionDigits + "#" + maxFractionDigits + "#" + minIntegerDigits + "#" + maxIntegerDigits;
        Format numberFormat = ( Format )numberFormatsMap.get( spec );
        if( numberFormat == null )
        {
            numberFormat = NumberFormat.getNumberInstance( locale );
            ( ( NumberFormat )numberFormat ).setMinimumFractionDigits( minFractionDigits );
            ( ( NumberFormat )numberFormat ).setMaximumFractionDigits( maxFractionDigits );
            ( ( NumberFormat )numberFormat ).setMinimumIntegerDigits( minIntegerDigits );
            ( ( NumberFormat )numberFormat ).setMaximumIntegerDigits( maxIntegerDigits );
            numberFormatsMap.put( spec, numberFormat );
        }
        return numberFormat;
    }

    /**
     * Format number according to locale using the set format.
     *
     * @param number  number to be formatted
     * @return        formatted number
     */
    public String formatNumber( long number )
    {
        return formatNumber( new Long( number ) );
    }

    /**
     * Format number according to locale using the set format.
     *
     * @param number  number to be formatted
     * @return        formatted number
     */
    public String formatNumber( Number number )
    {
        return numberFormatSet.format( number );
    }

    /**
     * Format number according to locale using the given format.
     *
     * @param number             number to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted number
     */
    public String formatNumber( long number,
                                int minFractionDigits, int maxFractionDigits,
                                int minIntegerDigits, int maxIntegerDigits )
    {
        return formatNumber( new Long( number ),
            minFractionDigits, maxFractionDigits,
            minIntegerDigits, maxIntegerDigits );
    }

    /**
     * Format number according to locale using the given format.
     *
     * @param number             number to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted number
     */
    public String formatNumber( Number number,
                                int minFractionDigits, int maxFractionDigits,
                                int minIntegerDigits, int maxIntegerDigits )
    {
        return getNumberFormat( minFractionDigits, maxFractionDigits,
            minIntegerDigits, maxIntegerDigits ).format( number );
    }

    /**
     * Set percent format to be used for formatting format-unspecified percents.
     *
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     */
    public void setPercentFormat( int minFractionDigits, int maxFractionDigits,
                                  int minIntegerDigits, int maxIntegerDigits )
    {
        percentFormatSet = getPercentFormat( minFractionDigits, maxFractionDigits,
            minIntegerDigits, maxIntegerDigits );
    }

    /**
     * Get instance of percent format from local cache bound to a specific percent format (creating
     * it if not available). This method is expected to return a non-null percent format.
     *
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   percent format
     */
    protected Format getPercentFormat( int minFractionDigits, int maxFractionDigits,
                                       int minIntegerDigits, int maxIntegerDigits )
    {
        String spec = "#" + minFractionDigits + "#" + maxFractionDigits + "#" + minIntegerDigits + "#" + maxIntegerDigits;
        Format percentFormat = ( Format )percentFormatsMap.get( spec );
        if( percentFormat == null )
        {
            percentFormat = NumberFormat.getPercentInstance( locale );
            ( ( NumberFormat )percentFormat ).setMinimumFractionDigits( minFractionDigits );
            ( ( NumberFormat )percentFormat ).setMaximumFractionDigits( maxFractionDigits );
            ( ( NumberFormat )percentFormat ).setMinimumIntegerDigits( minIntegerDigits );
            ( ( NumberFormat )percentFormat ).setMaximumIntegerDigits( maxIntegerDigits );
            percentFormatsMap.put( spec, percentFormat );
        }
        return percentFormat;
    }

    /**
     * Format percent according to locale using the set format.
     *
     * @param percent  percent to be formatted
     * @return         formatted percent
     */
    public String formatPercent( long percent )
    {
        return formatPercent( new Long( percent ) );
    }

    /**
     * Format percent according to locale using the set format.
     *
     * @param percent  percent to be formatted
     * @return         formatted percent
     */
    public String formatPercent( Number percent )
    {
        return percentFormatSet.format( percent );
    }

    /**
     * Format percent according to locale using the given format.
     *
     * @param percent            percent to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted percent
     */
    public String formatPercent( long percent,
                                 int minFractionDigits, int maxFractionDigits,
                                 int minIntegerDigits, int maxIntegerDigits )
    {
        return formatPercent( new Long( percent ),
            minFractionDigits, maxFractionDigits,
            minIntegerDigits, maxIntegerDigits );
    }

    /**
     * Format percent according to locale using the given format.
     *
     * @param percent            percent to be formatted
     * @param minFractionDigits  minimum number of digits for fraction part of number
     * @param maxFractionDigits  maximum number of digits for fraction part of number
     * @param minIntegerDigits   minimum number of digits for integer part of number
     * @param maxIntegerDigits   maximum number of digits for integer part of number
     * @return                   formatted percent
     */
    public String formatPercent( Number percent,
                                 int minFractionDigits, int maxFractionDigits,
                                 int minIntegerDigits, int maxIntegerDigits )
    {
        return getPercentFormat( minFractionDigits, maxFractionDigits,
            minIntegerDigits, maxIntegerDigits ).format( percent );
    }
}
