/*
 * SAPMarkets Copyright (c) 2001
 * All rights reserved
 *
 * @version $Id: //sapmarkets/BaseTech/630_VAL_REL/src/_util/java/com/sapmarkets/technology/util/Files.java#1 $
 */

package com.sapmarkets.technology.util;

import java.io.*;
import java.net.*;
import java.text.*;
import java.util.*;

/**
 * Class providing some file access methods.
 *
 * @created   23. Oktober 2001
 */
public class Files
{
    /**
     * standard execution observer that executes the copy and deletion
     * operation without calling a further observer.
     */
    private static IFilesObserver execute=new ExecutionObserver(null);

    /**
     * Copy file or folder recursively from source to destination.
     * All files in destination will be
     * removed beforehand.
     *
     * @param srcFile   source file
     * @param dstFile   destination file
     * @return          success return flag
     */
    public static boolean copyFile( File srcFile, File dstFile )
    {
        return handleCopyFile( srcFile, dstFile, execute );
    }

    /**
     * Copy file or folder recursively from source to destination.
     * All files in destination will be
     * removed beforehand.
     *
     * @param srcFile   source file
     * @param dstFile  destination file
     * @param observer  observer called on file operations
     * @return          success return flag
     */
    public static boolean copyFile( File srcFile, File dstFile,
				    IFilesObserver observer )
    { return handleCopyFile(srcFile, dstFile, observer==null?execute:
				  (observer instanceof ExecutionObserver?
				    observer:new ExecutionObserver(observer)));
    }

    /**
     * Handle copy file or folder recursively from source to destination.
     * All files in destination will be removed beforehand.
     * Only the observer is called. No other opertaion modifying a file
     * is executed directly.
     *
     *
     * @param srcFile   source file
     * @param dstFile  destination file
     * @param observer  observer called on file operations
     * @return          success return flag
     */
    private static boolean handleCopyFile( File srcFile, File dstFile,
					   IFilesObserver observer)
    {
      // Call observer
      observer.onCopyFileInvoke( srcFile, dstFile );

      // Check if source file exists
      if( srcFile.exists() ) {

	// Remove destination file
	if( dstFile.exists() ) {
	  handleDeleteFile( dstFile, observer );
	}

	// Check if source file is a directory
	if( srcFile.isDirectory() ) {
	  File[] srcFiles = srcFile.listFiles();

	  // Recursively call this method with content of source directory
	  if( srcFiles != null ) {
	    for( int i = 0; i < srcFiles.length; i++ ) {
	      handleCopyFile( srcFiles[i], 
			      new File( dstFile, srcFiles[i].getName() ),
			      observer);
	    }
	  }
	} // srcFile.isDirectory()

	// Call observer
	observer.onCopyFileProcessed( srcFile, dstFile );

	// Return success
	return true;
      }
      else {
	// Nothing to do
	return false;
      }
    }

    /**
     * Delete file or folder recursively.
     *
     * @param file  file to be deleted
     * @return      success return flag
     */
    public static boolean deleteFile( File file )
    {
        return handleDeleteFile( file, execute );
    }

    /**
     * Delete file or folder recursively.
     *
     * @param file      file to be deleted
     * @param observer  observer called on file operations
     * @return          success return flag
     */
    public static boolean deleteFile( File file, IFilesObserver observer )
    { return handleDeleteFile(file, observer==null?execute:
				  (observer instanceof ExecutionObserver?
				    observer:new ExecutionObserver(observer)));
    }

    /**
     * Handle delete file or folder recursively.
     * Only the observer is called for all files and directories that would be
     * deleted. No other opertaion modifying a file is executed directly.
     *
     * @param file        file to be deleted
     * @param observer    observer called on file operations
     *                    
     * @return            success return flag
     */
    private static boolean handleDeleteFile( File file, IFilesObserver observer)
    {
      // Call observer
      observer.onDeleteFileInvoke( file );

      // Check if file exists
      if( file.exists() ) {
	// Check if file is a directory
	if( file.isDirectory() ) {
	  // Check for contained files
	  File[] files = file.listFiles();
	  if( files != null ) {
	    for( int i = 0; i < files.length; i++ ) {
	      handleDeleteFile( files[i], observer);
	    }
	  }
	}

	// Call observer
	observer.onDeleteFileProcessed( file );

	// Return success
	return true;
      }
      return false;
    }

    /**
     * Synchronize file or folder recursively from source to destination based
     * on last modification time stamps. Files in destination, but last
     * modified before last modification of the same file in source, will be
     * replaced. Files in destination, but not in source will be removed.
     *
     * @param srcFile   source file
     * @param dstFile  destination file
     * @return          success return flag
     */
    public static boolean syncFileOld( File srcFile, File dstFile )
    {
        return syncFileOld( srcFile, dstFile, null );
    }

    /**
     * Synchronize file or folder recursively from source to destination based
     * on last modification time stamps.
     * Files in destination, but last modified before last modification of the
     * same file in source, will be replaced. Files in destination, but not in
     * source will be removed. In addition to the method signature without the
     * observer argument, this method will call the given observer on all file
     * operations.
     *
     * @param srcFile   source file
     * @param dstFile  destination file
     * @param observer  observer called on file operations
     * @return          success return flag
     */
    public static boolean syncFileOld( File srcFile, File dstFile,
				       IFilesObserver observer )
    {
        // Call observer
        if( observer != null )
        {
            observer.onSyncFileInvoke( srcFile, dstFile );
        }

        // Check if source file exists
        if( srcFile.exists() )
        {
            // Check if source file is a directory
            if( srcFile.isDirectory() )
            {
                File[] srcFiles = srcFile.listFiles();

                // Check if destination file exists
                if( dstFile.exists() )
                {
                    // Remove destination file if of different type than source file
                    if( dstFile.isFile() )
                    {
                        deleteFile( dstFile, observer );
                    }
                    else
                    {
                        // Remove all files from destination directory which are not contained within source directory
                        boolean foundFile;
                        File[] dstFiles = dstFile.listFiles();
                        if( dstFiles != null )
                        {
                            for( int i = 0; i < dstFiles.length; i++ )
                            {
                                // Search destination file in source directory
                                foundFile = false;
                                for( int j = 0; j < srcFiles.length; j++ )
                                {
                                    if( dstFiles[i].getName().equals( srcFiles[j].getName() ) )
                                    {
                                        foundFile = true;
                                    }
                                }

                                // Remove destination file if not found in source directory
                                if( !foundFile )
                                {
                                    deleteFile( dstFiles[i], observer );
                                }
                            }
                        }
                    }
                }

                // Recursively call this method with content of source directory
                dstFile.mkdirs();
                if( srcFiles != null )
                {
                    for( int i = 0; i < srcFiles.length; i++ )
                    {
                        syncFileOld( srcFiles[i], new File( dstFile, srcFiles[i].getName() ), observer );
                    }
                }
            }
            else
            {
                // Check if destination file exists
                if( dstFile.exists() )
                {
                    // Remove destination file if of different type than source file
                    if( dstFile.isDirectory() )
                    {
                        deleteFile( dstFile, observer );
                    }
                    else
                    {
                        // Remove destination file if older than source file
                        if( ( dstFile.lastModified() < srcFile.lastModified() ) )
                        {
                            deleteFile( dstFile, observer );
                        }
                    }
                }

                // Copy source file if destination file does not exist
                if( !dstFile.exists() )
                {
                    // Copy source file
                    copyFile( srcFile, dstFile, observer );
                }
            }

            // Call observer
            if( observer != null )
            {
                observer.onSyncFileProcessed( srcFile, dstFile );
            }

            // Return success
            return true;
        }
        else
        {
            // Remove destination file
            if( deleteFile( dstFile, observer ) )
            {
                // Call observer
                if( observer != null )
                {
                    observer.onSyncFileProcessed( srcFile, dstFile );
                }

                // Return success
                return true;
            }
            else
            {
                // Operation failed
                return false;
            }
        }
    }

    /**
     * Check for required synchronization of a file or folder recursively from
     * source to destination based on last modification time stamps. Files in
     * destination, but last modified before last modification of the same
     * file in source, will be replaced. Files in destination, but not in
     * source will be removed. In addition to the method signature without
     * the observer argument, this method will call the
     * given observer on all file operations.
     *
     * @param srcFile   source file
     * @param dstFile  destination file
     * @param observer  observer called on file operations
     * @return          true if update was required
     */
    public static boolean checkSyncFile(File srcFile, File dstFile,
					IFilesObserver observer )
    { boolean required=false;

      // Call observer
      if ( observer != null ) {
	observer.onSyncFileInvoke( srcFile, dstFile );
      }

      // Check if source file exists
      if( srcFile.exists() ) {
	// Check if source file is a directory
	if( srcFile.isDirectory() ) {
	  File[] srcFiles = srcFile.listFiles();

	  // Check if destination file exists
	  if( dstFile.exists() ) {
	    // Remove destination file if of different type than source file
	    if( !dstFile.isDirectory() ) {
	      required|=handleDeleteFile( dstFile, observer);
	    }
	    else {
	      // Remove all files from destination directory which are not
	      // contained within source directory
	      boolean foundFile;
	      File[] dstFiles = dstFile.listFiles();
	      if( dstFiles != null ) {
		for( int i = 0; i < dstFiles.length; i++ ) {
		  // Search destination file in source directory
		  foundFile = false;
		  for( int j = 0; j < srcFiles.length; j++ ) {
		    if( dstFiles[i].getName().equals(srcFiles[j].getName())) {
		      foundFile = true;
		      break;
		    }
		  }

		  // Remove destination file if not found in source directory
		  if( !foundFile ) {
		    required|=handleDeleteFile( dstFiles[i], observer);
		  }
		}
	      }
	    }
	  } // dstFile.exists()

	  // Recursively call this method with content of source directory
	  if( srcFiles != null ) {
	    for( int i = 0; i < srcFiles.length; i++ ) { 
	      required|=checkSyncFile( srcFiles[i], 
				   new File( dstFile, srcFiles[i].getName() ), 
				   observer );
	    }
	  }
	} // srcFile.isDirectory()
	else {
	  // Check if destination file exists
	  if( dstFile.exists() ) {
	    // Remove destination file if of different type than source file
	    if( dstFile.isDirectory() ) {
	      required|=handleDeleteFile( dstFile, observer);
	    }
	    else {
	      // Remove destination file if older than source file
	      if( ( dstFile.lastModified() < srcFile.lastModified() ) ) {
		required|=handleDeleteFile( dstFile, observer);
	      }
	    }
	  }

	  // Copy source file if destination file does not exist
	  if( !dstFile.exists() ) {
	    // Copy source file
	    required|=handleCopyFile( srcFile, dstFile, observer);
	  }
	}
      } // srcFile.exists()
      else {
	// Remove destination file
	if( handleDeleteFile( dstFile, observer) ) {
	  required=true;
	}
      }

      // Call observer
      if( observer != null ) {
	observer.onSyncFileProcessed( srcFile, dstFile );
      }

      return required;
    }


    /**
     * Synchronize file or folder recursively from source to destination based
     * on last modification time stamps. Files in destination, but last
     * modified before last modification of the same file in source, will be
     * replaced. Files in destination, but not in source will be removed.
     *
     * @param srcFile   source file
     * @param dstFile  destination file
     * @return          success return flag
     */
    public static boolean syncFile( File srcFile, File dstFile )
    {
        return checkSyncFile(srcFile, dstFile, execute);
    }

    /**
     * Synchronize file or folder recursively from source to destination based
     * on last modification time stamps.
     * Files in destination, but last modified before last modification of the
     * same file in source, will be replaced. Files in destination, but not in
     * source will be removed. In addition to the method signature without the
     * observer argument, this method will call the given observer on all file
     * operations.
     *
     * @param srcFile   source file
     * @param dstFile  destination file
     * @param observer  observer called on file operations
     * @return          success return flag
     */
    public static boolean syncFile( File srcFile, File dstFile,
				       IFilesObserver observer )
    { return checkSyncFile(srcFile, dstFile, observer==null?execute:
				  (observer instanceof ExecutionObserver?
				    observer:new ExecutionObserver(observer)));
    }

    /**
     * Check for required synchronization of a file or folder recursively from
     * source to destination based on last modification time stamps. Files in
     * destination, but last modified before last modification of the same
     * file in source, will be replaced. Files in destination, but not in
     * source will be removed. The required operations will be returned
     * as a collection of execution nodes.
     *
     * @param srcFile   source file
     * @param dstFile  destination file
     * @param observer  observer called on file operations
     * @return          success return flag
     */
    public static Collection checkSyncFile( File srcFile, File dstFile )
    { CollectObserver observer=new CollectObserver();
      checkSyncFile( srcFile, dstFile, observer );
      return observer.operations;
    }


    public static void executeSyncCollection(Collection c)
    { Iterator elems=c.iterator();
      while (elems.hasNext()) {
	ExecutionNode n=(ExecutionNode)elems.next();
	n.execute();
      }
    }


    /**
     * Fix file path, i.e. make it canonical without calling the appropriate slow method of File.
     *
     * @param filePath  file path to be fixed
     * @return          fixed file path
     */
    public static String fixFilePath( String filePath )
    {
        return fixFilePath( filePath, null );
    }

    /**
     * Fix file path, i.e. make it canonical without calling the appropriate slow method of File.
     * Extend file path by absolute file path if file path is relative.
     *
     * @param filePath     file path to be fixed
     * @param absFilePath  absolute file path to be used if file path is relative
     * @return             fixed file path
     */
    public static String fixFilePath( String filePath, String absFilePath )
    {
        if( filePath != null )
        {
            int pos = -1;

            // Fix file path separators
            filePath = fixFilePathSeparators( filePath );

            // Check for root relative file path and make it absolute
            if( absFilePath != null )
            {
                if( filePath.startsWith( "." ) )
                {
                    filePath = fixFilePathSeparators( absFilePath ) + File.separator + filePath;
                }
                if( filePath.startsWith( File.separatorChar + "." ) )
                {
                    filePath = fixFilePathSeparators( absFilePath ) + filePath;
                }
            }

            // Fix beginning of DOS file path
            if( filePath.indexOf( ":" ) != -1 )
            {
                while( filePath.startsWith( File.separator ) )
                {
                    filePath = filePath.substring( File.separator.length() );
                }
            }

            // Remove duplicate file path separators
            while( ( pos = filePath.indexOf( File.separator + File.separator ) ) != -1 )
            {
                filePath = filePath.substring( 0, pos ) + filePath.substring( pos + 1 );
            }

            // Remove relative path dots
            while( ( pos = filePath.indexOf( File.separator + "." + File.separator ) ) != -1 )
            {
                filePath = filePath.substring( 0, pos ) + filePath.substring( pos + File.separator.length() + 1 );
            }

            // Resolve parent path dots
            while( ( pos = filePath.indexOf( File.separator + ".." + File.separator ) ) != -1 )
            {
                filePath = filePath.substring( 0, filePath.lastIndexOf( File.separator, pos - File.separator.length() ) ) + filePath.substring( pos + File.separator.length() + 2 );
            }

            // Remove trailing file path separators
            while( filePath.endsWith( File.separator ) )
            {
                filePath = filePath.substring( 0, filePath.length() - File.separator.length() );
            }
        }

        // Return fixed file path
        return filePath;
    }

    /**
     * Fix file path separators, i.e. either change '/' or '\' into the correct file path separator.
     *
     * @param filePath  file path to be fixed
     * @return          fixed file path
     */
    public static String fixFilePathSeparators( String filePath )
    {
        if( File.separatorChar == '\\' )
        {
            return filePath.replace( '/', '\\' );
        }
        if( File.separatorChar == '/' )
        {
            return filePath.replace( '\\', '/' );
        }
        return filePath;
    }


  /*
   * nested classes for file processing
   */

  /**
   * File operation execution observer.
   * This observer executes the necessary file operations to achieve
   * the file modifications notifyed to the observer interface.
   * This observer is used by the handleXXX methods to really do the
   * work.
   */ 
  private static class ExecutionObserver implements IFilesObserver {
    private IFilesObserver observer;

    ExecutionObserver()
    { this(null);
    }

    ExecutionObserver(IFilesObserver observer)
    { this.observer=observer;
    }

    public void onCopyFileInvoke( File srcFile, File dstFile )
    {
      if (observer!=null) observer.onCopyFileInvoke(srcFile, dstFile);
    }

    public void onCopyFileProcessed( File srcFile, File dstFile )
    {
      if (srcFile.isDirectory()) {
	// do not recursivly copy a directory, this
	// was already handled by separate calls for each included file
	// just assert that the directory is created
	dstFile.mkdirs();
      }
      else { 
	// copy the file content
	InputStream srcStream=null;
	OutputStream destStream=null;

	// Copy file
	try {
	  dstFile.getParentFile().mkdirs();
	  srcStream = new FileInputStream( srcFile );
	  destStream = new FileOutputStream( dstFile );
	  byte[] data = new byte[4096];
	  int count = 0;
	  while( ( count = srcStream.read( data ) ) != -1 ) {
	    destStream.write( data, 0, count );
	  }
	}
	catch( Exception exception )
	{
	  return;
	}
	finally {
	  try {
	    if (destStream!=null) destStream.close();
	  }
	  catch( Exception exception ) {}
	  try {
	    if (srcStream!=null)  srcStream.close();
	  }
	  catch( Exception exception ) {}
	}
      }

      // call the observer
      if (observer!=null) observer.onCopyFileProcessed( srcFile, dstFile );
    }

    public void onDeleteFileInvoke( File file )
    {
      if (observer!=null) observer.onDeleteFileInvoke(file);
    }

    public void onDeleteFileProcessed( File file )
    { if (file.delete()) {
	if (observer!=null) observer.onDeleteFileProcessed(file);
      }
    }

    public void onSyncFileInvoke( File srcFile, File dstFile )
    {
      if (observer!=null) observer.onSyncFileInvoke( srcFile, dstFile );
    }

    public void onSyncFileProcessed( File srcFile, File dstFile )
    {
      if (observer!=null) observer.onSyncFileProcessed( srcFile, dstFile );
    }
  }

  /**
   * File operation collection observer.
   * This observer collects the necessary file operations to achieve
   * the file modifications notifyed to the observer interface.
   * This observer is used by the handleXXX methods to note down
   * operations to be done.
   */ 
  private static class CollectObserver extends FilesObserver {
    Vector operations=new Vector();

    CollectObserver()
    {
    }

    public Collection getOperations()
    { return operations;
    }

    public void onCopyFileProcessed( File srcFile, File dstFile )
    { ExecutionNode n=new CopyNode(srcFile,dstFile);
      operations.add(n);
    }

    public void onDeleteFileProcessed( File file )
    { ExecutionNode n=new DeleteNode(file);
      operations.add(n);
    }
  }

  public interface ExecutionNode {
    public void execute();
  }

  public static class CopyNode implements ExecutionNode {
    File srcFile;
    File dstFile;

    public CopyNode(File srcFile, File dstFile)
    { this.srcFile=srcFile;
      this.dstFile=dstFile;
    }

    public void execute()
    { execute.onCopyFileProcessed(srcFile,dstFile);
    }
  }

  public static class DeleteNode implements ExecutionNode {
    File file;

    public DeleteNode(File file)
    { this.file=file;
    }

    public void execute()
    { execute.onDeleteFileProcessed(file);
    }
  }
}
