package com.sapmarkets.technology.util;

import java.io.*;

/**
 * A <code>Base64OutputStream</code> encodes binary data in base 64 and writes
 * it to another output stream.
 *
 * @author       Peter Eberlein
 * @version      1.0
 */

public class Base64OutputStream extends FilterOutputStream {
  /** The line separator as a byte array */
  private static final byte[] lineSeparator =
    System.getProperty("line.separator", "\n").getBytes();

  /** Conversion table from binary to base 64 */
  protected static final char[] encodeTable = {
    'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
    'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
    'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',
    'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
    'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',
    'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
    'w', 'x', 'y', 'z', '0', '1', '2', '3',
    '4', '5', '6', '7', '8', '9', '+', '/'
  };

  /** An internal buffer array that stores the next 24 bits decoded. */
  protected int[] buf = new int[3];
  /** The index one greater than the index of the last valid byte in the
   *  buffer.
   */
  protected int count = 0;
  /** Length of one line in the output to the underlying stream. Defaults to
   *  76 (as specified in RFC 2045).
   */
  protected int lineLength = 76;
  /** Current column in the current line. */
  protected int linePos = 0;

  /**
   * Wraps an <code>OutputStream</code> in a <code>Base64OutputStream</code>
   * that encodes data in base 64 as it is written to the underlying output
   * stream.
   *
   * @param out the underlying output stream.
   */
  public Base64OutputStream(OutputStream out) {
    super(out);
  }

  /**
   * Sets the length of lines written to the underlying output stream. A
   * value greater than 0 causes a line break (using the system default for
   * line separator) in the underlying output stream at the specified length
   * or the next bigger value that can be divided by four (for example
   * 73 and 76 both cause a line break at 76).
   *
   * @param length the number of characters after which a line break is
   * inserted in the underlying output stream. Set to 0 to disable line breaks.
   */
  void setLineLength(int length) {
    lineLength = length;
  }

  /**
   * Writes the next 24 bits encoded to the underlying output stream.
   *
   * @throws IOException if an I/O error occurs.
   */
  public synchronized void write24Bits() throws IOException {
    out.write(encodeTable[(buf[0] & 252) >> 2]);
    out.write(encodeTable[((buf[0] & 3) << 4)
                          + ((count > 1) ? ((buf[1] & 240) >> 4) : 0)]);
    out.write((count > 1)
              ? encodeTable[((buf[1] & 15) << 2)
                            + ((count > 2) ? ((buf[2] & 252) >> 6) : 0)]
              : (int)'=');
    out.write((count > 2) ? encodeTable[buf[2] & 63] : (int)'=');
    count = 0;
    linePos += 4;
  }

  /**
   * Flushes this output stream to the underlying output stream. This
   * operation might pad the output which terminates the base 64 stream.
   * Therefore, it should only be called at the end of a base 64 encoding
   * in order to safely continue operating directly on the underlying output
   * stream.
   *
   * @throws IOException if an I/O error occurs.
   */
  public synchronized void flush() throws IOException {
    if (count > 0) {
      write24Bits();
    }
    if (lineLength > 0 && linePos > 0) {
      out.write(lineSeparator);
      linePos = 0;
    }
    out.flush();
  }

  /**
   * Writes the next byte of data to this output stream.
   *
   * @param b the byte to be written.
   * @throws IOException if an I/O error occurs.
   */
  public synchronized void write(int b) throws IOException {
    buf[count++] = b;
    if (count == 3) {
      write24Bits();
      if (lineLength > 0 && linePos >= lineLength) {
        out.write(lineSeparator);
        linePos = 0;
      }
    }
  }

  /**
   * Writes len bytes from the specified byte array starting at offset off to
   * this output stream.
   *
   * @param b the data.
   * @param off the start offset in the data.
   * @param len the number of bytes to write.
   * @throws IOException if an I/O error occurs.
   */
 public void write(byte[] b, int off, int len) throws IOException {
    for (int i = 0; i < len; i++) {
      write(b[off + i]);
    }
  }

  /**
   * Test method. Encodes data in base 64.
   *
   * @param inFile input file containing data to be encoded
   * @param outFile (optional) output file containing base 64 encoded data. If
   * no output file is specified, the decoded data is written to
   * <code>System.out</code>.
   */
  public static void main(String[] args) {
    if (args.length == 0) {
      System.err.println("Base64OutputStream inFile [outFile]");
    } else {
      try {
        InputStream inStream = new FileInputStream(args[0]);
        InputStreamReader inReader = new InputStreamReader(inStream);

        OutputStream outStream = new Base64OutputStream(((args.length > 1)
          ? new FileOutputStream(args[1])
          : (OutputStream)System.out));
        OutputStreamWriter outWriter =
          new OutputStreamWriter(outStream, "UTF-8");

        int c;
        while ((c = inReader.read()) != -1) {
          outWriter.write(c);
        }

        outWriter.close();
        inReader.close();
      } catch (Exception e) {
        System.err.println(e.toString());
      }
    }
  }
}