package com.sapmarkets.technology.util;

import java.io.*;

/**
 * A <code>Base64InputStream</code> decodes the base 64 encoded binary data
 * filtered from another input stream.
 *
 * @author       Peter Eberlein
 * @version      1.0
 */

public class Base64InputStream extends FilterInputStream {

  /** Conversion table from base 64 to binary */
  protected static final int[] decodeTable = {
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,
    52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1,  0, -1, -1,
    -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,
    -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1
  };

  /** An internal buffer array that stores the next 24 bits decoded. */
  protected int[] buf = new int[3];
  /** The index one greater than the index of the last valid byte in the
   *  buffer. */
  protected int count = 0;
  /** The current position in the buffer. */
  protected int pos = 0;

  /**
   * Wraps an <code>InputStream</code> in a <code>Base64InputStream</code> that
   * decodes base 64 encoded data while reading from it.
   *
   * @param in the underlying input stream.
   */
  public Base64InputStream(InputStream in) {
    super(in);
  }

  /**
   * Returns the next 6 bits decoded from the underlying input stream.
   *
   * @return the next 6 bits as an integer or -1 if the end of the stream was
   * reached.
   */
  private int read6Bits() throws IOException {
    int bits;
    int c;
    do {
      c = in.read();
      if (c < 0 || c == (int)('=')) {
        return -1;
      }
    } while (c > 0x7f || (bits = decodeTable[c]) == -1);
    return bits;
  }

  /**
   * Returns the number of bytes that can be read from this input stream without
   * blocking. There might be more bytes available that could be read without
   * blocking than this method returns but at least the number returned can be
   * read without blocking.
   *
   * @throws IOException if an I/O error occurs.
   */
  public int available() throws IOException {
    return count - pos;
  }

  /**
   * Reads the next byte of data from this input stream. The value byte is
   * returned as an int in the range 0 to 255. If no byte is available because
   * the end of the stream has been reached, the value -1 is returned. This
   * method blocks until input data is available, the end of the stream is
   * detected, or an exception is thrown.
   *
   * @return the next byte of data, or -1 if the end of the stream is reached.
   * @throws IOException if an I/O error occurs.
   */
  public synchronized int read() throws IOException {
    if (pos < count) {
      return buf[pos++];
    } else {
      pos = 0;
      count = 0;
      int a = read6Bits();
      if (a != -1) {
        int b = read6Bits();
        if (b != -1) {
          buf[count++] = (a << 2) | (b >> 4);
          int c = read6Bits();
          if (c != -1) {
            buf[count++] = (b & 15) << 4 | (c >> 2);
            int d = read6Bits();
            if (d != -1) {
              buf[count++] = (c & 3) << 6 | d;
            }
          }
        }
      }
      return (pos < count) ? buf[pos++] : -1;
    }
  }

  /**
   * Reads up to len bytes of data from this input stream into an array of
   * bytes.
   *
   * @param b the buffer into which the data is read.
   * @param off the start offset of the data.
   * @param len the maximum number of bytes read.
   * @return the total number of bytes read into the buffer, or -1 if there is
   * no more data because the end of the stream has been reached.
   * @throws IOException if an I/O error occurs.
   */
  public int read(byte[] b, int off, int len) throws IOException {
    int i = 0;
    try {
      while (i < len) {
        int c = read();
        if (c == -1) {
          if (i == 0) {
            i = -1; // indicate that end-of-stream occured on first read
          }
          break; // return
        } else {
          b[off + i++] = (byte)c;
        }
      }
    } catch (IOException e) {
      if (i == 0) {
        throw e; // rethrow if exception occured on first read
      }
    }
    return i;
  }

  /**
   * Skips over and discards n bytes of data from the input stream. The skip
   * method may, for a variety of reasons, end up skipping over some smaller
   * number of bytes, possibly 0. The actual number of bytes skipped is
   * returned.
   *
   * @param n the number of bytes to be skipped.
   * @throws IOException if an I/O error occurs.
   */
  public long skip(long n) throws IOException {
    for (long i = 0; i < n; i++) {
      if (read() == -1) {
        return i;
      }
    }
    return n;
  }

  /**
   * Returns <code>false</code> to indicate this class does not support the mark
   * and reset method.
   *
   * @return <code>false</code>
   */
  public boolean markSupported() {
    return false;
  }

  /**
   * Throws <code>IOException</code> because this class does not support the
   * mark and reset method.
   *
   * @throws IOException
   */
  public void reset() throws IOException {
    throw new IOException("mark and reset not supported");
  }

  /**
   * Test method. Decodes base 64 encoded date.
   *
   * @param inFile input file containing base 64 encoded data
   * @param outFile (optional) output file containing decoded data. If no
   * output file is specified, the decoded data is written to
   * <code>System.out</code>.
   */
  public static void main(String[] args) {
    if (args.length == 0) {
      System.err.println("Base64InputStream inFile [outFile]");
    } else {
      try {
        Base64InputStream inStream =
          new Base64InputStream(new FileInputStream(args[0]));
        InputStreamReader inReader = new InputStreamReader(inStream, "UTF-8");

        OutputStream outStream =
          (args.length > 1) ? new FileOutputStream(args[1])
                            : (OutputStream)System.out;
        OutputStreamWriter outWriter = new OutputStreamWriter(outStream);

        int c;
        while ((c = inReader.read()) != -1) {
          outWriter.write(c);
        }

        outWriter.close();
        inReader.close();
      } catch (Exception e) {
        System.err.println(e.toString());
      }
    }
  }
}