/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.notificator;

import com.sap.tc.logging.Location;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.Content;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStream;

import java.io.OutputStream;

import java.lang.String;

import javax.activation.DataHandler;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import javax.mail.internet.MimePart;

/**
 * MimeUtils for using mime messages with the notificator. <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author andreas.heix@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/service/notificator/core/com/sapportals/wcm/service/notificator/MimeUtils.java#6
 *      $
 */
public class MimeUtils {

  private static com.sap.tc.logging.Location _log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.service.notificator.MimeUtils.class);

  // --------------
  // Public Methods -----------------------------------------------------------
  // --------------

  public static void copyContentToOutputstream(IContent content, OutputStream out)
    throws WcmException {
    try {
      byte[] buffer = new byte[2 * 1024];
      int read = 0;
      InputStream in = content.getInputStream();
      while ((read = in.read(buffer)) != -1) {
        out.write(buffer, 0, read);
      }
      out.close();
      in.close();
    }
    catch (IOException e) {
      throw new WcmException(e);
    }
  }


  /**
   * convert a <code>String</code> to a content.
   *
   * @param text TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @return <code>IContent</code> is the content object for the converted
   *      String.
   * @exception WcmException Exception raised in failure situation
   */
  public static IContent getContentFromString(String text,
    String type)
    throws WcmException {
    //

    if (text == null) {
      return null;
    }

    if (type == null) {
      type = "text/plain";
    }

    IContent content = null;
    String step = "";

    try {
      step = "creating output stream";
      ByteArrayOutputStream out = new ByteArrayOutputStream(text.length());
      step = "writing text to output stream";
      out.write(text.getBytes(), 0, text.length());
      step = "creating input stream";
      ByteArrayInputStream in = new ByteArrayInputStream(out.toByteArray());
      step = "creating content from input stream";
      content = new Content(in, type, in.available());
    }
    catch (Exception e) {
      String message = "exception while " + step;
      _log.warningT("getContentFromString(114)", message + ", " + e.getMessage());
      throw new WcmException(message, e);
    }

    return content;
  }


  /**
   * convert a <code>IContent</code> to a string.
   *
   * @param content TBD: Description of the incoming method parameter
   * @return <code>String</code> is the content converted to a String.
   * @exception WcmException Exception raised in failure situation
   */
  public static String getContentAsString(IContent content)
    throws WcmException {
    //

    if (content == null) {
      return null;
    }

    String text = null;
    String step = "";

    try {
      step = "creating stream";
      ByteArrayOutputStream stream = new ByteArrayOutputStream();
      step = "copying content to stream";
      copyContentToOutputstream(content, stream);
      step = "getting string from stream";
      text = stream.toString();
    }
    catch (Exception e) {
      String message = "exception while " + step;
      _log.warningT("getContentAsString(150)", message + ", " + e.getMessage());
      throw new WcmException(message, e);
    }

    return text;
  }


  /**
   * set the content of a <code>MimeMessage</code> to the given body content.
   *
   * @param mimepart TBD: Description of the incoming method parameter
   * @return <code>MimeMessage</code> the mime message.
   * @throws WcmException if the message could not be generated.
   */
  public static IContent getContent(MimePart mimepart)
    throws WcmException {
    //

    if (mimepart == null) {
      return null;
    }

    Content content = null;
    String step = "";

    try {
      step = "creating data handler";
      DataHandler handler = new DataHandler(mimepart.getContent(), mimepart.getContentType());
      step = "getting input stream";
      InputStream stream = handler.getInputStream();
      step = "creating content";
      content = new Content(stream, handler.getContentType(), stream.available());
    }
    catch (Exception e) {
      String message = "exception while " + step;
      _log.warningT("getContent(187)", message + ", " + e.getMessage());
      throw new WcmException(message, e);
    }

    return content;
  }


  /**
   * set the content of a <code>MimeMessage</code> to the given body content.
   *
   * @param mimemessage content to be set
   * @param body content to be set
   * @return <code>MimeMessage</code> the mime message.
   * @throws WcmException if the message could not be generated.
   */
  public static MimeMessage setContent(MimeMessage mimemessage,
    IContent body)
    throws WcmException {
    //

    return setContent(mimemessage, body, null);
  }


  /**
   * set the content of a <code>MimeMessage</code> to a given body content with
   * several attachments.
   *
   * @param mimemessage content to be set
   * @param body content to be set
   * @param attachments content to be set
   * @return <code>MimeMessage</code> the mime message.
   * @throws WcmException if the message could not be generated.
   */
  public static MimeMessage setContent(MimeMessage mimemessage,
    IContent body,
    IResource attachments[])
    throws WcmException {
    //

    MimeMultipart multipartMessage = null;
    String step = "";

    try {

      step = "creating multipart";
      multipartMessage = new MimeMultipart();

      step = "creating body part";
      MimeBodyPart msgbody = new MimeBodyPart(body.getInputStream());
      step = "setting body part";
      msgbody.setDisposition(MimeBodyPart.INLINE);
      step = "adding body part";
      multipartMessage.addBodyPart(msgbody);

      for (int i = 0; i < attachments.length; i++) {
        if (attachments[i] != null) {
          step = "getting name for attachment " + i;
          String name = attachments[i].getDisplayName();
          step = "getting content for " + name;
          IContent content = attachments[i].getContent();
          step = "creating attachment part for " + name;
          MimeBodyPart msgpart = new MimeBodyPart(content.getInputStream());
          step = "setting attachment part for " + name;
          msgpart.setDisposition(MimeBodyPart.ATTACHMENT);
          step = "setting filename attachment part for " + name;
          msgpart.setFileName(attachments[i].getDisplayName());
          step = "adding attachment part for " + name;
          multipartMessage.addBodyPart(msgpart);
        }
        else {
          _log.infoT("setContent(261)", "missing attachment #" + i);
        }
      }

      step = "setting mime message content";
      mimemessage.setContent(multipartMessage);
      step = "saving changes";
      mimemessage.saveChanges();

    }
    catch (Exception e) {
      String emsg = "caught exception while " + step + ": " + e.getMessage();
      _log.warningT("setContent(273)", emsg + ", " + e.getMessage());
      throw new WcmException(emsg, e);
    }

    return mimemessage;
  }

}

// EOF
