/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.notificator;

import com.sap.tc.logging.Location;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.IResourceFactory;
import com.sapportals.wcm.repository.ResourceFactory;
import com.sapportals.wcm.service.IServiceTypesConst;
import com.sapportals.wcm.service.pipeline.IPipelineService;
import com.sapportals.wcm.service.pipeline.IXsltPipelineService;
import com.sapportals.wcm.service.pipeline.producer.IProducer;
import com.sapportals.wcm.service.pipeline.producer.ProducerFromResource;
import com.sapportals.wcm.util.channels.IChannel;
import com.sapportals.wcm.util.channels.IRecipient;
import com.sapportals.wcm.util.content.ContentException;
import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.resource.ResourceBundles;
import com.sapportals.wcm.util.uri.RID;
import java.io.InputStream;
import java.util.Iterator;
import java.util.LinkedList;

import java.util.Locale;
import java.util.Properties;
import java.util.ResourceBundle;

/**
 * Base class for Notifications used to notify a recipient. <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author andreas.heix@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/service/notificator/core/com/sapportals/wcm/service/notificator/AbstractNotification.java#9
 *      $
 */
public abstract class AbstractNotification
   implements INotification {

  // ---------
  // Constants ----------------------------------------------------------------
  // ---------

  /**
   * configuration tag: notificator's template directory.
   */
  private final static String CONFIG_TEMPLATES_TAG = "templates";

  /**
   * configuration default: the default notificator's template directory.
   */
  private final static String CONFIG_TEMPLATES_DEFAULT = "etc/notifications";

  /**
   * configuration tag: the format (e.g. "text/plain") to use per channel.
   */
  private final static String CONFIG_FORMAT_TAG = "format";

  /**
   * XML file suffix.
   */
  protected final static String XMLSUFFIX = ".xml";

  /**
   * XSL file suffix.
   */
  protected final static String XSLSUFFIX = ".xsl";

  // ----------------
  // Static Variables ---------------------------------------------------------
  // ----------------

  /**
   * log.
   */
  private static com.sap.tc.logging.Location g_Log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.service.notificator.AbstractNotification.class);

  /**
   * resource bundles.
   */
  private static ResourceBundles m_Res = ResourceBundles.getBundle("com.sapportals.wcm.service.notificator.wcm.Res");

  // ------------------
  // Instance Variables -------------------------------------------------------
  // ------------------

  /**
   * unique system ID for this notification event
   */
  protected String m_ID = null;

  /**
   * the context to use for the producers.
   */
  protected IResourceContext m_Context = null;


  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------

  // ------------------------------------------------------------------------
  private Locale getDefaultLocale() {

    Locale defaultLocale;

    if (m_Context != null) {
      defaultLocale = m_Context.getLocale();
    }
    else {
      defaultLocale = Locale.getDefault();
    }

    return defaultLocale;
  }


  // ------------------------------------------------------------------------
  private LinkedList buildXmlLocaleURIs(LinkedList rids,
    String prefix,
    Locale locale,
    String suffix) {

    if (locale == null) {
      return rids;
    }

    if (locale.getVariant().length() > 0) {
      rids.addLast(RID.getRID(prefix + "_" + locale.getLanguage() + "_" + locale.getCountry() + "_" + locale.getVariant() + suffix));
    }
    if (locale.getCountry().length() > 0) {
      rids.addLast(RID.getRID(prefix + "_" + locale.getLanguage() + "_" + locale.getCountry() + suffix));
    }
    if (locale.getLanguage().length() > 0) {
      rids.addLast(RID.getRID(prefix + "_" + locale.getLanguage() + suffix));
    }

    return rids;
  }


  // ------------------------------------------------------------------------
  private IContent createMessage(String templates,
    String format,
    IRecipient recipient,
    IChannel channel,
    Properties properties)
    throws WcmException {

    IResourceFactory factory = ResourceFactory.getInstance();

    IContent content = null;
    String step = "";
    try {
      step = "getting properties";
      properties = getProperties(recipient, channel, properties);
      step = "getting xml uri";
      RID xmlURI = getXmlRID(factory, templates, recipient);
      step = "getting xml producer";
      IProducer xmlProducer = getXmlProducer(factory, xmlURI);
      step = "getting xsl uri";
      RID xslURI = getXslRID(factory, templates, channel);
      step = "getting xsl producer";
      IProducer xslProducer = getXslProducer(factory, xslURI);
      step = "creating pipeline";
      IXsltPipelineService pipeline = (IXsltPipelineService)factory.getServiceFactory().getService(IServiceTypesConst.XSLT_PIPELINE_SERVICE);

      // MR deprecated method
      /*
       * if( format != null ) {
       * step = "setting format";
       * pipeline.setFormat(format);
       * }
       */
      step = "handling pipeline";
      if (format != null) {
        content = pipeline.handle(xmlProducer, xslProducer, properties, format);
      }
      else {
        content = pipeline.handle(xmlProducer, xslProducer, properties);
      }
    }
    catch (Exception e) {
      g_Log.debugT("createMessage(201)", "Notificaton: error creating message while " + step + ": " + e.getMessage());
      throw new WcmException("Notificaton: error creating message while " + step, e);
    }

    return content;
  }


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  // ------------------------------------------------------------------------
  /**
   * Create a notification for a given system ID.
   *
   * @param id a <code>String</code> with the system notification type.
   * @param context a <code>IResourceContext</code> with the context to use for
   *      the notification.
   */
  public AbstractNotification(String id,
    IResourceContext context) {

    m_ID = id;

    if (m_ID == null) {
      m_ID = "";
    }

    m_Context = context;

  }


  // -----------------
  // Protected Methods --------------------------------------------------------
  // -----------------

  // ------------------------------------------------------------------------
  /**
   * Get the uri to use for the XML by trying several locale-specific uris.
   *
   * @param factory the <code>IResourceFactory</code> to use for getting the
   *      resource.
   * @param templates a <code>String</code> with the name of the notificator's
   *      template directory.
   * @param recipient the <code>IRecipient</code> to get the XML for.
   * @return xmlRID
   * @exception WcmException Exception raised in failure situation
   */
  protected RID getXmlRID(IResourceFactory factory,
    String templates,
    IRecipient recipient)
    throws WcmException {

    Locale locale = null;
    if (recipient != null) {
      locale = recipient.getLocale();
    }
    Locale defaultLocale = getDefaultLocale();

    LinkedList rids = new LinkedList();

    rids = buildXmlLocaleURIs(rids, templates + m_ID, locale, XMLSUFFIX);
    rids = buildXmlLocaleURIs(rids, templates + m_ID, defaultLocale, XMLSUFFIX);

    Iterator ridIterator = rids.iterator();
    while (ridIterator.hasNext()) {
      RID triedRID = (RID)ridIterator.next();
      if (factory.checkExistence(triedRID, m_Context)) {
        return triedRID;
      }
    }

    return RID.getRID(templates + m_ID + XMLSUFFIX);
  }


  // ------------------------------------------------------------------------
  /**
   * Get the uri to use for the XSL by trying a channel-specific uri first.
   *
   * @param factory the <code>IResourceFactory</code> to use for getting the
   *      resource.
   * @param templates a <code>String</code> with the name of the notificator's
   *      template directory.
   * @param channel the <code>IChannel</code> to get the XSL for.
   * @return xslRID
   * @exception WcmException Exception raised in failure situation
   */
  protected RID getXslRID(IResourceFactory factory,
    String templates,
    IChannel channel)
    throws WcmException {

    RID triedRID = RID.getRID(templates + m_ID + "_" + channel.getID() + XSLSUFFIX);

    if (factory.checkExistence(triedRID, m_Context)) {
      return triedRID;
    }

    return RID.getRID(templates + m_ID + XSLSUFFIX);
  }


  // ------------------------------------------------------------------------
  /**
   * Get the XML producer for feeding the pipeline.
   *
   * @param factory the <code>IResourceFactory</code> to get resources from.
   * @param xmlRID TBD: Description of the incoming method parameter
   * @return an <code>IProducer</code> for the XML.
   * @exception WcmException Exception raised in failure situation
   */
  protected IProducer getXmlProducer(IResourceFactory factory,
    RID xmlRID)
    throws WcmException {

    IResource xmlResource = factory.getResource(xmlRID, m_Context);

    return new ProducerFromResource(xmlResource);
  }


  // ------------------------------------------------------------------------
  /**
   * Get the XSL producer for feeding the pipeline.
   *
   * @param factory the <code>ResourceFactory</code> to get resources from.
   * @param xslRID TBD: Description of the incoming method parameter
   * @return an <code>IProducer</code> for the XSL.
   * @exception WcmException Exception raised in failure situation
   */
  protected IProducer getXslProducer(IResourceFactory factory,
    RID xslRID)
    throws WcmException {

    IResource xslResource = factory.getResource(xslRID, m_Context);

    return new ProducerFromResource(xslResource);
  }


  // ------------------------------------------------------------------------
  /**
   * Get the properties to pass to the pipeline.
   *
   * @param recipient the <code>IRecipient</code> which will get the message.
   * @param channel the <code>IChannel</code> which will be used for sending the
   *      message.
   * @param properties the <code>Properties</code> from the notificator config.
   * @return the <code>Properties</code> to pass to the pipeline for adding to
   *      the XML-/XSL-output.
   */
  protected Properties getProperties(IRecipient recipient,
    IChannel channel,
    Properties properties) {

    return properties;
  }


  // -------------
  // INotification ---------------------------------------------------------------
  // -------------

  // ------------------------------------------------------------------------
  public String getID() {

    return m_ID;
  }


  // ------------------------------------------------------------------------
  public String getDescription() {

    Locale defaultLocale = getDefaultLocale();

    return getDescription(defaultLocale);
  }


  // ------------------------------------------------------------------------
  public String getDescription(Locale locale) {

    try {
      return m_Res.getString("Notification." + m_ID, locale);
    }
    catch (Exception e) {
      return m_ID;
    }

  }

  // ------------------------------------------------------------------------
  public String getDescription(Locale locale, Object obj1) {

    try {
      return m_Res.getString("Notification." + m_ID, locale, obj1);
    }
    catch (Exception e) {
      return m_ID;
    }

  }


  // ------------------------------------------------------------------------
  public String getFromAddress(IChannel channel) {

    return null;// use notificator's default from address.

  }


  // ------------------------------------------------------------------------
  public IContent getContent(IRecipient recipient,
    IChannel channel,
    Properties properties)
    throws WcmException {

    // get the XML-/XSL-template directory from notificator-config.
    String templates = properties.getProperty(CONFIG_TEMPLATES_TAG, CONFIG_TEMPLATES_DEFAULT);
    if (!(templates.equalsIgnoreCase("")
       || templates.endsWith("/")
      )) {
      templates += "/";
    }

    // get some variables (locale for XML, channelid for XML and format)
    String localeid = recipient.getLocale().toString();
    String channelid = channel.getID();
    String format = properties.getProperty(channelid + "." + CONFIG_FORMAT_TAG);

    if (g_Log.beDebug()) {
      g_Log.debugT("getContent(436)", "Notificaton: templates=" + templates + " format=" + format
         + " locale=" + localeid + " channel= " + channelid);
    }

    // now, build the message
    return createMessage(templates, format, recipient, channel, properties);
  }

}
