package com.sap.tc.webdynpro.tests.utils;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDAbstractDropDownByIndex;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDAbstractDropDownByKey;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDAbstractInputField;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDCaption;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDCheckBox;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDLink;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDProgressIndicator;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDRadioButton;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDTable;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDTableCellEditor;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDTableColumn;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDTextEdit;
import com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDTextView;
import com.sap.tc.webdynpro.progmodel.api.IWDAction;
import com.sap.tc.webdynpro.progmodel.api.IWDCustomEvent;
import com.sap.tc.webdynpro.progmodel.api.IWDNode;
import com.sap.tc.webdynpro.progmodel.api.IWDNodeElement;
import com.sap.tc.webdynpro.progmodel.api.IWDViewElement;

/**
 * Helper class that makes a Web Dynpro table UI element sortable (column-wise).
 */
public final class TableSorter
{
  /**
   * Creates a table sorter for the given table using the given sort action.
   * This constructor must be called from <code>wdDoModifyView()</code>, but
   * usually only when that hook is called for the first time. Store the newly
   * created instance in a context attribute with Java native type
   * <code>com.sap.tc.webdynpro.tests.utils.TableSorter</code>.
   * The given sort action's event handler will be bound to the <code>onAction</code>
   * event of sortable columns and must at least call this table sorter's
   * <code>sort(wdEvent)</code> method.
   * 
   * Every column of the table is made sortable if possible according to the
   * following rules.
   * If a comparator is given for a column's index and it is a
   * <code>NodeElementByAttributeComparator</code>, then that comparator defines
   * both the attribute and the ordering used to sort that column.
   * If any other comparator is given and an attribute can be determined from
   * that column's table cell editor, then that attribute is used to sort that
   * column according to the ordering imposed by the given comparator.
   * If no comparator is given but an attribute can be determined from
   * that column's table cell editor, then that attribute is used to sort that
   * column according to the natural ordering of that attribute's type.
   * Else that column is left untouched.
   * 
   * When a column is made sortable its <code>onAction</code> event (incl.
   * parameter mapping) is changed. Furthermore if a header is present for that
   * column then the header's <code>imageSource</code> property is also changed
   * to indicate the currently sorted column and its sort direction.
   * 
   * @see sort()
   * @see NodeElementByAttributeComparator
   * @see com.sap.tc.webdynpro.clientserver.uielib.standard.api.IWDTable
   */
  public TableSorter(IWDTable table, IWDAction sortAction, Comparator[] comparators)
  {
    this.table = table;

    // sanity checks
    if (sortAction == null)
      throw new IllegalArgumentException("Sort action must be given");
    if (table == null)
      throw new IllegalArgumentException("Table must be given");
    if (table.bindingOfDataSource() == null)
      throw new IllegalArgumentException("Data source of table with id '" + table.getId() + "' must be bound");

    String dataSourcePrefix = table.bindingOfDataSource() + ".";
    int index = 0;
    for (Iterator it = table.iterateColumns(); it.hasNext(); ++index)
    { // for every column: try to make it bindable
      IWDTableColumn column = (IWDTableColumn) it.next();
      Comparator comparator = null;
      if ( comparators != null
        && index < comparators.length )
        comparator = comparators[index];
      ReversableComparator reversable = null;

      if (comparator instanceof NodeElementByAttributeComparator)
      { // the easy one, attribute and ordering are given
        reversable = new ReversableComparator(comparator);
      }
      else
      { // attribute must be determined
        String bindingOfPrimaryProperty = bindingOfPrimaryProperty(column.getTableCellEditor());
        if ( bindingOfPrimaryProperty == null
          || !bindingOfPrimaryProperty.startsWith(dataSourcePrefix) )
          continue; // no attribute found or outside of data source
        String attributeName = bindingOfPrimaryProperty.substring(dataSourcePrefix.length());
        if (attributeName.indexOf('.') >= 0)
          continue; // attribute not immediately below data source
        
        reversable = new ReversableComparator(
          new NodeElementByAttributeComparator(attributeName, comparator));
      }

      // set up internal data structures
      comparatorForColumn.put(column, reversable);

      // prepare table column
      column.setOnAction(sortAction);
      column.mappingOfOnAction().addSourceMapping("col", "col");
      if (column.getHeader() != null)
        column.getHeader().setImageSource(null);
    }
  }
  
  /**
   * This method must be called from the event handler of this table sorter's
   * sort action. It performs the actual sort operation.
   */
  public void sort(IWDCustomEvent wdEvent, IWDNode dataSource)
  {
    // find the things we need
    String columnId = wdEvent.getString("col");
    IWDTableColumn column = (IWDTableColumn) table.getView().getElement(columnId);
    ReversableComparator reversable = (ReversableComparator) comparatorForColumn.get(column);
    if (reversable == null)
      return; // not a sortable column

    // remove icon of previously sorted column
    if ( currentlySortedColumn != null
      && currentlySortedColumn.getHeader() != null )
      currentlySortedColumn.getHeader().setImageSource(null);

    // bookkeeping
    if (column == currentlySortedColumn)
      reversable.toggleReversed();
    currentlySortedColumn = column;

    // set icon in currently sorted column
    if ( currentlySortedColumn.getHeader() != null )
      currentlySortedColumn.getHeader().setImageSource(reversable.isReversed()
        ? "~sapicons/s_b_srtd.GIF"
        : "~sapicons/s_b_srtu.GIF");

    // sorting
    dataSource.sortElements(reversable);
  }

  /**
   * Returns the binding of the given table cell editor's property that is
   * considered "primary" or <code>null</code> if no such binding exists or no
   * such property can be determined.
   */
  private static final String bindingOfPrimaryProperty(IWDTableCellEditor editor)
  {
    return editor instanceof IWDViewElement
      ? bindingOfPrimaryProperty((IWDViewElement) editor)
      : null;
  }

  /**
   * Returns the binding of the given view element's property that is
   * considered "primary" or <code>null</code> if no such binding exists or no
   * such property can be determined.
   */
  private static final String bindingOfPrimaryProperty(IWDViewElement element)
  {
    if (element instanceof IWDAbstractDropDownByIndex)
      return ((IWDAbstractDropDownByIndex) element).bindingOfTexts();
    if (element instanceof IWDAbstractDropDownByKey)
      return ((IWDAbstractDropDownByKey) element).bindingOfSelectedKey();
    if (element instanceof IWDAbstractInputField)
      return ((IWDAbstractInputField) element).bindingOfValue();
    if (element instanceof IWDCaption)
      return ((IWDCaption) element).bindingOfText();
    if (element instanceof IWDCheckBox)
      return ((IWDCheckBox) element).bindingOfChecked();
    if (element instanceof IWDLink)
      return ((IWDLink) element).bindingOfText();
    if (element instanceof IWDProgressIndicator)
      return ((IWDProgressIndicator) element).bindingOfPercentValue();
    if (element instanceof IWDRadioButton)
      return ((IWDRadioButton) element).bindingOfSelectedKey();
    if (element instanceof IWDTextEdit)
      return ((IWDTextEdit) element).bindingOfValue();
    if (element instanceof IWDTextView)
      return ((IWDTextView) element).bindingOfText();

    return null;
  }

  /**
   * Instance of a comparator according to the ordering imposed by the
   * implementation of <code>Comparable</code>.
   */
  private static final Comparator DEFAULT = new Comparator()
  {
    /**
     * Compares the given objects according to the ordering imposed by the first
     * ones <code>compareTo(Object)</code> function. Furthermore, <code>null</code>
     * is treated to be less than any object.
     * 
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     */
    public int compare(Object o1, Object o2)
    {
      if (o1 == null && o2 == null)
        return 0;
      if (o1 == null)
        return -1;
      if (o2 == null)
        return +1;
  
      return ((Comparable) o1).compareTo((Comparable) o2);
    }
  };
    
  /**
   * Map of table column to comparator (<code>ReversableComparator</code>)
   * used for sorting that column (sortable columns only).
   */
  private final Map comparatorForColumn = new HashMap();

  /**
   * Column that is currently sorted.
   */
  private IWDTableColumn currentlySortedColumn;

  /**
   * The table to be sorted.
   */
  private final IWDTable table;

  /**
   * Generic comparator that compares node elements by a given attribute with
   * the help of a given comparator.
   */
  public final class NodeElementByAttributeComparator implements Comparator
  {
    /**
     * Creates a new comparator for the given attribute name that compares values
     * of that attribute according to the natural ordering of that attribute's
     * type (which must implement <code>java.lang.Comparable</code>).
     */
    public NodeElementByAttributeComparator(String attributeName)
    {
      this(attributeName, null, false);
    }
  
    /**
     * Creates a new comparator for the given attribute name that compares values
     * of that attribute with the help of the given comparator. If no comparator
     * is given, the natural ordering of that attribute's type is used.
     */
    public NodeElementByAttributeComparator(String attributeName, Comparator comparator)
    {
      this(attributeName, comparator, false);
    }
  
    /**
     * Creates a new comparator for the given attribute name that compares values
     * of that attribute either as objects (i.e. "in internal format") or as text
     * (i.e. "in external format") as indicated. The ordering is the natural
     * ordering of that attribute's type (which must implement
     * <code>java.lang.Comparable</code>) in case objects are compared or the
     * natural ordering of <code>java.lang.String</code> in case texts are compared.
     */
    public NodeElementByAttributeComparator(String attributeName, boolean compareAsText)
    {
      this(attributeName, null, compareAsText);
    }
  
    /**
     * Internal constructor.
     */
    private NodeElementByAttributeComparator(String attributeName,
      Comparator comparator, boolean compareAsText)
    {
      if (attributeName == null)
        throw new IllegalArgumentException("Attribute name must not be null");
      if (comparator == null)
        comparator = DEFAULT;
  
      this.attributeName = attributeName;
      this.comparator = comparator;
      this.compareAsText = compareAsText;
    }
  
    /**
     * Compares the given objects which must be instances of <code>IWDNodeElement</code>
     * according to the values of the attribute given at construction time
     * with the help of the comparator given at construction time.
     * 
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     * @see com.sap.tc.webdynpro.progmodel.api.IWDNodeElement
     */
    public int compare(Object o1, Object o2)
    {
      IWDNodeElement element1 = (IWDNodeElement) o1;
      IWDNodeElement element2 = (IWDNodeElement) o2;
      Object attributeValue1 = compareAsText
        ? element1.getAttributeAsText(attributeName)
        : element1.getAttributeValue(attributeName);
      Object attributeValue2 = compareAsText
        ? element2.getAttributeAsText(attributeName)
        : element2.getAttributeValue(attributeName);
      
      return comparator.compare(attributeValue1, attributeValue2);
    }
  
    /**
     * Name of the attribute used for comparisons.
     */
    private final String attributeName;
    
    /**
     * Comparator used for comparing the attribute's values.
     */
    private final Comparator comparator;
    
    /**
     * Indicates whether attribute values are compared as text (as opposed to
     * "as objects").
     */
    private final boolean compareAsText;
  }

  /**
   * Comparator that knows how to reverse another comparator's sort direction.
   */
  private final class ReversableComparator implements Comparator
  {
    /**
     * Creates a new reversable comparator for the given other comparator.
     */
    public ReversableComparator(Comparator comparator)
    {
      if (comparator == null)
        throw new IllegalArgumentException("Comparator must not be null");
  
      this.comparator = comparator;
    }
  
    /**
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     */
    public int compare(Object o1, Object o2)
    {
      return reversed
        ? -comparator.compare(o1, o2)
        :  comparator.compare(o1, o2);
    }
  
    /**
     * Tells whether the other comparator's sort direction is reversed.
     */
    public boolean isReversed()
    {
      return reversed;
    }
  
    /**
     * Determines whether the other comparator's sort direction is reversed.
     */
    public void setReversed(boolean reversed)
    {
      this.reversed = reversed;
    }
    
    /**
     * Toggles the current "sort direction".
     */
    public void toggleReversed()
    {
      reversed = !reversed;
    }
  
    /**
     * The other comparator.
     */
    private final Comparator comparator;
  
    /**
     * Whether the other comparator's sort direction is reversed.
     */
    private boolean reversed;
  }
}
