/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/url/api/WDWebResource.java#2 $
 */

package com.sap.tc.webdynpro.services.sal.url.api;

import java.io.InputStream;

import com.sap.tc.webdynpro.services.sal.core.AbstractManager;
import com.sap.tc.webdynpro.services.sal.deployment.api.WDDeployableObjectPart;
import com.sap.tc.webdynpro.services.sal.util.cache.WDResourceNotFoundException;
import com.sap.tc.webdynpro.services.session.api.WDScopeType;

/**
 * The class <code>WDWebResource</code> is used to access instances 
 * of type <code>IWDWebResource</code>. 
 * <p>
 * A web resource belongs to a deployable object part. With an <code>IWDWebResource</code>, 
 * it is possible to generate the URL to this object. 
 * 
 * @see com.sap.tc.webdynpro.services.sal.url.api.IWDWebResource  
 * @see com.sap.tc.webdynpro.services.sal.url.api.WDWebResourceType
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public abstract class WDWebResource { 

  /**
   * Return is the <code>IWDWebResource</code> object that represents a 
   * Web Dynpro application, i.e. the URL of this Web resource points to the 
   * associated application. 
   * 
   * @param deployableObjectPart a deployable object part
   * @return the web resource representation
   */ 
  public static IWDWebResource getWebResource(WDDeployableObjectPart deployableObjectPart) {
    return AbstractManager.getServerPlatformImplementation().getWebResourceFactory().getWebResource(deployableObjectPart);
  }

  /**
   * Return is the <code>IWDWebResource</code> object that represents the 
   * specified web resource. The resource is specified by its type and name. 
   * 
   * @param deployableObjectPart a deployable object part
   * @param resourceType the type of the resource as defined by <code>WDWebResourceType</code>
   * @param resourceName the name of the web resource 
   * @return the web resource representation
   */
  public static IWDWebResource getWebResource(
    WDDeployableObjectPart deployableObjectPart,
    WDWebResourceType resourceType,
    String resourceName) {
    return AbstractManager.getServerPlatformImplementation().getWebResourceFactory().getWebResource(
      deployableObjectPart, resourceType, resourceName);
  }

  /**
   * Returns an <code>IWDWebResource</code> for a given web resource. If the 
   * supplied input parameter <code>makePersistent</code> is set to true, 
   * the web resource is made persistent. 
   *
   * @param webResource the web resource as byte array
   * @param resourceType  the MIME type
   * @return the web resource if existing; otherwise <code>null</code> is returned
   */
  public static IWDCachedWebResource getWebResource(byte[] webResource, WDWebResourceType resourceType) {
    return AbstractManager.getServerPlatformImplementation().getWebResourceFactory().getWebResource(
      webResource,
      resourceType);
  }

  /**
   * Returns an <code>IWDWebResource</code> for a given web resource.
   *
   * @param webResource the web resource as byte array
   * @param resourceType  the MIME type
   * @return the web resource if existing; otherwise <code>null</code> is returned
   */
  public static IWDCachedWebResource getWebResource(InputStream input, WDWebResourceType resourceType) {
    return AbstractManager.getServerPlatformImplementation().getWebResourceFactory().getWebResource(
      input,
      resourceType);
  }
  
  /**
   * returns a <code>IWDCachedWebResource</code>, which is saved directly to the <code>WDDeployableObject</code> directory.
   * Its lifetime is defined over <code>WDScopeType</code>. It is cached and saved with the given key. The developer has to be sure,
   * that it is identifiable over the defined scope
   * This resource gets an public web resource: it is available on the server as public web resource, like an image
   * 
   * @param webResource - the stream/bytearray
   * @param resourceType - the content type of the bytearray
   * @param scopeType - the lifetime scopetype
   * @param deployableObjectPart - the deployableobject, in which directory the cached source is saved
   * @param key - the name/key of the resource
   * @return the wrapper for the resource
   */  
  public static IWDCachedWebResource getPublicCachedWebResource(
    byte[] webResource,
    WDWebResourceType resourceType,
    WDScopeType scopeType,
    WDDeployableObjectPart deployableObjectpart,
    String key) {
    return AbstractManager.getServerPlatformImplementation().getWebResourceFactory().getPublicCachedWebResource(webResource, resourceType, scopeType, deployableObjectpart, key);
  }

  /**
   * returns a <code>IWDCachedWebResource</code>, which is saved directly to the <code>WDDeployableObject</code> directory.
   * Its lifetime is defined over <code>WDScopeType</code>. It is cached and saved with the given key. The developer has to be sure,
   * that it is identifiable over the defined scope
   * This resource gets an public web resource: it is available on the server as public web resource, like an image
   * 
   * @param webResource - the stream/bytearray
   * @param resourceType - the content type of the bytearray
   * @param scopeType - the lifetime scopetype
   * @param deployableObjectPart - the deployableobject, in which directory the cached source is saved
   * @param key - the name/key of the resource
   * @return the wrapper for the resource
   */
  public static IWDCachedWebResource getPublicCachedWebResource(
    InputStream webResource,
    WDWebResourceType resourceType,
    WDScopeType scopeType,
    WDDeployableObjectPart deployableObjectpart, 
    String key) {
    return AbstractManager.getServerPlatformImplementation().getWebResourceFactory().getPublicCachedWebResource(webResource, resourceType, scopeType, deployableObjectpart, key);
  }
  
  /**
   * Returns the cached web resource with a supplied key as identification and the scopetype for defining
   * the scope in which the resource is searched. 
   * 
   * @param key, the key to find the cached web resource
   * @param scopetype, defines the scope in which the resource is searched
   * @return the cached <code>IWDCachedWebResource</code> if existing; otherwise <code>null</code> is returned
   */
  public static IWDCachedWebResource getCachedWebResource(Object key, WDScopeType scopetype) throws WDResourceNotFoundException{
    return AbstractManager.getServerPlatformImplementation().getWebResourceFactory().getCachedWebResource(key, scopetype);
  }
}
