/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/url/api/WDURLGenerator.java#2 $
 */

package com.sap.tc.webdynpro.services.sal.url.api;

import java.util.Map;

import com.sap.tc.webdynpro.services.sal.core.AbstractManager;
import com.sap.tc.webdynpro.services.sal.deployment.api.WDAliasResolvingException;
import com.sap.tc.webdynpro.services.sal.deployment.api.WDDeployableObjectPart;

/**
 * <code>WDURLGenerator</code> provides static methods which generated URLs for 
 * any <code>IWDWebResource</code> objects that are referenced by Web Dynpro applications. 
 * <br>
 * The interface provides methods which are responsible for the following:
 * <ul>
 * <li> Get the real public resource path, independent of a deployable object and deployable object part
 * <li> Get the real public resource path on the level of a deployable object
 * <li> Get the real public resource path on the level of a deployable object part
 * <li> Get the URL of a specified web resource that is defined on the level of a deployable object
 * <li> Get the URL of a specified web resource that is defined on the level of a deployable object part
 * </ul>
 *
 * @see com.sap.tc.webdynpro.services.sal.url.api.IWDURLGenerator
 *
 * @author SAP
 * @SAPWebDynproPart 2
 */
public abstract class WDURLGenerator { 

  /**
   * Returns the file system path to the resource specified by <code>resourceName</code>
   * that is expected to be in the "public" directory of the deployable object 
   * (=development component) of the currently running Web Dynpro application. The "public" 
   * directory contains resources which are not accessible by others via URL. 
   * <p>
   * Using this method, it is possible to access resources which are imported in 
   * the SAP NetWeaver Developer Studio under the "/src/configuration" folder 
   * of a Web Dynpro project and then deployed together with this project.   
   * <p>
   * On SAP J2EE Engine 6.40, this is ".\temp\webdynpro\public\<name of deployable 
   * object of current application>\resourceName". Note that the path is relative to 
   * the engine's startup directory.     
   * 
   * @return the file system path to the specified resource under the 
   * root directory of the current deployable object
   */  
  public static String getPublicResourcePath(String resourceName) {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getPublicResourcePath(resourceName);     
  }
  
  /**
   * Returns the file system path to the resource specified by <code>resourceName</code>
   * that is expected to be in the "public" directory of the deployable object part specified 
   * b< <code>part</code>. The "public" directory contains resources which are not accessible 
   * by others via URL. The specified <code>part</code> must belong to a referenced deployable 
   * object (=development component) of the currently executed Web Dynpro application, 
   * otherwise an <code>IllegalArgumentException will be thrown</code>.
   * <p>
   * Using this method, it is possible to access resources which are imported in 
   * the SAP NetWeaver Developer Studio under the "/src/configuration/<deployable object 
   * part type>/<deployable object part>" folder of a Web Dynpro project and then deployed 
   * together with this project.   
   * <p>
   * On SAP J2EE Engine 6.40, this is ".\temp\webdynpro\public\<name of a referenced deployable 
   * object of current application>\<deployable object part type>\<deployable object part>\resourceName". 
   * Note that the path is relative to the engine's startup directory.     
   * 
   * @return the file system path to the specified resource under the 
   * root directory of the current deployable object
   * 
   * @throws IllegalArgumentException if the specified <code>part</code> does not belong 
   * to a referenced deployable object of the currently running application    
   */  
  public static String getPublicResourcePath(WDDeployableObjectPart part, String resourceName) {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getPublicResourcePath(part, resourceName);     
  }

  /**
   * Returns the real public resource path, which is independent from any
   * deployable object or deployable object part. 
   * <br>
   * On SAP J2EE Engine 6.40, this is ".\temp\webdynpro\web" (the path 
   * is relative to the engine's startup directory) 
   *
   * @return the application independent real resource path
   * 
   * @deprecated Method will throw a <code>java.lang.UnsupportedOperationException</code> in 
   * the next release after NetWeaver 04. Use <code>getResourcePath(String)</code> instead.
   */
  public static String getResourcePath() {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getResourcePath();
  }

  /**
   * Returns the real public resource path of a specified deployable object.
   * <br>
   * On SAP J2EE Engine 6.40, this is ".\temp\webdynpro\web\<objectName>" (the path 
   * is relative to the engine's startup directory)   
   * 
   * @param objectName the name of an deployable object
   * @return the deployable object dependent real resource path
   */
  public static String getResourcePath(String objectName) {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getResourcePath(
      objectName);
  }

  /**
   * Returns the real public resource path of a specified deployable object.
   * <br>
   * On SAP J2EE Engine 6.40, this is ".\temp\webdynpro\web\<objectName>\<resourceName>" 
   * (the path is relative to the engine's startup directory)
   *
   * @param objectName the name of an deployable object
   * @param resourceName the name of the resource for which the path should be 
   * created. This parameter must not be <code>null</code> or unspecified. It 
   * is interpreted as a relative path (relativ to the specified deployable object) 
   * @return the deployable object dependent real resource path
   */
  public static String getResourcePath(String objectName, String resourceName) {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getResourcePath(
      objectName, resourceName);
  }

  /**
   * Returns the real public resource path of a specified deployable object part.
   * <br>
   * On SAP J2EE Engine 6.40, this is ".\temp\webdynpro\web\<objectName>\<partType>\<deployableObjectPart>" 
   * (the path is relative to the engine's startup directory), where the <code>objectName</code> 
   * is the name of the deployable object of the <code>deployableObjectPart</code>, 
   * and <code>partType</code> is the type of the deployable object part.  
   *   
   * @param deployableObjectPart a deployable object part
   * @return the deployable object part dependent real resource path
   */
  public static String getResourcePath(WDDeployableObjectPart deployableObjectPart) {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getResourcePath(
      deployableObjectPart);
  }

  /**
   * Returns the real public resource path of a specified deployable object part.
   * <br>
   * On SAP J2EE Engine 6.40, this is ".\temp\webdynpro\web\<objectName>\<partType>\<deployableObjectPart>\<resourceName>" 
   * (the path is relative to the engine's startup directory), where the <code>objectName</code> 
   * is the name of the deployable object of the <code>deployableObjectPart</code>, 
   * and <code>partType</code> is the type of the deployable object part.  
   *
   * @param deployableObjectPart a deployable object part
   * @param resourceName the name of the resource for which the path should be 
   * created. This parameter must not be <code>null</code> or unspecified. It 
   * is interpreted as a relative path (relativ to the specified deployable object).    
   * @return the deployable object part dependent real resource path
   * @exception WDAliasResolvingException if the specified <code>resourceName</code> 
   * contains an alias that couldn't be resolved
   */
  public static String getResourcePath(WDDeployableObjectPart deployableObjectPart, String resourceName)
    throws WDAliasResolvingException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getResourcePath(
      deployableObjectPart,
      resourceName);
  }

  /**
   * Returns the web resource path, which is independent from any
   * deployable object or application.
   * <br>    
   * On SAP J2EE Engine 6.40, this is "/webdynpro/resources" 
   * (the relative URL to the Web resources root). 
   *
   * @return the application independent web resource path
   */
  public static String getWebResourcePath() {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWebResourcePath();
  }

  /**
   * Returns the used web resource path of a specified deployable object.
   * <br>    
   * On SAP J2EE Engine 6.40, this is "/webdynpro/resources/<objectName>" 
   * (the relative URL to the resource of the deployable object).   
   *
   * @param objectName the name of an deployable object.
   * @return the web resource path of a specified deployable object
   */
  public static String getWebResourcePath(String objectName) {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWebResourcePath(
      objectName);
  }

  /**
   * Returns the used web resource path of a specified deployable object part.
   * <br>    
   * On SAP J2EE Engine 6.40, this is "/webdynpro/resources/<objectName>/<partType>/<deployableObjectPart>" 
   * (the relative URL to the resource of the deployable object part), whereat the 
   * <code>objectName</code> is the name of the deployable object of the 
   * <code>deployableObjectPart</code>,  and <code>partType</code> is the type of the 
   * deployable object part.  
   *
   * @param deployableObjectPart a deployable object part
   * @return the web resource path of a specified deployable object part
   */
  public static String getWebResourcePath(WDDeployableObjectPart deployableObjectPart) {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWebResourcePath(
      deployableObjectPart);
  }

  /**
   * Returns an URL for a specified web resource located in a specified
   * deployable object.
   * <br>    
   * On SAP J2EE Engine 6.40, this is "/webdynpro/resources/<objectName>/<resourceName>" 
   * (the relative URL to the specified resource of the given deployable object).     
   *
   * @param objectName the name of an deployable object. This parameter must not be 
   * <code>null</code> or unspecified.
   * @param resourceName the name of the resource (inclusive any relative path information
   * like "pictures/myImage.gif"); This parameter must not be <code>null</code> or unspecified.
   * @return the URL for a web resource located in a specified deployable object
   * @exception WDURLException if the URL to the web resource couldn't be generated
   */
  public static String getWebResourceURL(String objectName, String resourceName) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWebResourceURL(
      objectName,
      resourceName);
  }
  
  /**
   * Returns an URL that points to the root of the public resources of the specified
   * deployable object. Returns the same like the method <code>getWebResourcePath(String)</code>. 
   *
   * @param objectName the name of an deployable object. This parameter must not be 
   * <code>null</code> or unspecified.   
   * @return the URL for a web resource located in a specified deployable object
   * @exception WDURLException if the URL to the web resource couldn't be generated
   * @see #getWebResourcePath(String)
   */
  public static String getWebResourceURL(String objectName)  throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWebResourceURL(
      objectName);   
  }

  /**
   * Returns an URL for a specified web resource of a specified deployable object part. 
   * <br>    
   * On SAP J2EE Engine 6.40, this is "/webdynpro/resources/<objectName>/<partType>/<depoyableObjectName>/<resourceName>" 
   * (the relative URL to the specified resource of the given deployable object part), 
   * whereat the <code>objectName</code> is the name of the deployable object of the 
   * <code>deployableObjectPart</code>.     
   *
   * @param deployableObjectPart a deployable object part
   * @param resourceName the name of the resource (inclusive any relative path information
   * like "pictures/myImage.gif"). This parameter must not be <code>null</code> or unspecified.
   * @return the URL for a web resource located in a specified deployable object part
   * @exception WDURLException if the URL to the web resource couldn't be generated
   */
  public static String getWebResourceURL(WDDeployableObjectPart deployableObjectPart, String resourceName)
    throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWebResourceURL(
      deployableObjectPart,
      resourceName);
  }
  
  /**
   * Returns an URL pointing to the root of the specified deployable object part. 
   * Returns the same like the method <code>getWebResourcePath(WDDeployableObjectPart)</code>. 
   *
   * @param deployableObjectPart a deployable object part
   * @return the URL for a web resource located in a specified deployable object part
   * @exception WDURLException if the URL to the web resource couldn't be generated
   * @see #getWebResourcePath(WDDeployableObjectPart)
   */
  public static String getWebResourceURL(WDDeployableObjectPart deployableObjectPart)
    throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWebResourceURL(
      deployableObjectPart);
  }

  /**
   * Returns an absolute URL for a specified web resource located in a specified
   * deployable object. See also <code>getWebResourceURL(String, String)</code>.   
   * 
   *
   * @param objectName the name of an deployable object. This parameter must not be 
   * <code>null</code> or unspecified.
   * @param resourceName the name of the resource (inclusive any relative path information
   * like "pictures/myImage.gif"). This parameter must not be <code>null</code> or unspecified.
   * @return the URL for a web resource located in a specified deployable object
   * @exception WDURLException if the URL to the web resource couldn't be generated
   * @see #getWebResourceURL(String, String)
   */
  public static String getAbsoluteWebResourceURL(String objectName, String resourceName) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getAbsoluteWebResourceURL(objectName, resourceName);
  }
  
  /**
   * Returns an absolute URL that points to the root of the public resources of the specified
   * deployable object. See also <code>getWebResourceURL(String)</code>.  
   *
   * @param objectName the name of an deployable object. This parameter must not be 
   * <code>null</code> or unspecified.
   * @return the URL for a web resource located in a specified deployable object
   * @exception WDURLException if the URL to the web resource couldn't be generated
   * @see #getWebResourceURL(String)
   */
  public static String getAbsoluteWebResourceURL(String objectName) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getAbsoluteWebResourceURL(objectName);
  }

  /**
   * Returns an absolute URL pointing to the root of the specified deployable object part.
   * See also <code>getWebResourceURL(WDDeployableObjectPart)</code>.  
   *
   * @param deployableObjectPart a deployable object part
   * @return the URL for a web resource located in a specified deployable object part
   * @exception WDURLException if the URL to the web resource couldn't be generated
   * @see #getWebResourceURL(WDDeployableObjectPart)
   */
  public static String getAbsoluteWebResourceURL(WDDeployableObjectPart deployableObjectPart)
    throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getAbsoluteWebResourceURL(deployableObjectPart);
  }
  
  /**
   * Returns an absolute URL for a specified web resource of a specified deployable object part.
   * See also <code>getWebResourceURL(WDDeployableObjectPart, String)</code>.
   *
   * @param deployableObjectPart a deployable object part
   * @param resourceName the name of the resource (inclusive any relative path information
   * like "pictures/myImage.gif"). This parameter must not be <code>null</code> or unspecified.
   * @return the URL for a web resource located in a specified deployable object part
   * @exception WDURLException if the URL to the web resource couldn't be generated
   * @see #getWebResourceURL(WDDeployableObjectPart, String)
   */
  public static String getAbsoluteWebResourceURL(WDDeployableObjectPart deployableObjectPart, String resourceName)
    throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getAbsoluteWebResourceURL(deployableObjectPart, resourceName);
  }

  /**
   * Return is the application URL to the deployable object part supplied as input parameter. 
   * Precondition: deployableObjectPart.getType() == WDDeployableObjectPartType.APPLICATION 
   *
   * @param deployableObjectPart a deployable object part representing a Web Dynpro application 
   * @return the URL to the application 
   * @exception WDURLException if the URL couldn't be generated
   */
  public static String getApplicationURL(WDDeployableObjectPart deployableObjectPart) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getApplicationURL(
      deployableObjectPart);
  }
  
  /**
   * Return is the application URL to the deployable object part supplied as input parameter. 
   * Precondition: There must be a deployable object part of type <code>WDDeployableObjectPartType.APPLICATION</code> with 
   * the given <code>applicationPartName</code> within the deployable object with 
   * name <code>deployableObjectName</code>.   
   * 
   * @param deployableObjectName the name of a deployable object
   * @param applicationPartName the name of a application deployable object part
   * @return the URL to the application
   * @exception WDURLException if the URL couldn't be generated
   */
  public static String getApplicationURL(String deployableObjectName, String applicationPartName) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getApplicationURL(deployableObjectName,applicationPartName); 
  }
  
  /**
   * Return is the dynamic part of the application URL for the application specified by 
   * <code>applicationPartName</code> and the deployable object <code>deployableObjectName</code>. 
   * 
   * @param deployableObjectName the name of the deployable object to which the application belongs
   * @param applicationPartName the (fully-qualified) name of the application part 
   * @return the dynamic part of the application URL (e.g. "local/Test/TestApp" for an application with 
   * application part name "com.sap.test.TestApp" that is in the deployable object "local/Test")
   * @throws WDURLException if the URL couldn't be generated
   */
  public static String getDynamicApplicationURL(String deployableObjectName, String applicationPartName) throws WDURLException {
    return AbstractManager
          .getServerPlatformImplementation()
          .getURLGeneratorFactory()
          .getURLGenerator()
          .getDynamicApplicationURL(deployableObjectName,applicationPartName);   
  }
  
  /**
   * Return is the dynamic part of the application URL for the application part specified by 
   * <code>deployableObjectPart</code>. <br>
   * Precondition: deployableObjectPart.getType() == WDDeployableObjectPartType.APPLICATION 
   * 
   * @param deployableObjectPart the deployable object part of the application 
   * @return the dynamic part of the application URL (e.g. "local/Test/TestApp" for an application with 
   * application part name "com.sap.test.TestApp" that is in the deployable object "local/Test")
   * @throws WDURLException if the URL couldn't be generated
   */  
  public static String getDynamicApplicationURL(WDDeployableObjectPart deployableObjectPart) throws WDURLException {
    return AbstractManager
          .getServerPlatformImplementation()
          .getURLGeneratorFactory()
          .getURLGenerator()
          .getDynamicApplicationURL(deployableObjectPart);     
  }
  
  /**
   * Return is the static part of a Web Dynpro application URL. 
   * <br>
   * On SAP J2EE Engine 6.40, this is "/webdynpro/dispatcher". 
   *  
   * @return the static part of a Web Dynpro application URL
   */
  public static String getStaticApplicationURL() {
    return AbstractManager.getServerPlatformImplementation().getURLGeneratorFactory().getURLGenerator().getStaticApplicationURL();
  }
  
  /**
   * Return is the application URL of the application deployable object part supplied as input parameter. 
   * The application URL refers to the host, port and protocol of the Web dispatcher that is 
   * specified in the "default" configuration property set. The Web dispatcher is responsible for 
   * dispatching incoming requests to the server nodes of the cluster in a workload balanced way.
   * <br>     
   * If no Web dispatcher is defined in the "default" properties, the method returns the same URL 
   * as when calling the method <code>getApplicationURL(...)</code>. 
   * <br>
   * Precondition: deployableObjectPart.getType() == WDDeployableObjectPartType.APPLICATION    
   * 
   * @param deployableObjectPart a deployable object part representing a Web Dynpro application 
   * @return the workload balanced URL of the application 
   * @exception WDURLException if the URL couldn't be generated
   */  
  public static String getWorkloadBalancedApplicationURL(WDDeployableObjectPart deployableObjectPart) throws WDURLException  {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWorkloadBalancedApplicationURL(deployableObjectPart);       
  }
  
  /**
   * Return is the application URL of the application deployable object part supplied as input 
   * parameter. The application URL refers to the host, port and protocol of the Web dispatcher 
   * that is specified in the "default" configuration property set. The Web dispatcher is 
   * responsible for dispatching incoming requests to the server nodes of the cluster in a 
   * workload balanced way.
   * <br>     
   * If no Web dispatcher is defined in the "default" properties, the method returns the same URL 
   * as when calling the method <code>getApplicationURL(...)</code>.
   * <br> 
   * Precondition: There must be a deployable object part of type 
   * <code>WDDeployableObjectPartType.APPLICATION</code> with the given <code>applicationPartName</code> within 
   * the deployable object with name <code>deployableObjectName</code>.   
   * 
   * @param deployableObjectName the name of a deployable object
   * @param applicationPartName the name of a application deployable object part
   * @return the URL to the application
   * @exception WDURLException if the URL couldn't be generated
   */
  public static String getWorkloadBalancedApplicationURL(String deployableObjectName, String applicationPartName) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWorkloadBalancedApplicationURL(deployableObjectName, applicationPartName); 
  }
   
  /**
   * Return is the application URL for the specified <code>deployableObjectPart</code> with appended  
   * URL parameters given by <code>urlParameters</code>. All key/value pairs in the map must 
   * be of type java.lang.String.
   * 
   * @param deployableObjectPart a deployable object part representing a Web Dynpro application 
   * @param urlParameters key/value pairs added as URL parameters; they must be of type String
   * @return the application URL for the specified <code>deployableObjectPart</code> with appended  
   * URL parameters given by <code>urlParameters</code>
   * @throws WDURLException if the URL couldn't be generated 
   * @see com.sap.tc.webdynpro.services.sal.url.api.WDURLGenerator#getApplicationURL(WDDeployableObjectPart)
   */
  public static String getApplicationURL(WDDeployableObjectPart deployableObjectPart, Map urlParameters) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getApplicationURL(deployableObjectPart, urlParameters);       
  } 
  
  /**
   * Return is the application URL for the specified deployable object part with appended  
   * URL parameters given by <code>urlParameters</code>. All key/value pairs in the map must 
   * be of type java.lang.String.
   * 
   * @param objectName name of a deployable object that contains the application
   * @param partName the name of a application deployable object part
   * @param urlParameters key/value pairs added as URL parameters; they must be of type String
   * @return the application URL for the specified deployable object part with appended  
   * URL parameters given by <code>urlParameters</code>
   * @throws WDURLException if the URL couldn't be generated 
   * @see com.sap.tc.webdynpro.services.sal.url.api.WDURLGenerator#getApplicationURL(String, String)
   */  
  public static String getApplicationURL(String objectName, String partName, Map urlParameters) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getApplicationURL(objectName, partName, urlParameters);       
  }
  
  /**
   * Return is the workload balanced application URL for the specified <code>deployableObjectPart</code> with appended  
   * URL parameters given by <code>urlParameters</code>. All key/value pairs in the map must 
   * be of type java.lang.String.
   * 
   * @param deployableObjectPart a deployable object part representing a Web Dynpro application 
   * @param urlParameters key/value pairs added as URL parameters; they must be of type String
   * @return the workload balanced application URL for the specified 
   * <code>deployableObjectPart</code> with appended URL parameters given by <code>urlParameters</code>
   * @throws WDURLException if the URL couldn't be generated 
   * @see com.sap.tc.webdynpro.services.sal.url.api.WDURLGenerator#getWorkloadBalancedApplicationURL(WDDeployableObjectPart)
   */  
  public static String getWorkloadBalancedApplicationURL(WDDeployableObjectPart deployableObjectPart, Map urlParameters) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWorkloadBalancedApplicationURL(deployableObjectPart, urlParameters);      
  } 
  
  /**
   * Return is the workload balanced application URL for the specified deployable object 
   * part with appended URL parameters given by <code>urlParameters</code>. All key/value 
   * pairs in the map must be of type java.lang.String.
   * 
   * @param objectName name of a deployable object that contains the application
   * @param partName the name of a application deployable object part
   * @param urlParameters key/value pairs added as URL parameters; they must be of type String
   * @return the workload balanced application URL for the specified deployable object part with appended  
   * URL parameters given by <code>urlParameters</code>
   * @throws WDURLException if the URL couldn't be generated 
   * @see com.sap.tc.webdynpro.services.sal.url.api.WDURLGenerator#getWorkloadBalancedApplicationURL(String, String)
   */   
  public static String getWorkloadBalancedApplicationURL(String objectName, String partName, Map urlParameters) throws WDURLException {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getWorkloadBalancedApplicationURL(objectName, partName, urlParameters);       
  }
  
  /**
   * Return is the resource path to the SAP icons.
   * <br>
   * On SAP J2EE Engine 6.40, this is ".\temp\webdynpro\web\sap.com\tc~wd~dispwda\global\sapIcons" 
   * (the path is relative to the engine's startup directory)    
   *  
   * @return the resource path to the SAP icons
   * @see com.sap.tc.webdynpro.services.sal.url.api.WDURLGenerator#getResourcePath()
   */
  public static String getSAPIconsResourcePath() {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getSAPIconsResourcePath();     
  }
  
  /**
   * Return is the Web resource path to the SAP icons.
   * <br>    
   * On SAP J2EE Engine 6.40, this is "/webdynpro/resources/sap.com/tc~wd~dispwda/global/sapIcons" 
   * (the relative URL to the root of the SAP icons).    
   *  
   * @return the Web resource path to the SAP icons
   * @see com.sap.tc.webdynpro.services.sal.url.api.WDURLGenerator#getWebResourcePath()
   */  
  public static String getSAPIconsWebResourcePath() {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getSAPIconsWebResourcePath();     
  } 
  
  /**
   * Return is the resource path to the Portal icons.
   * <br>
   * On SAP J2EE Engine 6.40, this is ".\temp\webdynpro\web\sap.com\tc~wd~dispwda\global\portalIcons" 
   * (the path is relative to the engine's startup directory)    
   *  
   * @return the resource path to the Portal icons
   * @see com.sap.tc.webdynpro.services.sal.url.api.WDURLGenerator#getResourcePath()
   */
  public static String getPortalIconsResourcePath() {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getPortalIconsResourcePath();     
  }
  
  /**
   * Return is the Web resource path to the Portal icons.
   * <br>    
   * On SAP J2EE Engine 6.40, this is "/webdynpro/resources/sap.com/tc~wd~dispwda/global/portalIcons" 
   * (the relative URL to the root of the SAP icons).    
   *  
   * @return the Web resource path to the portal icons
   * @see com.sap.tc.webdynpro.services.sal.url.api.WDURLGenerator#getWebResourcePath()
   */  
  public static String getPortalIconsWebResourcePath() {
    return AbstractManager
      .getServerPlatformImplementation()
      .getURLGeneratorFactory()
      .getURLGenerator()
      .getPortalIconsWebResourcePath();     
  }   
}
