/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/um/api/WDClientUser.java#4 $
 */

package com.sap.tc.webdynpro.services.sal.um.api;

import com.sap.tc.logging.Category;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;
import com.sap.tc.webdynpro.services.sal.core.AbstractManager;
import com.sap.tc.webdynpro.services.sal.deployment.api.WDDeployableObjectPart;

/**
 * <code>WDClientUser</code> provides static methods to access
 * an <code>IWDClientUser</code> instance.
 * <p>
 * <b>Example:</b> An <code>IWDClientUser</code> can be accessed as follows:
 * <pre>
 * // Let clientUserID be an identifier of a client user
 * IWDClientUser clientUser = WDClientUser.getClientUser(clientUserID);
 * </pre>
 *
 * @see com.sap.tc.webdynpro.services.sal.um.api.IWDClientUser
 *
 * @author SAP
 * @SAPWebDynproPart 2
 */
public abstract class WDClientUser {

  private static final Location logger = Location.getLocation(WDClientUser.class.getName());
  static {
    Location.getLocation("ID.com.sap.tc.webdynpro.services.sal.um.api").infoT("$Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/um/api/WDClientUser.java#4 $");
  }

  /**
   * Gets the <code>IWDClientUser</code> with the client user ID supplied
   * as input parameter, if existing. If not existing, an anonymous
   * client user is returned with an arbitrary <code>clientUserID</code>.
   * No user login is forced by this method call.
   *
   * @param clientUserID the unique ID of the client user; if <code>null</code>
   * or not existing yet, an anonymous client user is created. Note that this ID
   * is a Web Dynpro used identifier for an already created client user
   * and does not match with the user id of the user management system.
   * @return the created client user object
   */
  public static IWDClientUser getClientUser(String clientUserID) {
    return AbstractManager.getServerPlatformImplementation().getClientUserFactory().getClientUser(clientUserID);
  }

  /**
   * Gets the current <code>IWDClientUser</code> object, what is either an
   * authenticated or anonymous client user.
   *
   * @return the current client user object
   * @throws WDUMException if there is no user (neither authenticated nor anonymous user)
   * for the current request
   */
  public static IWDClientUser getCurrentUser() throws WDUMException {
    return AbstractManager.getServerPlatformImplementation().getClientUserFactory().getCurrentUser();
  }

  /**
   * Checks whether the client user is currently logged in and returns
   * the associated <code>IWDClientUser</code> object. If the user
   * is not yet logged in, a respective logon page is written as
   * response and null is returned. </p>
   *
   * <b>Note:</b> This method has been superceded by {@link #forceLoggedInClientUser()}.
   * Using this old method results in a call to forceLoggedInClientUser(), the
   * additional parameters <code>request</code> and <code>response</code> are ignored. </p>
   *
   * <b>Note:</b> Delayed logon is currently <it>not</it> supported by
   * Web Dynpro. User authentication should be enabled via the application
   * authentication flag that can be set in the SAP NetWeaver Developer Studio. </p>
   *
   * @param request the request object (ignored)
   * @param response the response object (ignored)
   * @return the currently logged in client user or <code>null</code> otherwise
   *
   * @deprecated This method has been superceded by the parameterless {@link #forceLoggedInClientUser()}
   *   and may be withdrawn with the first new NetWeaver release in 2006.
   */
  public static IWDClientUser forceLoggedInClientUser(Object request, Object response) {
    return AbstractManager.getServerPlatformImplementation().getClientUserFactory().forceLoggedInClientUser();
  }

  /**
   * Checks whether the client user is currently logged in and returns
   * the associated <code>IWDClientUser</code> object. If the user
   * is not yet logged in, a respective logon page is written as
   * response and null is returned.
   * <br>
   * <b>Note:</b> Delayed logon is currently <it>not</it> supported by
   * Web Dynpro. User authentication should be enabled via the application
   * authentication flag that can be set in the SAP NetWeaver IDE.
   *
   * @return the currently logged in client user or <code>null</code> otherwise
   */
  public static IWDClientUser forceLoggedInClientUser() {
    return AbstractManager.getServerPlatformImplementation().getClientUserFactory().forceLoggedInClientUser();
  }

  /**
   * Checks whether the client user is currently logged in and
   * returns his/her user id. If no client user is currently logged in,
   * <code>null</code> is returned.
   *
   * <b>Note:</b> This method has been superceded by {@link #getLoggedInClientUser()}.
   * Using this old method results in a call to getLoggedInClientUser(), the
   * additional parameters <code>request</code> and <code>response</code> are ignored. </p>
   *
   * @param request the request object (ignored)
   * @param response the response object (ignored)
   * @return the currently logged in client user or <code>null</code> otherwise
   *
   * @deprecated This method has been superceded by the parameterless {@link #getLoggedInClientUser()}
   *   and may be withdrawn with the first new NetWeaver release in 2006.
   */
  public static IWDClientUser getLoggedInClientUser(Object request, Object response) {
    return AbstractManager.getServerPlatformImplementation().getClientUserFactory().getLoggedInClientUser();
  }

  /**
   * Checks whether the client user is currently logged in and
   * returns his/her user id. If no client user is currently logged in,
   * <code>null</code> is returned.
   *
   * @return the currently logged in client user or <code>null</code> otherwise
   */
  public static IWDClientUser getLoggedInClientUser() {
    return AbstractManager.getServerPlatformImplementation().getClientUserFactory().getLoggedInClientUser();
  }

  /**
   * Logs off client user by invalidating the session and deleting the
   * authentication cookies. The user will also be sent to a logoff
   * page which acknowledges that a logoff has taken place with a
   * link for the user to logon again.
   * <p>
   * <b>Note:</b> This method has been superceded by {@link #forceLogoffClientUser(String)}.
   * Using this old method results in a call to forceLogoffClientUser(url), the
   * additional parameters <code>request</code> and <code>response</code> are ignored. </p>
   *
   * @param request the request object (ignored)
   * @param response the response object (ignored)
   * @param url the URL of the page that is shown to the user of logoff was done
   *
   * @deprecated This method has been superceded by {@link #forceLogoffClientUser(String)}
   *   and may be withdrawn with the first new NetWeaver release in 2006.
   */
  public static void forceLogoffClientUser(Object request, Object response, String url) {
    AbstractManager.getServerPlatformImplementation().getClientUserFactory().forceLogoffClientUser(url);
  }

  /**
   * Logs off client user by invalidating the session and deleting the
   * authentication cookies. The user will also be sent to a logoff
   * page which acknowledges that a logoff has taken place with a
   * link for the user to logon again.
   * <p>
   * A call of <code>forceLogoffClientUser</code> does the following concrete steps:
   * <ol>
   * <li> Logout message is sent to User Management Engine (UME). The UME deletes the
   * authentication cookie and deletes the current HTTP session.
   * <li> Web Dynpro exits the current application instance.
   * <li> Web Dynpro redirects to the specified URL. If the specified URL is null,
   *      the redirect is done to the "LogoffURL" URL that can be specified in
   *      the application properties. If this URL is also not defined, a redirect
   *      to a Web Dynpro internal logoff page is done.
   * </ol>
   *
   * @param url the URL of the page that is shown to the user after logoff was done.
   * If the parameter is null, the redirect is done to the "LogoffURL" URL that can be
   * specified in the application properties. If this URL is also not defined, a redirect
   * to a Web Dynpro internal logoff page is done.
   */
  public static void forceLogoffClientUser(String url) {
    AbstractManager.getServerPlatformImplementation().getClientUserFactory().forceLogoffClientUser(url);
  }

  /**
   * Releases the client user instance. This does not include a logoff of the current client
   * user, but just a removal from the <code>IWDClientUser</code> instance associated to the user.
   *
   * @param clientUser the client user to release
   *
   * @deprecated the method is no longer supported and may be withdrawn with the first new
   * NetWeaver release in 2006.
   */
  public static void releaseClientUser(IWDClientUser clientUser) {
    Category.SYS_USER_INTERFACE.log(
      Severity.WARNING,
      logger,
      "Usage of deprecated method ''WDClientUser.releaseClientUser'' that is no longer supported.");
  }

  /**
   * Return is the array of client users which currently run on
   * the Web Dynpro runtime.
   *
   * @return the array of currently registered client users
   */
  public static IWDClientUser[] getClientUsers() {
    return AbstractManager.getServerPlatformImplementation().getClientUserFactory().getClientUsers();
  }

  /**
   * This method checks the configuration whether authentication is required. If so, the <code>forceLoggedInClientUser</code>
   * method is called. The method returns true if authentication was successful, i.e. either the user was
   * correctly authenticated or no authentication is required in the current session. Otherwise, false is returned. In
   * this latter case, the application should abort execution of the program flow since authentication is required
   * but there is no authenticated user yet.
   * <br>
   * Contract of this method: the Web Dynpro user management is based on SAP UME 4.0 or any compliant version.
   * <br>
   * Note: Web Dynpro 6.40 does not support "delayed logon" yet. This means that after user authentication is
   * done, the application is currently restarted. Delayer logon will be supported in a later version of
   * Web Dynpro.
   *
   * @param applicationPart the application part containing the configuration of the application
   * @return true if authentication was successful, and false otherwise
   * @throws WDIllegalArgumentException if <code>applicationPart</code> is null
   * or <code>applicationPart.getType() != WDDeployableObjectPartType.APPLICATION</code>
   */
  public static boolean checkAuthentication(WDDeployableObjectPart applicationPart) {
    return AbstractManager.getServerPlatformImplementation().getClientUserFactory().checkAuthentication(applicationPart);
  }

  /**
   * Return is true if the authentication flag is set for the application specified as
   * input parameter.
   * @param applicationPart a deployable object part that specifies an application
   * @return true if authentication is required (i.e. authentication flag is set), otherwise false)
   * @throws WDIllegalArgumentException if applicationPart is null or
   * <code>applicationPart.getType() != WDDeployableObjectPartType.APPLICATION</code>
   */
  public static boolean isAuthenticationRequired(WDDeployableObjectPart applicationPart) {
    return AbstractManager.getServerPlatformImplementation().getClientUserFactory().isAuthenticationRequired(applicationPart);
  }
}
