/*
 * SAP Copyright (c) 2002
 * All rights reserved
 * 
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/um/api/IWDClientUser.java#3 $
 */
package com.sap.tc.webdynpro.services.sal.um.api;

import java.security.AccessControlException;
import java.security.Permission;
import java.util.Locale;

import com.sap.tc.webdynpro.services.sal.api.IWDService;
import com.sap.security.api.IUser; 

/**
 * <code>IWDClientUser</code> describes a Web Dynpro client user. It provides 
 * accessor methods to some attributes of the user profile. It also provides a 
 * method to access the <code>com.sap.security.api.IUser</code> which then can be 
 * used to get full access on the user's profile. 
 * <br>
 * An <code>IWDClientUser</code> can represent an authenticated user or an anonymous 
 * user. See method <code>isAnonymousUser()</code> for details. 
 *
 * @see com.sap.tc.webdynpro.services.sal.core.IClientUserFactory
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public interface IWDClientUser extends IWDService {

  /**
   * Return is the client user ID of the actual client user. This is the 
   * identifier used by Web Dynpro to identify an already existing client 
   * user. 
   *  
   * @return the client user ID of the actual client user
   */
  public String getClientUserID();

  /**
   * Return is the user's first name. 
   * 
   * @return the user's first name; in case of an anonymous user,
   * the empty String is returned
   */
  public String getFirstName();

  /**
   * Return is the user's last name. 
   * 
   * @return the user's last name; in case of an anonymous user,
   * the empty String is returned
   */
  public String getLastName();

  /**
   * Return is the user's title.
   * 
   * @return the user's title; in case of an anonymous user,
   * the empty String is returned
   */
  public String getTitle();

  /**
   * Return is the salutation for the user. 
   * 
   * @return the salutation; in case of an anonymous user,
   * the empty String is returned
   */
  public String getSalutation();

  /**
   * Return is the locale the is specified for the current user by the 
   * user management system. In case of an anonymous user, <code>null</code> 
   * is returned. <br>
   * Please note that the user locale must not be the same as the session locale. 
   * The session locale also evaluates a locale set by URL parameter and the locale 
   * set in the browser settings. See also 
   * {@link com.sap.tc.webdynpro.services.sal.localization.api.WDResourceHandler#getCurrentSessionLocale()}.    
   * 
   * @return the locale the user prefers; in case of an anonymous user,
   * <code>null</code> is returned
   */
  public Locale getLocale();
  
  /**
   * Return is true if the client user is an anonymous user, i.e.
   * a user who can not be identified.
   * 
   * @return true if the client user is an anonymous user
   */
  public boolean isAnonymousUser();
  
  /**
   * Return is true if the client user is an anonymous user, i.e.
   * a user who can not be identified.
   * 
   * @return true if the client user is an anonymous user
   * @deprecated The name of this method contained a typo, use {@link #isAnonymousUser()}
   *   instead. The method with the old name has been kept for backward compatibility reasons 
   *   only and may be withdrawn with the first new NetWeaver release in 2006.
   */
  public boolean isAnonymouseUser();
  
  /**
   * Return is the <code>com.sap.security.api.IUser</code> representing 
   * the current user. In case of the current user represents an anonymous 
   * user, <code>null</code> is returned. 
   * 
   * @return the <code>com.sap.security.api.IUser</code> representing 
   * the current user
   */
  public IUser getSAPUser(); 
  
  /**
   * Returns true if the user has (possibly via implication) the given permission.
   * In case of an anonymous user, <code>null</code> is returned.
   * 
   * @param permission the permission the user may have
   * @return true if the user has the given permission
   */
  public boolean hasPermission(Permission permission);

  /**
   * If the user has (possibly via implication) the specified permission, this
   * method does nothing. If not, it will throw an <code>AccessControlEcxeption</code>,
   * and possibly trigger appropriate tracing action.
   * <p>
   * In case of an anonymous user, an <code>AccessControlEcxeption</code> is thrown.
   *
   * @param permission the permission the user may have   
   * @throws AccessControlException if the user doesn't have the specified permission
   */
  public void checkPermission(Permission permission) throws AccessControlException;
}
