/*
 * SAP Copyright (c) 2002
 * All rights reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/sl/api/IWDJCOClientConnection.java#5 $
 */
package com.sap.tc.webdynpro.services.sal.sl.api;

import java.util.Locale;

import com.sap.mw.jco.JCO;
import com.sap.tc.webdynpro.services.sal.api.IWDService;

/**
 * This interface describes a JCO connection used by a Web Dynpro application.
 * It depends of the impementation if the used JCO connection is get from a <code>JCO.Pool</code> or if it
 * is realized by using one single <code>JCO.Client</code> instance.
 * 
 *@author SAP
 * @SAPWebDynproPart 2
 */
public interface IWDJCOClientConnection extends IWDService {

  /**
   * Return a <code>JCO.Client</code> instance.
   */
  public JCO.Client getClient();

  /**
   * Release the actual <code>JCOClientConnection</code> instance.
   */
  public void release();
  
  /**
   * Return the maximal pool size.
   * @return the maximal pool size
   */
  public int getMaxPoolSize();
  
	/**
	 * Return the maximal number of connections.
	 * @return the maximal number of connections.
	 */
	public int getMaxNumberOfConnections();
	
	/**
	 * Return the maximal wait time.
	 * @return the maximal wait time.
	 */
	public int getMaxWaitTime();
	
	/**
	 * Return the maximal connection time out.
	 * @return the maximal connection time out.
	 */
	public int getConnectionTimeOut();  
  

  /**
   * Return the name of the used message server instance. This value could be NULL if the connection is defined 
   * by application server and system id.
   * 
   * @return the name of the used message server instance.   
   */
  public String getMessageServer();

  /**
   * Return the name of the used system. This value could be NULL if the connection is defined 
   * by application server and system id.
   * 
   * @return the the name of the used system.   
   */
  public String getSystemName();

  /**
   * Return the name of the used logon group. This value could be NULL if the connection is defined 
   * by application server and system id.
   * 
   * @return the the name of the used logon group.   
   */
  public String getLogonGroup();

  /**
   * Return the name of the used application server instance. The value could be NULL if the connection is defined
   * by message server, system name and logon group.
   * 
   * @return the name of the used application server instance
   */
  public String getApplicationServer();

  /**
   * Return the used system number. The value could be NULL if the connection is defined
   * by message server, system name and logon group.
   * 
   * @return the used system number
   */
  public String getSystemNumber();

  /**
   * Return the name of the used client.
   * 
   * @return the the name of the used client.   
   */
  public String getClientName();

  /**
  * Return the name of the used user.
  * 
  * @return ther name of the used user.
  */
  public String getUser();

  /**
   * Return the used password. This value could be NULL depending on the defined user defintion.
   * 
   * @return the used password
   */
  public String getPassword();

  /** 
   * Return the used user definition.
   * 
   * @return the used user definition.
   */
  public String getUserDefinition();

  /**
   * Return the used language.
   * 
   * @return the used language.
   */
  public String getLanguage();
  
  /**
   * Defines the used language. Should not be used (only for xApps workaround)
   * 
   * @param language the needed language in a locale string 
   */
  public void setLanguage (Locale languageLocale);
  
  /**
   * Return the used Secure network connection (SNC) mode, 0 (off) or 1 (on)
   * @return the used Secure network connection (SNC) mode, 0 (off) or 1 (on)
   */
	public String getSNCMode();
	
	/**
	 * Return the used SNC level of security, 1 to 9 
	 * @return the used SNC level of security, 1 to 9 
	 */
	public String getSNCSecurityLevel();

	/**
	 * Return the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN"
	 * @return the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN"
	 */
	public String getSNCPartnerName();
	
	/**
	 * Return the used SNC name. Overrides default SNC partner
	 * @return the used SNC name. Overrides default SNC partner
	 */
	public String getSNCName();

	/** 
	 * Return the path to the library which provides SNC service
	 * @return the path to the library which provides SNC service
	 */ 
	public String getSNCLibraryPath();
  
  /**
   * Return the name of the J2EE cluster, to which the JCO connection is assigned
   * @return the name of the J2EE cluster, to which the JCO connection is assigned
   */
  public String getJ2EEClusterName();
  
  /**
   * Return an unique key identifiying the used technical system. 
   * If two <class>IWDJCOClientConnection</code> instances have the same
   * key they would use the same technical connection (i.e. the same technical <code>JCO.Pool</code>)
   * @return
   */
  public String getSystemIdentifier ();
  
	/**
	 * Pings the  <code>IWDJCOClientConnection</code>. 
	 * The ping does not authenticate the user. Only the connection properties defining the technical system are used 
	 * for the ping.
	 * 
	 * @return TRUE if the ping works, FALSE otherwise.
	 */
	public boolean ping();
	
	/**
	 * Updates a instance of <code>IJCOClientConnection</code> defined by the message server name,
	 * the system name and the logon group.
	 * The used repository depends on the implementation.
	 * 
   * @param maxPoolSize the maximal pool size (i.e. the maximal open connections managed by the pool)
   * @param maxNumberOfConnections the maximal number of connections provided by the pool.
   * @param maxWaitTime the maximal wait time (i.e. the maximal time, waiting for a new <code>JCO.Client</code>)
   * @param connectionTimeOut defines the time after that a open and unused connection managed by the pool is closed.
	 * @param msgServerName the name of the message server
	 * @param systemName the name of the R/3 system
	 * @param logonGroup the used logon group
	 * @param client the used client
	 * @param userDefinition defines which kind of user definition is used.
	 *                       Possible values are:
	 *                          'useSSO': the user is defined using SSO
	 *                          'useX509': the user is defined using X509 client certificate
	 *                          'useDefinedUser': the defined user is used
	 * @param user the name of the user
	 * @param password the password of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER)
	 * @param sncMode the used Secure network connection (SNC) mode, 0 (off) or 1 (on) (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param sncSecurityLevel the used SNC level of security, 1 to 9 (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param sncPartnerName the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN" (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param sncName the used SNC name. Overrides default SNC partner (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param sncLibraryPath the path to the library which provides SNC service (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param language the language of the user
	 * @throws JCOClientException if the update of the new JCO client connection failed.  
	 */
	public void update(
		int maxPoolSize,
		int maxNumberOfConnections,
		int maxWaitTime,
		int connectionTimeOut,
		String msgServerName,
		String systemName,
		String logonGroup,
		String client,
		WDUserDefinitionType userDefinition,
		String user,
		String password,
		String sncMode,
		String sncSecurityLevel,
		String sncPartnerName,
		String sncName,
		String sncLibraryPath,
		String language)
		throws WDSystemLandscapeException;	

  /**
   * Updates a instance of <code>IJCOClientConnection</code> defined by the message server name,
   * the system name and the logon group.
   * The used repository depends on the implementation.
   * 
   * @param maxPoolSize the maximal pool size (i.e. the maximal open connections managed by the pool)
   * @param msgServerName the name of the message server
   * @param systemName the name of the R/3 system
   * @param logonGroup the used logon group
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is used
   * @param user the name of the user
   * @param password the password of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER)
   * @param sncMode the used Secure network connection (SNC) mode, 0 (off) or 1 (on) (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncSecurityLevel the used SNC level of security, 1 to 9 (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncPartnerName the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN" (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncName the used SNC name. Overrides default SNC partner (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncLibraryPath the path to the library which provides SNC service (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param language the language of the user
   * @throws JCOClientException if the update of the new JCO client connection failed.  
   */
  public void update(
    int maxPoolSize,
    String msgServerName,
    String systemName,
    String logonGroup,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
	  String sncMode,
	  String sncSecurityLevel,
	  String sncPartnerName,
	  String sncName,
	  String sncLibraryPath,
    String language)
    throws WDSystemLandscapeException;
    
  /**
   * Updates a instance of <code>IJCOClientConnection</code> defined by the message server name,
   * the system name and the logon group.
   * The used repository depends on the implementation.
   * 
   * @param maxPoolSize the maximal pool size (i.e. the maximal open connections managed by the pool) 
   * @param msgServerName the name of the message server
   * @param systemName the name of the R/3 system
   * @param logonGroup the used logon group
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is used
   * @param user the name of the user
   * @param password the password of the user (is needed when <userDefinition> is set to WDUserDefinition.DEFINED_USER)
   * @param language the language of the user
   * @throws JCOClientException if the update of the new JCO client connection failed.  
   */
  public void update(
    int maxPoolSize,
    String msgServerName,
    String systemName,
    String logonGroup,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
    String language)
    throws WDSystemLandscapeException;    
    
	/**
	 * Updates a instance of <code>IJCOClientConnection</code> defined by application server name 
	 * and system number.
	 * The used repository depends on the implementation.
	 * 
   * @param maxPoolSize the maximal pool size (i.e. the maximal open connections managed by the pool)
   * @param maxNumberOfConnections the maximal number of connections provided by the pool.
   * @param maxWaitTime the maximal wait time (i.e. the maximal time, waiting for a new <code>JCO.Client</code>)
   * @param connectionTimeOut defines the time after that a open and unused connection managed by the pool is closed.
	 * @param appServerName the name of the application server
	 * @param systemNumber the system number name of the R/3 system
	 * @param client the used client
	 * @param userDefinition defines which kind of user definition is used.
	 *                       Possible values are:
	 *                          'useSSO': the user is defined using SSO
	 *                          'useX509': the user is defined using X509 client certificate
	 *                          'useDefinedUser': the defined user is used
	 * @param user the name of the user
	 * @param password the password of the user (is needed when <userDefinition> is set to 'WDUserDefinition.DEFINED_USER)
	 * @param sncMode the used Secure network connection (SNC) mode, 0 (off) or 1 (on) (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param sncSecurityLevel the used SNC level of security, 1 to 9 (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param sncPartnerName the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN" (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param sncName the used SNC name. Overrides default SNC partner (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param sncLibraryPath the path to the library which provides SNC service (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
	 * @param language the language of the user
	 * @param j2eeClusterName the name of the J2EE cluster, to which the JCO connection is assigned
	 * @throws JCOClientException if the creation of the new JCO client connection failed.  
	 */
	public void update(
		int maxPoolSize,
		int maxNumberOfConnections,
		int maxWaitTime,
		int connectionTimeOut,
		String appServerName,
		String systemNumber,
		String client,
		WDUserDefinitionType userDefinition,
		String user,
		String password,
		String sncMode,
		String sncSecurityLevel,
		String sncPartnerName,
		String sncName,
		String sncLibraryPath,
		String language)
			throws WDSystemLandscapeException;
    
  /**
   * Updates a instance of <code>IJCOClientConnection</code> defined by application server name 
   * and system number.
   * The used repository depends on the implementation.
   * 
   * @param maxPoolSize the maximal pool size (i.e. the maximal open connections managed by the pool)
   * @param appServerName the name of the application server
   * @param systemNumber the system number name of the R/3 system
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is used
   * @param user the name of the user
   * @param password the password of the user (is needed when <userDefinition> is set to 'WDUserDefinition.DEFINED_USER)
   * @param sncMode the used Secure network connection (SNC) mode, 0 (off) or 1 (on) (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncSecurityLevel the used SNC level of security, 1 to 9 (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncPartnerName the SNC partner, e.g. "p:CN=R3, O=XYZ-INC, C=EN" (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncName the used SNC name. Overrides default SNC partner (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param sncLibraryPath the path to the library which provides SNC service (is needed when <userDefinition> is set to WDUserDefinition.SNC_USER)
   * @param language the language of the user
   * @param j2eeClusterName the name of the J2EE cluster, to which the JCO connection is assigned
   * @throws JCOClientException if the creation of the new JCO client connection failed.  
   */
  public void update(
    int maxPoolSize,
    String appServerName,
    String systemNumber,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
    String sncMode,
    String sncSecurityLevel,
    String sncPartnerName,
    String sncName,
    String sncLibraryPath,
    String language)
      throws WDSystemLandscapeException;
    
  /**
   * Updates a instance of <code>IJCOClientConnection</code> defined by application server name 
   * and system number.
   * The used repository depends on the implementation.
   * 
   * @param maxPoolSize the maximal pool size (i.e. the maximal open connections managed by the pool)
   * @param appServerName the name of the application server
   * @param systemNumber the system number name of the R/3 system
   * @param client the used client
   * @param userDefinition defines which kind of user definition is used.
   *                       Possible values are:
   *                          'useSSO': the user is defined using SSO
   *                          'useX509': the user is defined using X509 client certificate
   *                          'useDefinedUser': the defined user is used
   * @param user the name of the user
   * @param password the password of the user (is needed when <userDefinition> is set to 'WDUserDefinition.DEFINED_USER)
   * @param language the language of the user
   * @param j2eeClusterName the name of the J2EE cluster, to which the JCO connection is assigned
   * @throws JCOClientException if the creation of the new JCO client connection failed.  
   */
  public void update(
    int maxPoolSize,
    String appServerName,
    String systemNumber,
    String client,
    WDUserDefinitionType userDefinition,
    String user,
    String password,
    String language)
      throws WDSystemLandscapeException;
}
