/*
 * @(#) $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/repository/api/IWDRepositoryFactory.java#2 $
 *
 * Copyright(c) 2003, SAP AG, All Rights Reserved.
 */
package com.sap.tc.webdynpro.services.sal.repository.api;

import java.util.Locale;

import com.sap.security.api.IUser;
import com.sap.tc.webdynpro.rdc.api.IClient;
import com.sap.tc.webdynpro.repository.api.IWDCustomizingLayer;
import com.sap.tc.webdynpro.repository.api.IWDRepository;
import com.sap.tc.webdynpro.repository.api.IWDRepositoryStructureManager;
import com.sap.tc.webdynpro.repository.api.IWDVariant;

/**
 * Factory for repositories and repository structure manager.
 *
 * @author SAP
 * @SAPWebDynproPart 2
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/repository/api/IWDRepositoryFactory.java#2 $
 */
public interface IWDRepositoryFactory {

  /**
   * Returns a new repository instance.
   * The repository instance is opened in <b>runtime mode</b>, i.e. only
   * transient changes of the repository are possible.    
   * 
   * @param locale the default locale for which the repository is opened
   * @param variant the variant for which the repository is opened
   * @return the newly created repository instance    
   * @throws NullPointerException if one of the input parameters is null 
   */
  public IWDRepository getRepository(Locale locale, IWDVariant variant);

  /**
   * Returns a new repository instance.
   * 
   * @param locale the default locale for which the repository is opened
   * @param variants the variants (variant with highest priority is on index 0)
   * for which the repository is opened
   * @return the newly created repository instance    
   * @throws NullPointerException if one of the input parameters is null, or
   * <code>variants</code> is empty or contains a null value 
   */
  public IWDRepository getRepository(Locale locale, IWDVariant[] variants);

  /**
   * Returns a new repository instance with personalization.
   * The repository instance is opened in <b>runtime mode</b>, i.e. only
   * transient changes of the repository are possible.    
   * 
   * @param locale the default locale for which the repository is opened
   * @param user the user which is used for personalization
   * @param variant the variant for which the repository is opened
   * @return the newly created repository instance   
   * 
   * @throws NullPointerException if one of the input parameters is null 
   */
  public IWDRepository getPersonalizedRepository(Locale locale, IUser user, IWDVariant variant);

  /**
   * Returns a new repository instance with personalization.
   * 
   * @param locale the default locale for which the repository is opened
   * @param user the user which is used for personalization
   * @param variants the variants (variant with highest priority is on index 0)
   * for which the repository is opened
   * @return the newly created repository instance    
   * @throws NullPointerException if one of the input parameters is null, or
   * <code>variants</code> is empty or contains a null value 
   */
  public IWDRepository getPersonalizedRepository(Locale locale, IUser user, IWDVariant[] variants);

  /**
   * Returns a new repository instance.
   * The repository instance is opened in <b>maintenance mode</b>, i.e.
   * persistent changes of the repository are possible. 
   * 
   * @param locale the default locale for which the repository is opened
   * @param variant the variant for which the repository is be opened
   * @param customizingLayer the customizing layer for which the repository is
   * opended 
   * @return the newly created repository instance
   * @throws NullPointerException if one of the input parameters are null 
   * 
   * @deprecated This method may be withdrawn with the first new NetWeaver release in 2006. 
   * No replacement is available, since the editable mode of the Web Dynpro runtime repository 
   * might no longer be supported.    
   */
  public IWDRepository getEditableRepository(Locale locale, IWDVariant variant, IWDCustomizingLayer customizingLayer);

  /**
   * Returns a new structure manager instance.
   * @return a new structure manager instance
   */
  public IWDRepositoryStructureManager getStructureManager();

  /**
   * Returns the specified variant. 
   * The returned variant cannot be modified and should only be used to 
   * open a repository. To modify a variant, the
   * <code>IWDRepositoryStructureManager</code> has to be used. 
   * @param name the name of the variant
   * @return the requested variant or null if no such variant exists
   */
  public IWDVariant getVariant(String name);

  /**
   * Returns the specified customizing layer. 
   * The returned customizing layer cannot be modified and should only be used
   * to open a repository. To modify a customizing layer , the
   * <code>IWDRepositoryStructureManager</code> has to be used.
   * @param name the name of the customizing layer
   * @return the requested customizing layer or null if no such customizing
   * layer exists
   */
  public IWDCustomizingLayer getCustomizingLayer(String name);

  /** 
   * Sets the versioning client.
   * This versioning client is passed to newly created editable
   * <code>IWDRepository</code> and <code>IWDRepositoryStructureManager</code> 
   * instances.
   * @param dtrClient versioning client
   * @deprecated Use {@link #setDtrInformation(String, int, String, String, String, String)} instead.
   *   This API may be withdrawn with the first new NetWeaver release in 2006.
   */
  public void setVersioningClient(IClient dtrClient);

  /**
   * Sets the information needed to create the versioning client.
   * The method must only be called if the repository is to be connected to
   * a design time repository (DTR) in maintenance mode.
   * @param host host of DTR
   * @param port port of DTR
   * @param basePath base path of DTR (usually "/dtr")
   * @param scLocation location of the software component
   * @param user user used to connect to the DTR
   * @param pwd password of the user
   * 
   * @deprecated This method may be withdrawn with the first new NetWeaver release in 2006. 
   * No replacement is available, since the connection to DTR of the Web Dynpro runtime repository 
   * might no longer be supported. 
   */
  public void setDtrInformation(String host, int port, String basePath, String scLocation, String user, String pwd);
  
  /** 
   * Sets the development component.
   * This is passed to newly created editable <code>IWDRepository</code> and 
   * <code>IWDRepositoryStructureManager</code> instances.
   * @param name name of the development component  
   */
  public void setDevelopmentComponent(String name);

  /**
   * Sets the name of the customizing layer.
   * This customizing layer is used for repositories opened in runtime mode.
   * If this method is not called the system wide customizing layer is used.
   * This setting takes only effect on the current factory instance.
   * @param customizingLayerName Name of the customizing layer to set
   */
  public void setRuntimeCustomizingLayer(String customizingLayerName);
}
