/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/localization/api/WDResourceHandler.java#3 $
 */

package com.sap.tc.webdynpro.services.sal.localization.api;

import java.util.Locale;
import java.util.TimeZone;

import com.sap.tc.webdynpro.services.sal.core.AbstractManager;

/**
 * The class <code>WDResourceHandler</code> is used to access instances
 * of type <code>IWDResourceHandler</code>.
 * <p>
 * <b>Example:</b> An <code>IWDResourceHandler</code> object can be accessed as follows:
 * <pre>
 * IWDResourceHandler resourceHandler = WDResourceHandler.createResourceHandler(Locale.GERMAN);
 * </pre>
 *
 * @see com.sap.tc.webdynpro.services.sal.localization.api.IWDResourceHandler
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public abstract class WDResourceHandler {

  /**
   * Return is the resource handler representing the
   * set locale and timezone. If one of the input parameters is set to null,
   * the default locale / timezone of the virtual machine is used.
   *
   * @param locale the locale to be used
   * @param timezone the timezone to be used
   * @return the resource handler according to the specified I18N settings
   */
  public static IWDResourceHandler createResourceHandler(Locale locale, TimeZone timezone) {
    return AbstractManager.getServerPlatformImplementation().getResourceHandlerFactory().createResourceHandler(
      locale,
      timezone);
  }

  /**
   * Return is the resource handler representing the
   * set locale. If the input parameter is set to <code>null</code>,
   * the default locale of the virtual machine is used.
   *
   * @param locale the locale to be used
   * @return the resource handler according to the specified I18N settings
   */
  public static IWDResourceHandler createResourceHandler(Locale locale) {
    return AbstractManager.getServerPlatformImplementation().getResourceHandlerFactory().createResourceHandler(locale);
  }

  /**
   * Return is the resource handler with the locale for the current client user. 
   * For a description how the locale of the current client user is detected, 
   * see method <code>getCurrentUserLocale</code>. 
   * 
   * @return the resource handler with the locale for the current user
   * @see com.sap.tc.webdynpro.services.sal.localization.api.WDResourceHandler#getCurrentUserLocale
   */
  public static IWDResourceHandler createResourceHandlerForCurrentUser() {
    return AbstractManager
      .getServerPlatformImplementation()
      .getResourceHandlerFactory()
      .createResourceHandlerForCurrentUser();
  }

  /**
   * Return is the resource handler with the locale for the current 
   * session. For a description how the locale of the current client user is detected, 
   * see method <code>getCurrentSessionLocale()</code>.  
   * 
   * @return the resource handler with the locale for the current session
   * @see com.sap.tc.webdynpro.services.sal.localization.api.WDResourceHandler#getCurrentSessionLocale
   */
  public static IWDResourceHandler createResourceHandlerForCurrentSession() {
    return AbstractManager
      .getServerPlatformImplementation()
      .getResourceHandlerFactory()
      .createResourceHandlerForCurrentSession();
  }

  /**
   * Return is the locale for the current client user. The concrete locale is  
   * determined in the following order: 
   * in the following way: 
   * <ol>
   * <li> If a default locale is specified for the current (authenticated) client user 
   *      (by the user's profile), this locale is taken. 
   * <li> If the HTTP header contains an "Accept-Language" header with a valid ISO-639 
   *      language, this language is used as current user locale. 
   * <li> The default locale of the current application (the application 
   *      of the current session) is used if specified. 
   * <li> If no such default locale is specified, the default locale for the 
   *      Web Dynpro runtime is used if specified. 
   * <li> If this one is also not specified, the default locale of the server 
   *      VM is used as last fallback locale.   
   * </ol>
   * 
   * @return the locale for the current client user
   * 
   * @deprecated Use <code>WDClientUser.getCurrentUser().getLocale()</code> instead.
   *   This method may be withdrawn with the first new NetWeaver release in 2006.
   */
  public static Locale getCurrentUserLocale() {
    return AbstractManager.getServerPlatformImplementation().getResourceHandlerFactory().getCurrentUserLocale();
  }

  /**
   * Return is the locale for the current session. See {@link 
   * com.sap.tc.webdynpro.services.sal.um.api.IWDClientUser#getLocale()} for 
   * a description of the fallback sequence that is used to determine the 
   * locale for the current session.  
   * <p>
   * Note that the session locale is determined once for each client session 
   * and then stored as part of the session. If e.g. the browser locale is used 
   * as session locale and the locale is changed in the browser settings,  
   * the change first becomes effective after a new browser instance is opened. 
   *   
   * @return the locale for the current session
   */
  public static Locale getCurrentSessionLocale() {
    return AbstractManager.getServerPlatformImplementation().getResourceHandlerFactory().getCurrentSessionLocale();
  }
}
