/*
 * SAP Copyright (c) 2002
 * All rights reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/localization/api/IWDResourceHandler.java#1 $
 */
package com.sap.tc.webdynpro.services.sal.localization.api;

import java.text.DateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.TimeZone;

import com.sap.tc.webdynpro.services.sal.api.IWDService;

/**
 * <code>IWDResourceHandler</code> represents a handler for locale 
 * specific resources. A resource handler is associated to a 
 * specified locale and optionally to a certain time zone. A 
 * resource handler is responsible for 
 * <ul>
 * <li> loading resource bundles 
 * <li> accessing translated messages
 * <li> formatting locale specific strings like numbers, percentages, 
 * dates, time, etc. 
 * </ul>
 *
 * @see com.sap.tc.webdynpro.services.sal.core.IResourceHandlerFactory
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public interface IWDResourceHandler extends IWDService {
  
  /** Pattern for date. */
  public final static String DEFAULT_DATE_PATTERN = "yyyy-MM-dd HH:mm:ss:SSSS z";
  
  /** Style for date and time. */
  public final static int DEFAULT_DATETIMESTYLE = DateFormat.FULL;

  /** Style for date and time. */
  public final static int DATETIME_STYLE_SHORT = DateFormat.SHORT;
  
  /** Style for date and time. */  
  public final static int DATETIME_STYLE_MEDIUM = DateFormat.MEDIUM;
  
  /** Style for date and time. */  
  public final static int DATETIME_STYLE_LONG = DateFormat.LONG;
  
  /** Style for date and time. */  
  public final static int DATETIME_STYLE_FULL = DateFormat.FULL;
  
  /** Style for date and time. */  
  public final static int DATETIME_STYLE_NONE =
    DATETIME_STYLE_SHORT + DATETIME_STYLE_MEDIUM + DATETIME_STYLE_LONG + DATETIME_STYLE_FULL;

  /** Minimum number of digits for currency amounts. */
  public final static int DEFAULT_CURRENCY_MINFRACTIONDIGITS = 2;
  
  /** Maximum number of digits for currency amounts. */  
  public final static int DEFAULT_CURRENCY_MAXFRACTIONDIGITS = 2;
  
  /** Minimum number of digits for currency amounts (integer). */  
  public final static int DEFAULT_CURRENCY_MININTEGERDIGITS = 1;
  
  /** Maximum number of digits for currency amounts (integer). */  
  public final static int DEFAULT_CURRENCY_MAXINTEGERDIGITS = Integer.MAX_VALUE;

  /** Minimum number of digits for numbers. */
  public final static int DEFAULT_NUMBER_MINFRACTIONDIGITS = 0;
  
  /** Maximum number of digits for numbers. */
  public final static int DEFAULT_NUMBER_MAXFRACTIONDIGITS = 3;
  
  /** Minimum number of digits for integer numbers. */  
  public final static int DEFAULT_NUMBER_MININTEGERDIGITS = 1;
  
  /** Maximum number of digits for integer numbers. */  
  public final static int DEFAULT_NUMBER_MAXINTEGERDIGITS = Integer.MAX_VALUE;

  /** Minimum number of digits for percentages. */
  public final static int DEFAULT_PERCENT_MINFRACTIONDIGITS = 0;
  
  /** Maximum number of digits for percentages. */
  public final static int DEFAULT_PERCENT_MAXFRACTIONDIGITS = 0;
  
  /** Minimum number of digits for integer percentages. */  
  public final static int DEFAULT_PERCENT_MININTEGERDIGITS = 1;
  
  /** Maximum number of digits for integer percentages. */  
  public final static int DEFAULT_PERCENT_MAXINTEGERDIGITS = Integer.MAX_VALUE;

  /**
   * Initializes the resource handler with the specified 
   * locale and time zone. 
   *
   * @param locale locale of this resource handler
   * @param timezone timezone of this resource handler; if <code>null</code>, 
   * the default time zone for the specified locale is taken. 
   */
  public void init(Locale locale, TimeZone timezone);

  /**
   * Return is the time zone of this resource handler.
   * 
   * @return the time zone of this resource handler
   */
  public TimeZone getTimezone();

  /**
   * Return is the locale used for this resource handler.
   * 
   * @return the actually used locale
   */
  public Locale getLocale();

  /**
   * Loads the resource bundle specified by <code>resourceBundleName</code> using
   * the classloader supplied as input parameter.
   * 
   * @param resourceBundleName base name of the resource bundle that should be loaded; the 
   * resource bundle can be a java.utl.ListResourceBundle or a java.util.PropertyResourceBundle;  
   * this parameter must not be <code>null</code>
   * @param loader classloader used to load the resource bundle; if set to 
   * <code>null</code> the default classloader is used
   * @exception MissingResourceException if the specified resource bundle couldn't be loaded
   */
  public void loadResourceBundle(String resourceBundleName, ClassLoader loader) throws MissingResourceException;

  /**
   * Loads the resource bundle specified by <code>resourceBundleName</code> using
   * the default classloader of the virtual machine.
   * 
   * @param resourceBundleName the name of the resource bundle that should be loaded;
   * must not be <code>null</code>
   * @exception MissingResourceException if the specified resource bundle couldn't be loaded
   */
  public void loadResourceBundle(String resourceBundleName) throws MissingResourceException;

  /**
   * Loads the resource bundle specified by <code>resourceBundleName</code> using
   * the default classloader of the virtual machine and returns the loaded resource bundle.
   * 
   * @param resourceBundleName the name of the resource bundle that should be loaded;
   * must not be <code>null</code>
   * @return the loaded resource bundle 
   * @exception MissingResourceException if the specified resource bundle couldn't be loaded
   */
  public ResourceBundle getResourceBundle( String resourceBundleName ) throws MissingResourceException;
  
  /**
   * Loads the resource bundle specified by <code>resourceBundleName</code> using
   * the classloader supplied as input parameter and returns the loaded resource bundle.
   * 
   * @param resourceBundleName base name of the resource bundle that should be loaded; the 
   * resource bundle can be a java.utl.ListResourceBundle or a java.util.PropertyResourceBundle;  
   * this parameter must not be <code>null</code>
   * @param loader classloader used to load the resource bundle; if set to 
   * <code>null</code> the default classloader is used
   * @return the loaded resource bundle 
   * @exception MissingResourceException if the specified resource bundle couldn't be loaded
   */
  public ResourceBundle getResourceBundle(String resourceBundleName, ClassLoader loader) throws MissingResourceException;

  /**
   * Return is the translated string found in the loaded resource bundle,
   * if an entry with the given key exists. Otherwise the supplied key
   * parameter is returned.
   * 
   * @param key the key of the searched NLS string
   * @return the translated string, if existing; otherwise <code>key</code> is returned
   */
  public String getString(String key); 
  
  /**
   * Returns an enumeration of all keys. If no resource bundle is loaded yet, 
   * <code>null</code> is returned. 
   * 
   * @return an enumeration of all keys
   */
  public Enumeration getKeys(); 

  /**
   * Return is the message identified by key using the set resource bundle and
   * inserted arguments according to java.text.MessageFormat specification.
   *
   * @param key the key of the searched NLS string
   * @param args the message arguments according to java.text.MessageFormat
   * @return the translated, formatted string, if existing; otherwise <code>key</code>
   * is returned
   */
  public String getFormattedString(String key, Object[] args);

  /**
   * Sets date format (see documentation on SimpleDateFormat) to be used for
   * formatting format-unspecified dates.
   * 
   * @param pattern date pattern
   */
  public void setDateFormat(String pattern);

  /**
   * Sets date format (see documentation on DateFormat) to be used for formatting
   * format-unspecified dates.
   * 
   * @param dateStyle  date formatting style; possible values are DATETIME_STYLE_XXX
   * constants defined in this interface.
   * @param timeStyle  time formatting style; possible values are DATETIME_STYLE_XXX
   * constants defined in this interface
   */
  public void setDateFormat(int dateStyle, int timeStyle);

  /**
   * Formats date according to locale/timezone using the set pattern/style.
   *
   * @param date  date to be formatted
   * @return      formatted date
   */
  public String formatDate(long date);

  /**
   * Formats date according to locale/timezone using the set pattern/style.
   *
   * @param date  date to be formatted
   * @return      formatted date
   */
  public String formatDate(Date date);

  /**
   * Sets the currency format to be used for formatting format-unspecified
   * currencies.
   *
   * @param minFractionDigits minimum number of digits for fraction part of number
   * @param maxFractionDigits maximum number of digits for fraction part of number
   * @param minIntegerDigits minimum number of digits for integer part of number
   * @param maxIntegerDigits maximum number of digits for integer part of number
   */
  public void setCurrencyFormat(
    int minFractionDigits,
    int maxFractionDigits,
    int minIntegerDigits,
    int maxIntegerDigits);

  /**
   * Formats currency according to locale using the set format.
   *
   * @param currency  currency to be formatted
   * @return formatted currency
   */
  public String formatCurrency(long currency);

  /**
   * Formats currency according to locale using the set format.
   *
   * @param currency  currency to be formatted
   * @return formatted currency
   */
  public String formatCurrency(Number currency);

  /**
   * Sets the number format to be used for formatting
   * format-unspecified numbers.
   *
   * @param minFractionDigits minimum number of digits for fraction part of number
   * @param maxFractionDigits maximum number of digits for fraction part of number
   * @param minIntegerDigits minimum number of digits for integer part of number
   * @param maxIntegerDigits maximum number of digits for integer part of number
   */
  public void setNumberFormat(int minFractionDigits, int maxFractionDigits, int minIntegerDigits, int maxIntegerDigits);

  /**
   * Formats number according to locale using the set format.
   *
   * @param number  number to be formatted
   * @return formatted number
   */
  public String formatNumber(long number);

  /**
   * Formats number according to locale using the set format.
   *
   * @param number  number to be formatted
   * @return formatted number
   */
  public String formatNumber(Number number);

  /**
   * Sets percentage format to be used for formatting format-unspecified percents.
   *
   * @param minFractionDigits minimum number of digits for fraction part of number
   * @param maxFractionDigits maximum number of digits for fraction part of number
   * @param minIntegerDigits minimum number of digits for integer part of number
   * @param maxIntegerDigits maximum number of digits for integer part of number
   */
  public void setPercentageFormat(
    int minFractionDigits,
    int maxFractionDigits,
    int minIntegerDigits,
    int maxIntegerDigits);

  /**
   * Formats percentage according to locale using the set format.
   *
   * @param percent  percent to be formatted
   * @return formatted percent
   */
  public String formatPercentage(long percent);

  /**
   * Formats percentage according to locale using the set format.
   *
   * @param percent  percent to be formatted
   * @return         formatted percent
   */
  public String formatPercentage(Number percent);
  
}
