/*
 * SAP Copyright (c) 2002
 * All rights reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/deployment/api/WDDeployableObjectState.java#2 $
 */
package com.sap.tc.webdynpro.services.sal.deployment.api;

import java.util.HashMap;
import java.util.Map;

import com.sap.tc.webdynpro.basesrvc.util.StringUtil;
import com.sap.tc.webdynpro.services.exceptions.WDIllegalArgumentException;

/**
 * Constants representing the possible deployment states of Web Dynpro applications. 
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public class WDDeployableObjectState {
  
  /** The application is deployed but it's state is stopped, i.e. not ready-to-use currently. */
  public static final WDDeployableObjectState STOPPED;
  
  /** The application is deployed and started, i.e. it is ready-to-use. */
  public static final WDDeployableObjectState STARTED;

  /** The application was deployed before and is removed in the meantime. */
  public static final WDDeployableObjectState REMOVED;
  
  /** The application state is unknown. */
  public static final WDDeployableObjectState UNKNOWN;

  private static Map valueMap = new HashMap(3, 1f); 
  static {
    valueMap.put("stopped", STOPPED = new WDDeployableObjectState("stopped"));
    valueMap.put("started", STARTED = new WDDeployableObjectState("started"));
    valueMap.put("removed", REMOVED = new WDDeployableObjectState("removed"));
    valueMap.put("unknown", UNKNOWN = new WDDeployableObjectState("unknown")); 
  }

  /** deployment state an a deployable object */
  private final String state; 

  /**
   * Constructor for WDDeployableObjectState.
   */
  private WDDeployableObjectState(String state) {
    this.state = state; 
  }
  
  /**
   * Return is the name of the state. 
   * @author SAP
   */
  public String toString() {
    return state;  
  }
  
  /**
   * Return is the <code>WDDeployableObjectState</code> instance belonging to the  
   * integer identifier specified as input parameter.   
   * 
   * @param state the state identifier 
   * @return the <code>WDDeployableObjectState<code> instance representing the 
   * state specified by the integer in the argument
   * 
   * @deprecated For better consistency across the Web Dynpro API this method
   *   has been renamed to {@link #valueOf(String)}. The method with the old 
   *   name has been kept for backward compatibility reasons and may be withdrawn
   *   from the public API with the first new NetWeaver release in 2006.
   */
  public static WDDeployableObjectState getStateObject(int state) {
    switch (state) {
    case 0: return UNKNOWN;
    case 1: return STARTED;
    case 2: return STOPPED; 
    case 3: return REMOVED; 
    default: 
      throw new WDIllegalArgumentException("WDDeployableObjectState: Invalid enumeration value ''" + state + "''.");    
    }  
  } 
    
  /**
   * Return is the <code>WDDeployableObjectState</code> instance representing the specified state. 
   * If the given state is not defined, an <code>IllegalArgumentException</code> is thrown.
   *  
   * @param state the name of the state
   * @return the <code>WDDeployableObjectState</code> instance representing the specified state
   * @throws WDIllegalArgumentException if the given state is undefined
   */
  public static WDDeployableObjectState valueOf(String state) {
    if (!StringUtil.isEmpty(state)) {       
      Object value = valueMap.get(state.toLowerCase());
      if (value != null) {
        return (WDDeployableObjectState) value;
      }
    }
    throw new WDIllegalArgumentException("WDDeployableObjectState: Invalid enumeration value ''" + state + "''.");  
  }
  
  /**
   * Return is the URL to the icon representing the current deployable object state. 
   * 
   * @return the URL to the icon representing the current deployable object state 
   * @deprecated Method not supported in 6.30 and later.
   *   This method may be withdrawn with the first new NetWeaver release in 2006.
   */ 
  public String getIconURL() {
    // todo 
    //return URLGeneratorInternal.getWebResourceURL(ServerConstants.SYSTEM_DEPLOYABLE_OBJECT_NAME, '/' + ServerConstants.SYSTEM_RELATIVE_RESOURCE_PATH + ???); 
    return "";   
  }
}
