/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/config/api/WDConfiguration.java#3 $
 */

package com.sap.tc.webdynpro.services.sal.config.api;

import com.sap.tc.webdynpro.services.sal.core.AbstractManager;
import com.sap.tc.webdynpro.services.sal.deployment.api.IWDDeployableObject;
import com.sap.tc.webdynpro.services.sal.deployment.api.WDDeployableObjectPart;

/**
 * <code>WDConfiguration</code> provides static methods to access  
 * an <code>IWDConfiguration</code> instance. The configuration can be 
 * located on the following different context levels: 
 * <ul>
 * <li> On the level of the deployable object. 
 * <li> On the level of a specific deployable object part. 
 * </ul> 
 * The name of the configuration unit can be automatically derived from the 
 * context using the methods <code>getConfiguration</code>, or it is stated 
 * explicitly by the user via the methods <code>getConfigurationByName</code>.
 * 
 * <b>Example:</b>  An <code>IWDConfiguration</code> with name "testConfig" that is 
 * located on the level of a deployable object, can be accessed as follows: 
 * <pre>
 * // let objectName be the name of the deployable object
 * IWDConfiguration configuration = WDConfiguration.getConfiguration(objectName, "testConfig"); 
 * </pre>
 * 
 * @see com.sap.tc.webdynpro.services.sal.config.api.IWDConfiguration
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public abstract class WDConfiguration {

  /** 
   * Returns the default configuration object named <code>default.properties</code> 
   * of the specified deployable object if existing. If not existing, a 
   * <code>WDConfigurationNotFoundException</code> is thrown. 
   *
   * @param objectName the name of the deployable object for which the configuration should be loaded
   * @return the default configuration object named "default.properties"   
   * @throws WDConfigurationNotFoundException if the configuration instance couldn't be created
   */
  public static IWDConfiguration getConfiguration(String objectName) throws WDConfigurationNotFoundException {
    return AbstractManager.getServerPlatformImplementation().getConfigurationFactory().getConfiguration(objectName);
  }

  /**
   * Returns the configuration specified by <code>configName</code>. This configuration unit 
   * is expected to be contained in the specified deployable object. 
   * <br>
   * Example: 
   * On the level of a deployable Object named "testObject", there is a property file named "testConfig.properties". 
   * Then, you can access this configuration unit using the following call: 
   * <pre>
   * IWDConfiguration cfg = WDConfiguration.getConfigurationByName("testObject", "testConfig.properties");   
   * // alternatively: 
   * cfg = WDConfiguration.getConfigurationByName("testObject", "testConfig");
   * </pre>
   * 
   * @param objectName the name of the deployable object for which the configuration should be loaded
   * @param configName the name of the configuration unit
   * @return the configuration specified by <code>configName</code>
   * @throws WDConfigurationNotFoundException if the configuration instance couldn't be created
   */  
  public static IWDConfiguration getConfigurationByName(String objectName, String configName)
    throws WDConfigurationNotFoundException {
    return AbstractManager.getServerPlatformImplementation().getConfigurationFactory().getConfigurationByName(
      objectName,
      configName);
  }
  
  /** 
   * Returns the default configuration object named <code>default.properties</code> 
   * of the specified deployable object if existing. If not existing, a 
   * <code>WDConfigurationNotFoundException</code> is thrown. 
   *
   * @param deployableObject the deployable object for which the configuration should be loaded
   * @return the default configuration object named "default.properties"    
   * @throws WDConfigurationNotFoundException if the configuration instance couldn't be created
   */ 
  public static IWDConfiguration getConfiguration(IWDDeployableObject deployableObject) throws WDConfigurationNotFoundException {
    return AbstractManager.getServerPlatformImplementation().getConfigurationFactory().getConfiguration(deployableObject.getName());
  }

  /**
   * Returns the configuration specified by <code>configName</code>. This configuration unit 
   * is expected to be contained in the specified deployable object. Otherwise a 
   * <code>WDConfigurationNotFoundException</code> is thrown.
   * <br>
   * Example: On the level of a deployable Object "testObject", there is a property file named "testConfig.properties". 
   * Then, you can access this configuration unit using the following call: 
   * <pre>
   * IWDDeployableObject testObject = ...;   // some coding to get the deployable object
   * IWDConfiguration cfg = WDConfiguration.getConfigurationByName(testObject, "testConfig.properties");   
   * // alternatively: 
   * cfg = WDConfiguration.getConfigurationByName(testObject, "testConfig");
   * </pre>   
   * 
   * @param objectName the deployable object for which the configuration should be loaded
   * @param configName the name of the configuration unit 
   * @return the configuration specified by <code>configName</code>
   * @throws WDConfigurationNotFoundException if the configuration instance couldn't be created
   */
  public static IWDConfiguration getConfigurationByName(IWDDeployableObject deployableObject, String configName)
    throws WDConfigurationNotFoundException {
    return AbstractManager.getServerPlatformImplementation().getConfigurationFactory().getConfigurationByName(
      deployableObject.getName(),
      configName);
  }      

  /**
   * Returns the configuration object named <code><unqualified name of the deployable object part>.properties</code> 
   * for the specified deployable object part if existing. If not existing, a 
   * <code>WDConfigurationNotFoundException</code> is thrown.
   * <br>
   * Example: Let "com.sap.test.TestPart" be the name of a deployable object part. Then, using a reference 
   * to a WDDeployableObjectPart that represents this part, the method will look for a configuration unit 
   * called "TestPart.properties" on the level of the given part.    
   *
   * @param deployableObjectPart the name of the deployable object part
   * @return the configuration object
   * @throws WDConfigurationNotFoundException if the configuration instance couldn't be created
   */ 
  public static IWDConfiguration getConfiguration(WDDeployableObjectPart deployableObjectPart)
    throws WDConfigurationNotFoundException {
    return AbstractManager.getServerPlatformImplementation().getConfigurationFactory().getConfiguration(deployableObjectPart);
  }

  /**
   * Returns the configuration specified by <code>configName</code>. This configuration unit 
   * is expected to be contained in the specified deployable object under the specified 
   * application context. Otherwise a <code>WDConfigurationNotFoundException</code> is thrown.
   * 
   * @param deployableObjectPart the name of the deployable object part
   * @param configName the name of the configuration unit
   * @return the configuration specified by <code>configName</code>
   * @throws WDConfigurationNotFoundException if the configuration instance couldn't be created
   */  
  public static IWDConfiguration getConfigurationByName(WDDeployableObjectPart deployableObjectPart, String configName)
    throws WDConfigurationNotFoundException {
    return AbstractManager.getServerPlatformImplementation().getConfigurationFactory().getConfigurationByName(
      deployableObjectPart,
      configName);
  } 
}
