/*
 * SAP Copyright (c) 2002
 * All rights reserved
 * 
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/config/api/IWDConfiguration.java#1 $ 
 */

package com.sap.tc.webdynpro.services.sal.config.api;

import java.util.Date;
import com.sap.tc.webdynpro.services.sal.api.IWDService;

/**
 * <code>IWDConfiguration</code> specifies the interface to configuration data. 
 * Thereby, the configuration consists of name/value pairs. The interface 
 * provides methods for typed access to these name/value pairs, as well as 
 * methods to define default values which are taken in case a certain 
 * configuartion is not set. 
 * 
 * @see com.sap.tc.webdynpro.services.sal.core.IConfigurationFactory
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public interface IWDConfiguration extends IWDService {
    
  /**
   * Gets configuration entry of type string by key.
   *
   * @param key key of configuration entry
   * @return value of configuration entry
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public String getStringEntry(String key) throws WDInvalidConfigParameterException;

  /**
   * Gets configuration entry of type string by key.
   *
   * @param key                  key of configuration entry
   * @param defaultValue         default value for configuration entry
   * @return                     value of configuration entry
   */
  public String getStringEntry(String key, String defaultValue);

  /**
   * Gets configuration entry of type multiple strings by key.
   *
   * @param key                  key of configuration entry
   * @return                     values of configuration entry
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public String[] getStringEntries(String key) throws WDInvalidConfigParameterException;

  /**
   * Gets configuration entry of type date by key.
   *
   * @param key                  key of configuration entry
   * @return                     value of configuration entry
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public Date getDateEntry(String key) throws WDInvalidConfigParameterException;

  /**
   * Gets configuration entry of type date by key.
   *
   * @param key                  key of configuration entry
   * @param defaultValue         default value for configuration entry
   * @return                     value of configuration entry
   */
  public Date getDateEntry(String key, Date defaultValue);

  /**
   * Gets configuration entry of type multiple dates by key.
   *
   * @param key                  key of configuration entry
   * @return                     values of configuration entry
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public Date[] getDateEntries(String key) throws WDInvalidConfigParameterException;

  /**
   * Gets configuration entry of type int by key.
   *
   * @param key                  key of configuration entry
   * @return                     value of configuration entry
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public int getIntEntry(String key) throws WDInvalidConfigParameterException;

  /**
   * Gets configuration entry of type int by key.
   *
   * @param key                  key of configuration entry
   * @param defaultValue         default value for configuration entry
   * @return                     value of configuration entry
   */
  public int getIntEntry(String key, int defaultValue);

  /**
   * Gets configuration entry of type multiple ints by key.
   *
   * @param key                  key of configuration entry
   * @return                     values of configuration entry
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public int[] getIntEntries(String key) throws WDInvalidConfigParameterException;

  /**
   * Gets configuration entry of type long by key.
   *
   * @param key                  key of configuration entry
   * @return                     value of configuration entry
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public long getLongEntry(String key) throws WDInvalidConfigParameterException;

  /**
   * Gets configuration entry of type long by key.
   *
   * @param key                  key of configuration entry
   * @param defaultValue         default value for configuration entry
   * @return                     value of configuration entry
   */
  public long getLongEntry(String key, long defaultValue);

  /**
   * Gets configuration entry of type multiple longs by key.
   *
   * @param key                  key of configuration entry
   * @return                     values of configuration entry
   * @exception ConfigException  Exception raised in failure situation
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public long[] getLongEntries(String key) throws WDInvalidConfigParameterException;

  /**
   * Gets configuration entry of type boolean by key.
   *
   * @param key                  key of configuration entry
   * @return                     value of configuration entry
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public boolean getBooleanEntry(String key) throws WDInvalidConfigParameterException;

  /**
   * Gets configuration entry of type boolean by key.
   *
   * @param key                  key of configuration entry
   * @param defaultValue         default value for configuration entry
   * @return                     value of configuration entry
   */
  public boolean getBooleanEntry(String key, boolean defaultValue);

  /**
   * Gets configuration entry of type multiple booleans by key.
   *
   * @param key                  key of configuration entry
   * @return                     values of configuration entry
   * @throws WDInvalidConfigParameterException if the specified parameter cannot be found in the configuration
   */
  public boolean[] getBooleanEntries(String key) throws WDInvalidConfigParameterException;
  
  /**
   * Get all configuration entry keys.
   *
   * @return all configuration entry keys
   */
  public String[] getAllKeys();
}
