/*
 * SAP Copyright (c) 2002
 * All rights reserved.
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_services/java/com/sap/tc/webdynpro/services/sal/admin/api/WDServerState.java#2 $
 */
package com.sap.tc.webdynpro.services.sal.admin.api;

import com.sap.tc.webdynpro.services.sal.core.AbstractManager;

/**
 * <code>WDServerState</code> provides static methods to access a snapshot of 
 * the server state.
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public abstract class WDServerState {

  private static ServerStateMBean mBean = AbstractManager.getServerPlatformImplementation().getServerState(); 
   
  /** 
   * Return is the array of client user identifiers actually registered client users.
   * @return the array of actually registered client users
   */
  public static String[] getActualClientUsers() {
    return mBean.getActualClientUsers(); 
  }

  /**
   * Return is the number of actually registered client users.
   * @return the number of actually registered client users
   */
  public static int getNumberOfClientUsers() { 
    return mBean.getNumberOfClientUsers(); 
  }

  /**
   * Return is the array of actually running applications. An entry of 
   * the array is <application-name>/<application-id>.
   * @return the array of actually running applications
   */
  public static String[] getActualApplications() {
    return mBean.getActualApplications();  
  }

  /**
   * Return is the number of actually running applications.
   * @return the number of actually running applications
   */
  public static int getNumberOfActualApplications() {
    return mBean.getNumberOfActualApplications(); 
  }

  /**
   * Return is the array of actually running application names and id's for
   * the specified client user. An entry of the array is 
   * <application-name>/<application-id>.
   * @param clientUserId a client user identifier
   * @return the array of actually running applications for the
   * specified client user
   */
  public static String[] getActualApplications(String clientUserId) { 
    return mBean.getActualApplications(clientUserId); 
  }

  /**
   * Return is the number of actually running applications for
   * the specified client user.
   * @param clientUserId a client user
   * @return the number of actually running applications for the
   * specified client user
   */
  public static int getNumberOfActualApplications(String clientUserId) {
    return mBean.getNumberOfActualApplications(clientUserId);
  }

  /**
   * Return is the array of names of application deployable object parts contained in the  
   * deployable object specified as input parameter. 
   * @return the array of names of application deployable object parts contained in the  
   * deployable object specified as input parameter
   */
  public static String[] getApplicationParts(String deployableObjectName) {
    return mBean.getApplicationParts(deployableObjectName); 
  }

  /**
   * Return is the array of names of component deployable object parts contained in the  
   * deployable object specified as input parameter. 
   * @return the array of names of component deployable object parts contained in the  
   * deployable object specified as input parameter
   */
  public static String[] getComponentParts(String deployableObjectName) {
    return mBean.getComponentParts(deployableObjectName);
  }

  /**
   * Return is the array of names of configuration deployable object parts contained in the  
   * deployable object specified as input parameter. 
   * @return the array of names of configuration deployable object parts contained in the  
   * deployable object specified as input parameter
   */
  public static String[] getConfigurationParts(String deployableObjectName) {
    return mBean.getConfigurationParts(deployableObjectName);
  }

  /**
   * Return is the array of deployable object names of currently deployed 
   * Web Dynpro applications. 
   * @return he array of deployable object names of currently deployed 
   * Web Dynpro applications
   */
  public static String[] getDeployableObjects() {
    return mBean.getDeployableObjects();    
  }

  /**
   * Return is the array of names of interface deployable object parts contained in the  
   * deployable object specified as input parameter. 
   * @return the array of names of interface deployable object parts contained in the  
   * deployable object specified as input parameter
   */
  public String[] getInterfaceParts(String deployableObjectName) {
    return mBean.getInterfaceParts(deployableObjectName);
  } 

  /**
   * Return is the array of names of deployable object parts belonging to
   * the specified deployable object.
   * @param deployableObjectName the name of the deployable object for which the 
   * deployable object parts should be returned
   * @return the array of names of deployable object parts belonging to
   * the specified deployable object
   */
  public static String[] getDeployableObjectParts(String deployableObject) {
    return mBean.getDeployableObjectParts(deployableObject);
  }

  /**
   * Return is the number of deployable object parts belonging to
   * the specified deployable object.
   * @param deployableObjectName the name of the deployable object for which the 
   * number of deployable object parts should be returned
   * @return the number of deployable object parts belonging to
   * the specified deployable object
   */
  public static int getNumberOfDeployableObjectParts(String deployableObject) {
    return mBean.getNumberOfDeployableObjectParts(deployableObject);
  }

  /**
   * Return is the array of actual server sessions. An entry in the 
   * array is the <server-session-id>. 
   * @return the array of actual server session identifiers
   */
  public static String[] getActualServerSessions() {
    return mBean.getActualServerSessions(); 
  }

  /**
   * Return is the number of actual server sessions.
   * @return the number of actual server sessions
   */
  public static int getNumberOfActualServerSessions() {
    return mBean.getNumberOfActualServerSessions();
  }

  /**
   * Return is the server session associated to the specified
   * client user.
   * @param clientUserId the client user identifier for which the server session should 
   * be returned
   * @return the server session id associated to the specified
   * client user
   */
  public static String getActualServerSession(String clientUserId) {
    return mBean.getActualServerSession(clientUserId);
  }

  /**
   * Return is the array of actually registered client sessions.
   * @return the array of actually registered client sessions
   */
  public static String[] getActualClientSessions() {
    return mBean.getActualClientSessions(); 
  }

  /**
   * Return is the number of all actually registered client sessions.
   * @return the number of actually registered client sessions
   */
  public static int getNumberOfActualClientSessions() {
    return mBean.getNumberOfActualClientSessions();
  }

  /**
   * Return is the array of actually registered client sessions
   * for the specified server sessions.
   * @param serverSessionID a server session ID
   * @return the array of actually registered client sessions
   * for the specified server sessions
   */
  public static String[] getActualClientSessions(String serverSessionID) {
    return mBean.getActualClientSessions(serverSessionID);
  }

  /**
   * Return is the number of actually registered client sessions
   * for the specified server sessions.
   * @param serverSessionID a server session ID
   * @return the number of actually registered client sessions
   * for the specified server sessions
   */
  public static int getNumberOfActualClientSessions(String serverSessionID) {
    return mBean.getNumberOfActualClientSessions(serverSessionID);
  }

  /**
   * Return is the array of all actually registered client windows.
   * @return the array of all actually registered client window identifiers
   */
  public static String[] getActualClientWindows() {
    return mBean.getActualClientWindows(); 
  }

  /**
   * Return is the number of all actually registered client windows.
   * @return the number of all actually registered client windows
   */
  public static int getNumberOfActualClientWindows() {
    return mBean.getNumberOfActualClientWindows(); 
  }

  /**
   * Return is the array of all actually registered client windows
   * for the specified client session.
   * @param clientSessionID a client session ID 
   * @return the array of all actually registered client window identifiers 
   * for the specified client session
   */
  public static String[] getActualClientWindows(String clientSessionID) {
    return mBean.getActualClientWindows(clientSessionID);
  }

  /**
   * Return is the number of all actually registered client windows
   * for the specified client session.
   * @return the number of all actually registered client windows
   * for the specified client session
   */
  public static int getNumberOfActualClientWindows(String clientSessionID) {
    return mBean.getNumberOfActualClientWindows(clientSessionID);
  }

  /**
   * Return is the string representation of the actual server state,
   * organized in an array of strings. This method can be used, for
   * instance, to log the actual server state in a log file or
   * to print the actual server state on the console. Each entry in 
   * the array represents a separate line. 
   *
   * @return the string representation of the actual server state,
   * organized in an array of strings
   */
  public static String[] getStateAsStringArray() {
    return mBean.getStateAsStringArray();
  }
  
  /**
   * Return is the associated MBean. 
   * @return the associated MBean
   */
  public static ServerStateMBean getMBean() {
    return mBean;   
  }   
}
