#!/usr/bin/perl
#
# $Header: //sapdb/V74/develop/sys/src/install/perl/SAPDB/Install/Instance/Check/Volumes.pm#5 $
# $DateTime: 2002/02/25 10:20:44 $
# $Change: 16934 $
#

package SAPDB::Install::Instance::Check::Volumes;

sub BEGIN {
	@ISA = ('SAPDB::Install::Exporter');
	@EXPORT = ();
	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages = (
		'Instance::Base',
		'Instance::Check::Conditions'
	);

	foreach my $package (@neededPackages) {
	  	unless (defined $repo->Eval 
		("SAPDB::Install::$package", 1.01)) {
				print join ("\n", $repo->GetErr)."\n";
				die;
		}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	} 
}

push @ISA, 
	'SAPDB::Install::Instance::Base',
	'SAPDB::Install::Instance::Check::Conditions';

#
# check access to volumes
#
sub volumes {
	my ($self) = @_; 
	my $dbm = $self->{dbm};

	# no access check on windows
	return 0 if ($^O =~ /^mswin/i);

	# starting release is required
	unless (defined $self->{'starting_release'}) {
		my $rc = $self->conditions ();
		return (-1) unless (defined $rc && $rc == 0);
	}

	$self->set_errorstate ('ERROR');
	$self->msgbegin ("checking volume access");
	$self->msg0 ("checking volume access...\n");

	my $kernelversion = $self->{'starting_release'};

	# analyze volumes by using all parameters in 7.2, 7.3 and 7.4.x x <=2
	my $use_directgetall =
	$kernelversion =~ /^7\.[23]\./ ? 1 :
	$kernelversion =~ /^7\.4\.(\d+)/ && $1 <= 2 ? 1 : 0;

	my $rc = $use_directgetall ? $self->volumes_by_directgetall () :
	                             $self->volumes_by_getvolsall ();

	unless (defined $rc && $rc == 0) {
		$self->msgend ();
		return (-1);
	}

	$self->set_errorstate ('OK');
	$self->msgend ();
	return 0;
}

#
# this should be used beginning with 7.4.03 or better
#
sub volumes_by_getvolsall {
	my ($self) = @_; 
	my $dbm = $self->{dbm};

	$self->msg1 ("analyzing volumes by param_getvolsall\n");

	my $rundirectory = $dbm->param_directget ('RUNDIRECTORY');
	unless (defined $rundirectory) {
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	my $volinfo = $dbm->param_getvolsall ();
	unless (defined $volinfo) {
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	#
	# analyze log configuration
	# 7.4.3 less or equal build 10 are using LOG_MODE,
	# 7.4.3 greater or equal build 11 are using LOG_MIRRORED
	# other keys have been changed too
	#

	my $maxdatavol = 
		defined $volinfo->{MAXDATAVOLUMES} ?
			$volinfo->{MAXDATAVOLUMES} :
		defined $volinfo->{MAXDATADEVSPACES} ?
			$volinfo->{MAXDATADEVSPACES} :
		undef;

	my $maxlogvol =
		defined $volinfo->{MAXLOGVOLUMES} ?
			$volinfo->{MAXLOGVOLUMES} :
		defined $volinfo->{MAXARCHIVELOGS} ?
			$volinfo->{MAXARCHIVELOGS} :
		undef;

	my $has_mirrored_log =
		defined $volinfo->{LOG_MIRRORED} ? 
			($volinfo->{LOG_MIRRORED} =~ /YES/ ? 1 : 0) :
		defined $volinfo->{LOG_MODE} ? 
			($volinfo->{LOG_MODE} =~ /DUAL/ ? 1 : 0) :
		undef;

	unless (defined $has_mirrored_log) {
		$self->msg1 ("cannot determine log configuration\n");
		$self->msg1 ("LOG_MODE or LOG_MIRRORED not found\n");
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	$self->msg1 ("rundirectory     = ".$rundirectory."\n");
	$self->msg1 ("mirrored log     = ".$has_mirrored_log."\n");
	$self->msg1 ("max data volumes = ".$maxdatavol."\n");
	$self->msg1 ("max log volumes  = ".$maxlogvol."\n");

	#
	# build a hash of all volumes
	# fist level key is category
	# second level key is number
	# then array of file name, file type, file size
	#
	my $volumes = {};
	while (my ($key, $val) = each %$volinfo) {
		my $cat =
		$key =~ /^DATA_VOLUME_NAME_/  ? 'DATA' :
		$key =~ /^DATADEV_/           ? 'DATA' :
		$key =~ /^LOG_VOLUME_NAME_/   ? 'LOG'  :
		$key =~ /^ARCHIVE_LOG_/       ? 'LOG'  :
		$key =~ /^M_LOG_VOLUME_NAME_/ ? 'MLOG' :
		$key =~ /^M_ARCHIVE_LOG_/     ? 'MLOG' :
		undef;

		my ($num) = ($key =~ /_(\d+)$/);		
		next unless (defined $num && defined $cat);
		$num = int ($num);

		next if (ref ($val) ne 'HASH');
		$volumes->{$cat}->{$num} = [$val->{name}, $val->{type}, $val->{size}];
	}

	my $knltracefile = $dbm->param_directget ('_KERNELTRACEFILE');
	unless (defined $knltracefile) {
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	my $knltracesize = $dbm->param_directget ('KERNELTRACESIZE');
	unless (defined $knltracesize) {
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	$volumes->{'KNLTRACE'}->{1} = [$knltracefile, 'F', $knltracesize];

	unless (defined $self->{instancetypename}) {
		$self->{instancetypename} =
		$self->{instancetype} =~ /LVC/ ? 'liveCache' : 'database';
	}

	# get db process owner
	my $dbowner = $self->getdbprocessowner ();
	return -1 unless (defined $dbowner);

	my $dbgroup = getgrgid ((getpwnam ($dbowner))[3]);
	$self->msg1	($self->{instancetypename}.
	             " process owner is ".$dbowner.":".$dbgroup."\n");

	# check file access for each volume
	my $rc = 0;
	foreach $cat ('KNLTRACE', 'DATA', 'LOG') {
		 for (my $i = 1; defined $volumes->{$cat}->{$i}; $i++) {
			$rc |= $self->getvolaccess
			($cat, $i, $volumes->{$cat}->{$i}->[0], $rundirectory, $dbowner);

			next unless ($has_mirrored_log == 1 && $cat eq 'LOG');

			# return error if volume mirror is missing
			return -1 unless (defined $volumes->{'MLOG'}->{$i});

			# set size of volume mirror
			$volumes->{'MLOG'}->{$i}->[2] = $volumes->{'LOG'}->{$i}->[2];
			$rc |= $self->getvolaccess
			($cat, $i, $volumes->{'MLOG'}->{$i}->[0], $rundirectory, $dbowner);
		}
	}

	return $rc;
}

#
# this should be used up to 7.4.02
#
sub volumes_by_directgetall {
	my ($self) = @_; 
	my $dbm = $self->{dbm};

	$self->msg1 ("analyzing volumes by param_directgetall\n");

	my $paramall = $dbm->param_directgetall ();	
	unless (defined $paramall) {
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	# first find out general configuration
	# SAPDB 7.2 and 7.3 has system volumes

	my $kernelversion = $self->{'starting_release'};
	my $has_sysdevspaces = $kernelversion =~ /^7\.[23]\./ ? 1 : 0;

	my $mirrored_data = 
	defined $paramall->{'_MIRRORED_DATA'} &&
		$paramall->{'_MIRRORED_DATA'} eq 'YES' ? 1 : 0;

	my $maxsysdev  = 0;
	unless ($has_sysdevspaces == 0) {
		$maxsysdev = $paramall->{'_MAXSYSDEVSPACES'};
		unless (defined $maxsysdev) {
			$self->msg1 (
			"cannot get _MAXSYSDEVSPACES by param_directgetall\n");
			return (-1);
		}
	}

	my $maxlogdev = $paramall->{'MAXARCHIVELOGS'};
	unless (defined $maxlogdev) {
		$self->msg1 (
		"cannot get MAXARCHIVELOGS by param_directgetall\n");
		return (-1);
	}

	my $maxdatadev = $paramall->{'MAXDATADEVSPACES'};
	unless (defined $maxdatadev) {
		$self->msg1 (
		"cannot get MAXDATADEVSPACES by param_directgetall\n");
		return (-1);
	}

	my $logmode = $paramall->{'LOG_MODE'};
	unless (defined $logmode) {
		$self->msg1 ("cannot get LOG_MODE by param_directgetall\n");
		return -1;
	}

	my $rundirectory = $paramall->{'RUNDIRECTORY'};
	unless (defined $rundirectory) {
		$self->msg1 (
		"cannot get RUNDIRECTORY by param_directgetall\n");
		return (-1);
	}

	$self->msg1 ("rundirectory  = ".$rundirectory."\n");
	$self->msg1 ("logmode       = ".$logmode."\n");
	$self->msg1 ("mirrored_data = ".$mirrored_data."\n")
		if ($mirrored_data > 0);
	$self->msg1 ("maxsysdev     = ".$maxsysdev."\n");
	$self->msg1 ("maxdatadev    = ".$maxdatadev."\n");
	$self->msg1 ("maxlogdev     = ".$maxlogdev."\n");

	#
	# build a hash of all volumes
	# fist level key is category
	# second level key is number
	# then array of file name, file type, file size
	#
	my %dev;
	while (my ($key, $val) = each %$paramall) {
		my $top =
		$key =~ /^ARCHIVE_LOG_/     ? 'LOG'      :
		$key =~ /^M_ARCHIVE_LOG_/   ? 'MLOG'     :
		$key =~ /^DATADEV_/         ? 'DATA'     :
		$key =~ /^M_DATADEV_/       ? 'MDATA'    :
		$key =~ /^SYSDEV_/          ? 'SYS'      :
		$key =~ /^M_SYSDEV_/        ? 'MSYS'     :
		$key =~ /^_KERNELTRACEFILE/ ? 'KNLTRACE' :
		undef;

		next unless (defined $top);

		if ($key =~ /^_KERNELTRACEFILE/) {
			my $size = $paramall->{'KERNELTRACESIZE'};
			my $type = 'F';
			my $num = 1;

			my @mydev = ($val, $type, $size); 
			${$dev{$top}}{$num} = \@mydev;
		} else {
			my ($num) = ($key =~ /.+_(\d+)$/);
			$num = int ($num);

			my $sizekey = $key;
			if ($sizekey =~ /^M_/) {
				($sizekey) = ($sizekey =~ /^M_(.+)/); 
			}
			$sizekey =~ s/DEV|_LOG/_SIZE/;

			my $size = 
			$sizekey =~ /^SYS_/ ? undef :
			$paramall->{$sizekey};

			my $typekey = $key;
			$typekey =~ s/DEV|_LOG/_TYPE/;
			my $type = $paramall->{$typekey};

			my @mydev = ($val, $type, $size); 
			${$dev{$top}}{$num} = \@mydev;
		}
	}

	my @catall = ('KNLTRACE', 'SYS', 'DATA', 'LOG');

	# check consitence of mirrors and maximum parameters
	foreach $cat (@catall) {
		my $count = keys (%{$dev{$cat}});
		my $mcount;

		if (($mirrored_data != 0 && $cat =~ /SYS|DATA/) || 
		   ($logmode =~ /DUAL/ && $cat =~ /LOG/)) {
			$mcount = keys (%{$dev{'M'.$cat}});
			if ($mcount != $count) {
				$self->msg1 (
				"number of ".$cat.
				" MIRROR DEVSPACES (".$mcount.") ".
				"does not match number of ".$mcount.
				" DEVSPACES (".$count.")\n"
				);
				return (-1);
			}
			if ($cat eq 'SYS' && $count > $maxsysdev) {
				$self->msg1 (
				"number of SYS DEVSPACES (".$count.
				") is greater then param ".
				"_MAXSYSDEVSPACES (".$maxsysdev.")\n"
				);
				return (-1);
			}
			if ($cat eq 'DATA' && $count > $maxdatadev) {
				$self->msg1 (
				"number of DATA DEVSPACES (".$count.
				") is greater then param ".
				"MAXDATADEVSPACES (".$maxdatadev.")\n"
				);
				return (-1);
			}
			if ($cat eq 'LOG' && $count > $maxlogdev) {
				$self->msg1 (
				"number of LOG DEVSPACES (".$count.
				") is greater then param ".
				"MAXARCHIVELOGS (".$maxlogdev.")\n"
				);
				return (-1);
			}
		}
	}

	unless (defined $self->{instancetypename}) {
		$self->{instancetypename} =
		$self->{instancetype} =~ /LVC/ ? 'liveCache' : 'database';
	}

	# get db process owner
	my $dbowner = $self->getdbprocessowner ();
	unless (defined $dbowner) {
		return (-1);
	}

	my $dbgroup = getgrgid ((getpwnam ($dbowner))[3]);
	$self->msg1	($self->{instancetypename}.
	             " process owner is ".$dbowner.":".$dbgroup."\n");

	# check file access for each volume
	my $rc = 0;
	foreach $cat (@catall) {
		 for (my $i = 1; defined $dev{$cat}{$i}; $i++) {
		 	my @mydev = @{$dev{$cat}{$i}};
			$rc |= $self->getvolaccess
			($cat, $i, $mydev[0], $rundirectory, $dbowner);

			next unless (
				($mirrored_data != 0 && $cat =~ /SYS/) ||
				($mirrored_data != 0 && $cat =~ /DATA/) ||
				($logmode =~ /DUAL/ && $cat =~ /LOG/)
			);

		 	my @mydev = @{$dev{'M'.$cat}{$i}};
			$rc |= $self->getvolaccess
			($cat, $i, $mydev[0], $rundirectory, $dbowner);
		}
	}

	return $rc;
}

sub getvolaccess {
	my ($self, $cat, $num, $filename, $rundirectory, $dbowner) = @_;

	my $mode = 0666;
	my $euid = (getpwnam ($dbowner))[2];
	my $egid = (getpwnam ($dbowner))[3];

	$cat .= "     " if ($cat =~ /SYS|LOG/);
	$cat .= "    " if ($cat =~ /DATA/);
	my $txt = $cat." ".$num." ".$filename." ";

	$filename = $rundirectory.'/'.$filename unless ($filename =~ /^\/.+/);

	my (@info) = stat ($filename);
	if ($#info == -1) {
		# stat failed, probably no such file or directory
		$txt .= $!;
		$self->msg1 ($txt."\n");
		return (-1);
	}

	my $fmode = $info[2] & 0777;
	my $fuid = $info[4];
	my $fgid = $info[5];

	$txt .= getpwuid ($fuid).':';
	$txt .=	getgrgid ($fgid).' ';
	$txt .= sprintf ('%03o', $fmode).' ';

	if ($fuid == $euid) {
		my $umode = $mode & 0700;
		if (($umode & $fmode) == $umode) {
			$txt .= "can be accessed as owner ".$dbowner;
			$self->msg1 ($txt."\n");
			return 0;
		} else {
			$txt .= "cannot be accessed as owner ".$dbowner;
			$self->msg1 ($txt."\n");
			return -1;
		}
	} 

	my $gmode = $mode & 0070;
	if ($fgid == $egid) {
		if (($gmode & $fmode) == $gmode) {
			$txt .= "can be accessed by primary group ";
			$txt .= getgrgid ($egid);
			$self->msg1 ($txt."\n");
			return 0;
		} else {
			$txt .= "cannot be accessed by primary group ";
			$txt .= getgrgid ($egid);
			$self->msg1 ($txt."\n");
			return -1;
		}
	}

	my @secondaries = split (' ', (getgrgid ($fgid))[3]);
	foreach my $uname (@secondaries) {
		if (($uname eq $dbowner) && (($gmode & $fmode) == $gmode)) {
			$txt .= "can be accessed by secondary group ";
			$txt .= getgrgid ($fgid);
			$self->msg1 ($txt."\n");
			return 0;
		}
	}

	my $omode = $mode & 0007;
	if (($omode & $fmode) == $omode) {
		$txt .= "can be accessed as other";
		$self->msg1 ($txt."\n");
		return 0;
	}

	$txt .= "cannot be accessed";
	$self->msg1 ($txt."\n");
	return (-1);
}

1;
