#!/usr/bin/perl
#
# $Header: //sapdb/V75/c_00/b_07/sys/src/install/perl/SAPDB/Install/Instance/Check/Finalize/ExternalBackup.pm#1 $
# $DateTime: 2003/11/20 17:48:59 $
# $Change: 57359 $
#

package SAPDB::Install::Instance::Check::Finalize::ExternalBackup;

sub BEGIN {
	@ISA = ('SAPDB::Install::Exporter');
	@EXPORT = ();
	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages = (
		'Instance::Base',
		'Instance::Check::Common',
		'Instance::Check::DomainUser'
	);

	foreach my $package (@neededPackages) {
	  	unless (defined $repo->Eval 
		("SAPDB::Install::$package", 1.01)) {
                	print join ("\n", $repo->GetErr)."\n";
                	die;
        	}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	  } 
}

push @ISA, 
	'SAPDB::Install::Instance::Base',
	'SAPDB::Install::Instance::Check::Common',
	'SAPDB::Install::Instance::Check::DomainUser';

#
# check for migration backup
# when using migration strategy
# EXTERNAL_CONSISTENT_BACKUP
#
sub finalizecheck {
	my ($self) = @_; 

	$self->set_errorstate ('ERROR');
	$self->msgbegin
	("running finalize check for EXTERNAL_CONSISTENT_BACKUP");
	$self->msg1
	("running finalize check for EXTERNAL_CONSISTENT_BACKUP...\n");

	my $rc = $self->is_instance_recovered ();
	unless (defined $rc) {
		#
		# somthing went completely wrong during finalize check
		#
		$self->msgend ();
		return undef;
	}

	my $state = $self->get_errorstate ();
	if ($state eq 'INIT_FOR_RECOVERY_REQUIRED') {
		#
		# we have to initialize database before
		# restoring backup
		#
		$self->msgend ();
		return 0;
	}

	if ($state eq 'OK') {
		#
		# restore of migration backup was successful
		# check if domainuser is known by dbmsrv
		#
		$rc = $self->domainuser ();
		unless (defined $rc && $rc == 0) {
			$self->set_errorstate ('ERROR');
			$self->msgend ();
			return undef;
		}
	
		#
		# instance is ready for finalization
		#
		$self->set_errorstate ('OK');
		$self->msgend ();
		return 0;
	}

	#
	# unexpectect error state
	#
	$self->set_errorstate ('ERROR');
	$self->msgend ();
	return undef;
}

#
# is_instance_recovered ()
#
sub is_instance_recovered {
	my ($self) = @_;
	my $dbm = $self->{'dbm'};

	$self->msg0 ("looking for recovered instance\n");

	#
	# first find migration backup
	#
	unless (defined ($dbm->backup_history_open ())) {
		$self->msg1 ("cannot open backup history\n");
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	my $history = $dbm->backup_history_list
	('-r LAST -c KEY,LABEL,ACTION,STAMP1,LOG,RC');
	
	unless (defined $history) {
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	unless (defined ($dbm->backup_history_close ())) {
		$self->msg1 ("cannot close backup history\n");
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	$self->msg1 ("looking for migration backup\n");

	my $history_lines = $#{$history->{'LABEL'}} + 1;
	if ($history_lines <= 0) {
		$self->msg1 ("empty backup history\n");
		$self->msg1 ("no migration backup found\n");
		$self->set_errorstate ('ERROR');
		return undef;
	}

	$self->msg1 ("\n");
	$self->msg1 
	("  KEY          LABEL     ACTION    STAMP1             LOG  RC\n");
	for (my $i = 0; $i < $history_lines; $i++) {
		my $txt = '  ';
		$txt .= $history->{'KEY'}->[$i]." ";
		$txt .= $history->{'LABEL'}->[$i]." ";
		$txt .= $history->{'ACTION'}->[$i]." ";
		$txt .= $history->{'STAMP1'}->[$i]." ";
		$txt .= $history->{'LOG'}->[$i]." ";
		$txt .= $history->{'RC'}->[$i]." ";
		$self->msg1 ($txt."\n");
	}
	$self->msg1 ("\n");

	if ($history_lines != 1) {
		$self->msg1 ("more than one backup needed for recovery\n");
		$self->msg1 ("no migration backup found\n");
		$self->set_errorstate ('ERROR');
		return undef;
	}

	unless ($history->{'ACTION'}->[0] =~ /^SAVE COLD/) {
		$self->msg1 ("migration backup must be done in cold mode\n");
		$self->msg1 ("no migration backup found\n");
		$self->set_errorstate ('ERROR');
		return undef;
	}

	unless ($history->{'LABEL'}->[0] =~ /^DAT/) {
		$self->msg1 ("migration backup must be a data backup\n");
		$self->msg1 ("no migration backup found\n");
		$self->set_errorstate ('ERROR');
		return undef;
	}

	unless ($history->{'LOG'}->[0] =~ /^NO/) {
		$self->msg1 ("migration backup must be consistent\n");
		$self->msg1 ("no migration backup found\n");
		$self->set_errorstate ('ERROR');
		return undef;
	}

	unless ($history->{'RC'}->[0] == 0) {
		$self->msg1 ("migration backup must be successfull\n");
		$self->msg1 ("no migration backup found\n");
		$self->set_errorstate ('ERROR');
		return undef;
	}

	#
	# keep backup label and backup timestamp of migration backup
	#
	my $backup_label = $history->{'LABEL'}->[0];
	my $backup_stamp = $history->{'STAMP1'}->[0];

	#
	# then find latest recovery
	#
	$self->msg1 ("looking for latest restore\n");

	unless (defined ($dbm->backup_history_open ())) {
		$self->msg1 ("cannot open backup history\n");
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	$history = $dbm->backup_history_list
	('-a RESTORE -c KEY,LABEL,ACTION,STAMP1,LOG,RC');
	
	unless (defined $history) {
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	unless (defined ($dbm->backup_history_close ())) {
		$self->msg1 ("cannot close backup history\n");
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->set_errorstate ('ERROR');
		return undef;
	}

	$history_lines = $#{$history->{'LABEL'}} + 1;
	if ($history_lines <= 0) {
		$self->msg1 ("empty backup history\n");
		$self->msg1 ("no restore found\n");
		$self->set_errorstate ('ERROR');
		return 0;
	}

	#
	# remove any other action then 'RESTORE' from hash
	# output is polluted with 'HISTLOST'
	#
	my @colnames = ('KEY', 'LABEL', 'ACTION', 'STAMP1', 'LOG', 'RC');
	for (my $i = 0; $i < $history_lines; ) {
		if ($history->{'ACTION'}->[$i] =~ /RESTORE/) {
			$i++;
			next;
		}
		foreach my $col (@colnames) {
			splice @{$history->{$col}}, $i, 1;
		}
		$history_lines--;
	}

	#
	# make sure that at least one restore is kept
	#
	if ($history_lines <= 0) {
		$self->msg1 ("no restore found after cleanup\n");
		$self->set_errorstate ('INIT_FOR_RECOVERY_REQUIRED');
		return 0;
	}

	#
	# take care only of latest restore
	#
	for (; $history_lines > 1; $history_lines--) {
		foreach my $col (@colnames) {
			shift @{$history->{$col}};
		}
	}

	#
	# make sure that we got one restore
	#
	if ($history_lines != 1) {
		$self->msg1 ("no restore of migration backup found\n");
		$self->set_errorstate ('INIT_FOR_RECOVERY_REQUIRED');
		return 0;
	}

	#
	# print latest restore
	#
	$self->msg1 ("\n");
	$self->msg1 
	("  KEY          LABEL     ACTION    STAMP1             LOG  RC\n");
	for (my $i = 0; $i < $history_lines; $i++) {
		my $txt = '  ';
		$txt .= $history->{'KEY'}->[$i]." ";
		$txt .= $history->{'LABEL'}->[$i]." ";
		$txt .= $history->{'ACTION'}->[$i]." ";
		$txt .= $history->{'STAMP1'}->[$i]." ";
		$txt .= $history->{'LOG'}->[$i]." ";
		$txt .= $history->{'RC'}->[$i]." ";
		$self->msg1 ($txt."\n");
	}
	$self->msg1 ("\n");

	unless ($history->{'LABEL'}->[0] =~ /^DAT/) {
		$self->msg1 ("latest restore was no complete data bakup\n");
		$self->set_errorstate ('INIT_FOR_RECOVERY_REQUIRED');
		return 0;
	}

	unless ($history->{'RC'}->[0] == 0) {
		$self->msg1 ("latest restore was not successful\n");
		$self->set_errorstate ('INIT_FOR_RECOVERY_REQUIRED');
		return 0;
	}

	unless ($history->{'LOG'}->[0] =~ /^NO/) {
		$self->msg1 ("latest restore was not consistent\n");
		$self->set_errorstate ('INIT_FOR_RECOVERY_REQUIRED');
		return 0;
	}

	#
	# make sure that this restore matches with migration backup 
	#
	unless ($history->{'LABEL'}->[0] eq $backup_label &&
	        $history->{'STAMP1'}->[0] eq $backup_stamp) {
		$self->msg1
		("latest restore does not match with migration backup\n");
		$self->set_errorstate ('INIT_FOR_RECOVERY_REQUIRED');
		return 0;
	}

	$self->msg1 ("latest restore matches with migration backup\n");

	#
	# find command id of latest recovery
	#	
	my $cmdid = $history->{'KEY'}->[0];
	$self->msg1 ("looking in dbmutl for cmdid ".$cmdid."\n");

	my $dbmutl = $self->get_dbmutl ($cmdid);
	unless (defined $dbmutl) {
		$self->set_errorstate ('ERROR');
		return undef;
	}

	my $dbmutl_lines = $#{$dbmutl->{'TEXT'}} + 1;
	if ($dbmutl_lines <= 0) {
		$self->msg1 ("cmdid not found in dbmutl\n");
		$self->set_errorstate ('ERROR');
		return undef;
	}

	my @utlcmd = ();
	for (my $i = 0; $i < $dbmutl_lines; $i++) {
		push @utlcmd, $dbmutl->{'TEXT'}->[$i];
	}

	$self->msg1 ("\n");
	foreach (@utlcmd) {
		$self->msg1 ("  ".$_."\n");
	}
	$self->msg1 ("\n");

	$self->msg1 ("migration backup was sucessfully restored\n");
	$self->set_errorstate ('OK');
	return 0;
}

1;
