#!/usr/bin/perl
#
# $Header: //sapdb/V74/develop/sys/src/install/perl/SAPDB/Install/Instance/Check/DataCons/ConData.pm#5 $
# $DateTime: 2002/02/27 11:03:24 $
# $Change: 17183 $
#

package SAPDB::Install::Instance::Check::DataCons::ConData;

sub BEGIN {
	@ISA = ('SAPDB::Install::Exporter');
	@EXPORT = ();
	my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages = (
		'Instance::Base',
		'Instance::Check::Common'
	);

	foreach my $package (@neededPackages) {
	  	unless (defined $repo->Eval
		("SAPDB::Install::$package", 1.01)) {
			print join ("\n", $repo->GetErr)."\n";
			die;
		}

		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	}
}

push @ISA, 
	'SAPDB::Install::Instance::Base',
	'SAPDB::Install::Instance::Check::Common';

#
# datacons ()
# check if data are consistent for migration
# if instance type is 'OLTP' instance must be restartable without log
#
sub datacons {
	my $rc;
	my ($self) = @_;
	my $dbm = $self->{dbm};
	
	my $instance_type = $self->{'instancetype'};
	unless (defined $instance_type &&
	$instance_type =~ /^OLTP/) {
		$self->msgend ();
		return -1;
	}

	my $migration_strategy = $self->{'migration_strategy'};
	unless (defined $migration_strategy &&
	$migration_strategy =~ /^CONSISTENT_DATA/) {
		$self->msgend ();
		return -1;
	}

	$self->msg1 ("looking for backup log position\n");

	my $restartinfo = $self->get_cold_restartinfo ();
	my $currpos = $self->get_logpos_from_restartinfo ($restartinfo);

	my $backup_logpos = $self->{'backup_logpos'};
	unless (defined $backup_logpos) {
		$backup_logpos = $currpos;
	}

	my $logpos = $self->{'logpos'};
	if (defined $logpos) {
		$self->msg1 ("last known log position ".$logpos."\n");
	}

	if (defined $logpos && $logpos != $currpos) {
		$self->msg1 ("last known log position is out of date\n");
		$backup_logpos = $currpos;
	}

	$self->msg1 ("backup log position ".$backup_logpos."\n");

	my $is_consistent = $self->is_consistent ($restartinfo);
	unless (defined $is_consistent && $is_consistent == 1) {
		$self->msgend ();
		return -1;
	}

	my $is_uptodate = $self->is_backup_uptodate ($backup_logpos, $restartinfo);
	unless (defined $is_uptodate && $is_uptodate == 1) {
		$self->msgend ();
		return -1;
	}

	# store backup logpos persistently
	$self->{'backup_logpos'} = $backup_logpos;

	$rc = $self->online_checks ();
	unless (defined $rc && $rc == 0) {
		$self->msgend ();
		return -1;
	}

	$rc = $self->prepare_migration ();
	unless (defined $rc && $rc == 0) {
		$self->msgend ();
		return -1;
	}

	$self->set_errorstate ('OK');
	return 0;
}

sub prepare_migration {
	my ($self) = @_;
	my $dbm = $self->{dbm};

	#
	# switch instance to cold mode to prepare migration
	#
	my $state = $dbm->db_state ();
	unless (defined $state) {
		$self->msg1 ($dbm->lastdialog ());
		return -1;
	}

	if ($state =~ /COLD|ADMIN/) {
		$self->msg1 ($self->{instancetypename}.
		" state is ".$state."\n");
	} else {
		$self->msg0 ($self->{instancetypename}.
		" state is ".$state."\n");
	}

	#
	# shutdown database if we are still online
	#
	if ($state =~ /WARM|ONLINE/) {
		$state =
		$self->switchto ('OFFLINE', "switch database state to OFFLINE\n");
	}

	#
	# we should switch instance offline from unknown mode
	#
	unless ($state =~ /COLD|ADMIN|OFFLINE/) {
		$state = $self->switchto ('CLEAR', "clear database state\n");
		unless ($state eq 'OFFLINE') {
			return -1;
		}
	}

	#
	# if the instance is not already cold,
	# we have to bring it into cold mode
	#
	my $prev_state;
	if ($state =~ /COLD|ADMIN/) {
		$prev_state = $state;
	} else {
		$prev_state = 'OFFLINE';

		if ($self->{'can_db_admin'}) {
			$state =  
			$self->switchto ('ADMIN', "switch database state to ADMIN\n"); 
		} else {
			$state =  
			$self->switchto ('COLD', "switch database state to COLD\n"); 
		}

		unless ($state =~ /COLD|ADMIN/) {
			$self->get_errmsg_from_knldiag ();
			return -1;
		}
	}
	
	#
	# we are cold, so we can measure current log pos
	#
	my $restartinfo;
	$restartinfo = $self->get_restartinfo ();
	$self->{'logpos'} = $self->get_logpos_from_restartinfo ($restartinfo);

	$self->msg0 ("prepare database migration\n");
	my $msg = $dbm->util_execute ('migrate');
	unless (defined $msg && $msg =~ /^OK/) {
		$self->msg1 ($dbm->lastdialog ());
		$self->get_errmsg_from_knldiag ();
		$state = $self->switchto_prev ($state, $prev_state);
		return -1;
	}

	#
	# we are cold, so we can measure current log pos
	#
	$restartinfo = $self->get_restartinfo ();
	$self->{'logpos'} = $self->get_logpos_from_restartinfo ($restartinfo);

	$state = $self->switchto_prev ($state, $prev_state);
	unless ($state eq $prev_state) {
		return -1;
	}

	return 0;
}

sub online_checks {
	my ($self) = @_;
	my $dbm = $self->{dbm};

	$self->msg1 ("checking free space, bad indicies and bad volumes\n");

	#
	# switch instance to online mode to prepare migration
	#
	my $state = $dbm->db_state ();
	unless (defined $state) {
		$self->msg1 ($dbm->lastdialog ());
		return -1;
	}

	if ($state =~ /WARM|ONLINE/) {
		$self->msg1 ($self->{instancetypename}.
		" state is ".$state."\n");
	} else {
		$self->msg0 ($self->{instancetypename}.
		" state is ".$state."\n");
	}

	#
	# stop database if we are still admin
	#
	if ($state =~ /COLD|ADMIN/) {
		$state =
		$self->switchto ('OFFLINE', "switch database state to OFFLINE\n");
	}

	#
	# we should switch instance offline from unknown mode
	#
	unless ($state =~ /WARM|ONLINE|OFFLINE/) {
		$state = $self->switchto ('CLEAR', "clear database state\n");
		unless ($state eq 'OFFLINE') {
			return -1;
		}
	}

	#
	# if the instance is not already online,
	# we have to bring it into online mode
	#
	my $prev_state;
	if ($state =~ /WARM|ONLINE/) {
		$prev_state = $state;
	} else {
		$prev_state = 'OFFLINE';

		if ($self->{'can_db_online'}) {
			$state =
			$self->switchto ('ONLINE', "switch database state to ONLINE\n"); 
		} else {
			$state =
			$self->switchto ('WARM', "switch database state to WARM\n"); 
		}

		unless ($state =~ /WARM|ONLINE/) {
			$self->get_errmsg_from_knldiag ();
			return -1;
		}

		#
		# we cannot measure logpos exactly,
		# but in all cases we know 2 pages a needed for restart
		#
		$self->{'logpos'} = $self->{'logpos'} + 2;
	}
	
	my ($cmd, $sql);

	$cmd =
	'select * from sysdd.server_db_state where '.
	'upper(description) = \'BAD DEVSPACE\'';

	$sql = $dbm->sql_execute ($cmd);
	unless (defined $sql) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot read sysdd.server_db_state\n");
		return -1;
	}

	if (defined $sql->{error} && ${$sql->{error}}[0] != 100) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot read sysdd.server_db_state\n");
		return -1;
	}

	unless (defined $sql->{error} && ${$sql->{error}}[0] == 100) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("found bad volumes\n");
		return -1;
	}

	$self->msg0 ("found no bad volume\n");
	
	$cmd =
	'select * from sysdd.bad_indexes';

	$sql = $dbm->sql_execute ($cmd);
	unless (defined $sql) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot read sysdd.bad_indexes\n");
		return -1;
	}

	if (defined $sql->{error} && ${$sql->{error}}[0] != 100) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot read sysdd.bad_indexes\n");
		return -1;
	}

	unless (defined $sql->{error} && ${$sql->{error}}[0] == 100) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("found bad indicies\n");
		return -1;
	}

	$self->msg0 ("found no bad index\n");

	$cmd =
	'select unused_pages from sysdd.serverdb_stats';

	$sql = $dbm->sql_execute ($cmd);
	unless (defined $sql) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot read sysdd.serverdb_stats\n");
		return -1;
	}

	if (defined $sql->{error}) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot read sysdd.serverdb_stats\n");
		return -1;
	}

	my ($free_space) = (${$sql->{result}}[0] =~ /(\d+)/);
	unless (defined $free_space) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot get database size\n");
		return -1;
	}

	$cmd =
	'select value from sysdd.xparameters '.
	'where description = \'MAXDATAPAGES\'';

	$sql = $dbm->sql_execute ($cmd);
	unless (defined $sql) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot read sysdd.xparameters\n");
		return -1;
	}

	if (defined $sql->{error}) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot read MAXDATAPAGES from sysdd.xparameters\n");
		return -1;
	}

	my ($max_datapages) = (${$sql->{result}}[0] =~ /(\d+)/);
	unless (defined $max_datapages) {
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("cannot get max. database size\n");
		return -1;
	}

	my $needed_space = int (($max_datapages + 1854) / 1855);
	$needed_space += int (($needed_space + 9) / 10);

	if ($needed_space > $free_space) {
		$self->msg1 ($dbm->lastdialog ());
		$self->msg1 ("needed database size    ".$needed_space." pages\n");
		$self->msg1 ("free database size      ".$free_space." pages\n");
		$self->msg1 ("not enough space for migration of system devspace\n");
		return -1;
	}

	$self->msg1 ("needed database size    ".$needed_space." pages\n");
	$self->msg1 ("free database size      ".$free_space." pages\n");
	$self->msg0 ("found enough space for migration of system devspace\n");

	$state = $self->switchto_prev ($state, $prev_state);
	unless ($state eq $prev_state) {
		return -1;
	}

	return 0;
}

sub get_cold_restartinfo {
	my ($self) = @_;
	my $dbm = $self->{dbm};

	$self->msg1 ("looking for restart info in cold mode\n");
	
	#
	# switch instance to cold mode to get 'db_restartinfo'
	#
	my $state = $dbm->db_state ();
	unless (defined $state) {
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	if ($state =~ /COLD|ADMIN/) {
		$self->msg1 ($self->{instancetypename}.
		" state is ".$state."\n");
	} else {
		$self->msg0 ($self->{instancetypename}.
		" state is ".$state."\n");
	}

	#
	# if instance is still online backup is out of date, because
	# the current log page is not contained in last backup
	#
	if ($state =~ /WARM|ONLINE/) {
		$self->msg1 ("backup is out of date\n");
		return undef;
	}

	#
	# we should switch instance offline from unknown mode
	#
	unless ($state =~ /WARM|ONLINE|COLD|ADMIN|OFFLINE/) {
		$state = $self->switchto ('CLEAR', "clear database state\n");
		unless ($state eq 'OFFLINE') {
			return undef;
		}
	}

	#
	# if the instance is not already cold,
	# we have to bring it into cold mode
	#
	my $prev_state;
	if ($state =~ /COLD|ADMIN/) {
		$prev_state = $state;
	} else {
		$prev_state = 'OFFLINE';

		if ($self->{'can_db_admin'}) {
			$state =  
			$self->switchto ('ADMIN', "switch database state to ADMIN\n"); 
		} else {
			$state =  
			$self->switchto ('COLD', "switch database state to COLD\n"); 
		}

		unless ($state =~ /COLD|ADMIN/) {
			$self->get_errmsg_from_knldiag ();
			return undef;
		}
	}

	my $restartinfo = $self->get_restartinfo ();
	
	$state = $self->switchto_prev ($state, $prev_state);
	unless ($state eq $prev_state) {
		return ndef;
	}

	return ($restartinfo);
}

sub is_consistent {
	my ($self, $restartinfo) = @_;

	return undef unless (defined $restartinfo);
	return undef unless (defined ($restartinfo->{'Consistent'}));
	return undef unless (defined ($restartinfo->{'Restartable'}));

	if (($restartinfo->{'Consistent'} &&
	     $restartinfo->{'Restartable'}) == 0) {
		$self->msg1 ("database was stoped without checkpoint\n");
		return 0;
	}

	$self->msg1 ("database was stoped with checkpoint\n");
	return 1;
}

sub is_backup_uptodate {
	my ($self, $backup_logpos, $restartinfo) = @_;

	return undef unless (defined $backup_logpos);
	return undef unless (defined $restartinfo);
	
	#
	# look for backup in backup history
	# and get utility cmdid for this backup
	#
	my $dbm = $self->{'dbm'};
	unless (defined ($dbm->backup_history_open ())) {
		$self->msg1 ("cannot open backup history\n");
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	my $history = 
	$dbm->backup_history_list
	('-c KEY,LABEL,ACTION,LOG,FIRSTLOG,LASTLOG,START,STOP,RC');

	unless (defined $history) {
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	unless (defined ($dbm->backup_history_close ())) {
		$self->msg1 ("cannot close backup history\n");
		$self->msg1 ("\n");
		$self->msg1 ($dbm->lastdialog ());
		return undef;
	}

	$self->msg1 ("check ability to recover in backup history\n");

	my $history_lines = $#{$history->{'KEY'}} + 1;
	if ($history_lines <= 0) {
		$self->msg1 ("empty backup history\n");
		$self->msg1 ("found no backup\n");
		return undef;
	}

	$self->msg1 ("\n");
	$self->msg1
	("  KEY          LABEL     ACTION    LOG FIRSTLOG   LASTLOG    RC\n");

	for (my $i = $history_lines - 1; $i >= 0; $i--) {
		my $txt = '';
		$txt .= sprintf ("%-12s ", $history->{'KEY'}->[$i]);
		$txt .= sprintf ("%-9s ", $history->{'LABEL'}->[$i]);
		$txt .= sprintf ("%-9s ", $history->{'ACTION'}->[$i]);
		$txt .= sprintf ("%-3s ", $history->{'LOG'}->[$i]);
		$txt .= sprintf ("%-10s ", $history->{'FIRSTLOG'}->[$i]);
		$txt .= sprintf ("%-10s ", $history->{'LASTLOG'}->[$i]);
		$txt .= sprintf ("%-6s", $history->{'RC'}->[$i]);
		$self->msg1 ('  '.$txt."\n");

		if ($history->{'ACTION'}->[$i] =~ /^HISTLOST/) {
			last;
		}

		if ($history->{'LABEL'}->[$i] =~ /^DAT/ &&
		    $history->{'ACTION'}->[$i] =~ /^SAVE/ &&
		    $history->{'RC'}->[$i] == 0) {
			last;
		}
	}
	$self->msg1 ("\n");

	my $idx_histlost = -1;
	my $idx_full = -1;
	my $idx_data = -1;
	my $idx_log = -1;

	for (my $i = $history_lines - 1; $i >= 0; $i--) {
		if (
		$history->{LABEL}->[$i] =~ /^LOG/ &&
		$history->{LASTLOG}->[$i] == $backup_logpos &&
		$history->{RC}->[$i] == 0 &&
		$history->{ACTION}->[$i] =~ /^SAVE COLD/) {
			$idx_log = $i;
			next;
		}

		if 
		($history->{LABEL}->[$i] =~ /^PAG/ &&
		$history->{FIRSTLOG}->[$i] == $backup_logpos &&
		$history->{RC}->[$i] == 0 &&
		$history->{ACTION}->[$i] =~ /^SAVE COLD/) {
			$idx_data = $i;
			next;
		}

		if 
		($history->{LABEL}->[$i] =~ /^DAT/ &&
		$history->{RC}->[$i] == 0 &&
		$history->{FIRSTLOG}->[$i] == $backup_logpos &&
		$history->{ACTION}->[$i] =~ /^SAVE COLD/) {
			$idx_data = $i;
			$idx_full = $i;
			last;
		}

		if 
		($history->{LABEL}->[$i] =~ /^DAT/ &&
		$history->{RC}->[$i] == 0 &&
		$history->{ACTION}->[$i] =~ /^SAVE/) {
			$idx_full = $i;
			last;
		}

		if ($history->{ACTION}->[$i] =~ /HISTLOST/) {
			$idx_histlost = $i;
			last;
		}
	}
	
	if ($idx_data == -1) {
		$self->msg1 ("no up to date data backup found\n");
		return undef;
	}

	if ($idx_histlost > $idx_data) {
		$self->msg1 ("backup history is broken for data backup\n");
		return undef;
	}

	$self->msg1 (
	"data backup ". $history->{LABEL}->[$idx_data]." is up to date\n");

	if ($idx_data == $idx_full) {
		$self->msg1 ("no log backup needed\n");
	} else {
		if ($idx_log == -1) {
			$self->msg1 ("no up to date log backup found\n");
			return undef;
		}

		if ($idx_histlost > $idx_log) {
			$self->msg1
			("backup history is broken for log backup\n");
			return undef;
		}

		$self->msg1 ("log backup ".
		$history->{LABEL}->[$idx_log]." is up to date\n");
	}

	if ($idx_full == -1) {
		$self->msg1 ("complete data backup required\n");
		return undef;
	}

	my $their = {};
	($their->{year}, $their->{mon}, $their->{day}) =
	($history->{STOP}->[$idx_full] =~ /(\d+)-(\d+)-(\d+)\s+/);

	unless 
	(defined $their->{year} &&
	 defined $their->{mon} &&
	 defined $their->{day}) {
		$self->msg1 ("cannot read date of complete data backup\n");
		return undef;
	}

	my $their_date =
	10000 * $their->{year} + 100 * $their->{mon} + $their->{day};

	$self->msg1 (
	"complete data backup ". $history->{LABEL}->[$idx_full].
	" was made on ".$their_date."\n");

	my $our = {};
	($our->{sec}, $our->{min}, $our->{hour},
	 $our->{day}, $our->{mon}, $our->{year}) =
	 localtime (time () - 7 * 24 * 3600);

	#
	# corrections of localtime output
	#
	$our->{year} += 1900;
	$our->{mon} ++;

	my $our_date =
	10000 * $our->{year} + 100 * $our->{mon} + $our->{day};

	$self->msg1 
	("complete data backup should be made after ".$our_date."\n");

	if ($their_date < $our_date) {
		$self->msg1 ("complete data backup is older then expected\n");
		return undef;
	}

	return (1);
}

#
# get restartinfo and look for last used log page number
# we have to be in cold state to get this
#
sub get_logpos_from_restartinfo {
	my ($self, $restartinfo) = @_;

	my $pos = $restartinfo->{'Used LOG Page'};
	$pos = -1 unless (defined $pos);

	$self->msg1 ("current log position ".$pos."\n");
	return ($pos);
}

1;
