#!/usr/bin/perl
#
# $Header: //sapdb/V75/c_00/b_07/sys/src/install/perl/SAPDB/Install/InstallRegistry/Verify.pm#1 $
# $DateTime: 2003/11/20 17:48:59 $
# $Change: 57359 $
#
# Desc: check consistence of data inside InstallRegistry


package SAPDB::Install::InstallRegistry::Verify;

$VERSION = 1.01;

sub BEGIN {
      my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages=(
		'System',
		'Registry',
		'Collector',
		'StdIO',
		'InstallRegistry',
		'Diagnose::InstallRegistry',
		'BuildInfo',
		'Version',
		'Values',
		'MD5Sum',
		'SetDebug',
		'Log',
		'Tools',
		'Cwd',
		'Getopt::Long',
		'Misc',
		'Cleaner',
		'SigHandler',
		'Trace'
	);
	@EXPORT=('main');
	foreach my $package (@neededPackages){
	  	unless (defined $repo->Eval ("SAPDB::Install::$package", 1.01)) {
                print join ("\n", $repo->GetErr)."\n";
                die;
        	}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package"); 
	}
}


$DEBUG=0;


sub checkVersion{
	my ($refData,$package_name,$package_path) = @_;
	my $rc = 1;
	my %RegistryData = %$refData;
	my %packdata = %{${%{$RegistryData{$package_name}}}{$package_path}};
	unless($packdata{'TestFile'} =~ /\S/){
		print2stdout("\npackage has no testfile\n");
		return 1;
	}
	my $file = $package_path.'/'.$packdata{'TestFile'};
	-f $file or print2stdout("\ntestfile \"$file\" not found\n") and return 0;
	my @build = GetBuild($file);
	@build or print2stdout("\ncannot get build info\n") and return 0;
	unless(release2num($packdata{'Version'}) == release2num(join('.',@build))){
		print2stdout("\nregistered Version: $packdata{Version}\n");
		print2stdout("\nreal Version: ".join('.',@build)."\n");
		$rc = 0;
	}
	return $rc;
}



sub checkDependencies{
	my ($refData,$package_name,$package_path) = @_;
	my %RegistryData = %$refData;
	my %packdata = %{${%{$RegistryData{$package_name}}}{$package_path}};
	my $rc = 1;
	foreach my $req_string (@{$packdata{'Require'}}){
		my ($req) = SAPDB::Install::Collector::parseRequire($req_string);
 		my %req = %$req;
		my $valid_found = 0;
		foreach my $test_path (keys(%{$RegistryData{$req{'name'}}})){
			my %test_data = %{${%{$RegistryData{$req{'name'}}}}{$test_path}};
			$test_data{'Valid'} or next;
			if(checkRequired('package_name' => $req{'name'}, 'version' => $test_data{'Version'},
					'mode' => $test_data{'Mode'}, 'required' => $req)){
				$valid_found = 1;	
				last;
			}
		}
		unless($valid_found){
			my $mode_string = $req{'bit'} =~ /32|64/ ? $req{'bit'}.' bit' : '';
			print2stdout("\nunresolved dependency: need $req{name} $req{operator} $req{release} $mode_string\n");
			$rc = 0;
		}
	}
	
	
	if($packdata{'IsSubPackage'}){
		my ($parent_name,$parent_path) = @{$packdata{'ParentPackage'}};
		if(exists ${%{$RegistryData{$parent_name}}}{$parent_path}){
			my %parent_data = %{${%{$RegistryData{$parent_name}}}{$parent_path}};
			unless($parent_data{'Valid'}){
				print2stdout("\nparent package $parent_name in $parent_path is not valid\n");
				$rc = 0;
			}
						
			my ($req) = SAPDB::Install::Collector::parseRequire (${@{$packdata{'Require'}}}[0]);
			
			unless(checkRequired ('required' => $req,'package_name' => $parent_name, 
					'version' => $parent_data{'Version'}, 'mode' => $parent_data{'Mode'})){
				print2stdout("\nparent package description dont match with required\n");
				$rc = 0;
			} 
		}else{
			print2stdout("\nparent package $parent_name in $parent_path not found\n");
			$rc = 0;
		}
	}
	
	return $rc;
}


sub checkPackage{
	local (*package,$install_mode) = @_;
	
	#
	# new argument $install_mode: true means its called 
	#							  as part of installer and 
	#							  not as part of verify tool
	#							  - there is no dependency check needed
	#							  - no headlines required
	#
	#
	
	
	my %returnhash;
	unless($install_mode){
		my $line = $package->DispName.' '.$package->Version.' '.
				($package->Mode =~ /\S/ ? $package->Mode.' bit ' : '').
				'in '.$package->Path;
		print2stdout("\n\n$line\n");	
		print2stdout(('-' x length($line))."\n\n");
	}
	print2stdout("check files... ");
	my %sums = %{$package->FileList};	
	my @msgbuf = ();
	my $count_all_files = 0;
	my $count_missed_files = 0;
	my $count_modified_files = 0;
	$count_perm_changed_files = 0;
	my $status = 1;
	foreach my $file (keys(%sums)){
		$count_all_files++;
		my $path = $package->Path.'/'.$file;
		unless(-f $path){
			push @msgbuf,"$path not found\n";
			$count_missed_files++;
			next;
		}
		if(MD5Sum($path) eq $sums{$file}){
			$package->Registry->Log->SetMsg("$path is ok\n");
		}
		else{
			$count_modified_files++;
			push @msgbuf,"$path was modified\n"; 
		}
		if($^O !~ /mswin/i and defined $package->FileInfos){
			my @statbuf = stat($path) or print2stderr("cannot stat file $path: $!\n") and next;
			ref($package->FileInfos) eq 'HASH' or next;
			my %fileinfos = %{$package->FileInfos};
			my $key = $sums{$file};
			exists $fileinfos{$key} or next;
			my %info = %{$fileinfos{$key}} or next;
			my $perm_ok = 1;
			unless($info{'mode'} == $statbuf[2]){
				$perm_ok = 0;
				push @msgbuf,"mode of $path was modified\n"; 
			}
			unless($info{'uid'} == $statbuf[4]){
				$perm_ok = 0;
				push @msgbuf,"owner of $path was modified\n"; 
			}
			unless($info{'gid'} == $statbuf[5]){
				$perm_ok = 0;
				push @msgbuf,"group of $path was modified\n"; 
			}
			$count_perm_changed_files++ unless($perm_ok);
		}

	}
	unless($#msgbuf == -1){
		print2stdout("failed\n");
		foreach my $line (@msgbuf){
			print2stdout($line);
		}
		$status = 0;
	}
	print2stdout("ok\n") if $status == 1;
	
	print2stdout("\n") unless $install_mode;
	
		
	#print2stdout("check software version... ");
	
	my $hrData = $package->Registry->RegData;
	my %RegistryData=%{${%$hrData}{'HashRef_Packages'}};
		
	#my $rc = checkVersion(\%RegistryData,$package->Name,$package->Path);
	#print2stdout("ok\n\n") if $rc == 1;
	#print2stdout ("\n") and $status = 0 unless ($rc);
	
	

	unless($install_mode){
		print2stdout("check dependencies... ");
		my $rc = checkDependencies(\%RegistryData,$package->Name,$package->Path);
		print2stdout("ok\n\n") if $rc == 1;
		print2stdout ("\n") and $status = 0 unless ($rc);
	
	}

	$package->evalScript();
	if(exists ${%$package}{'verify'} and defined $package->verify){
		print2stdout("check rte registration of package... ");
		eval{
			if(&{$package->verify}){
				print2stdout("ok\n\n");
			}
			else{
				print2stdout("failed\n\n");
				$status = 0;
			}
		}; $@ and print2stderr("error in verify(): $@\n") and diesoft($diemsg);
	}

	return ('status' => $status,'all' => $count_all_files,
			 'missed' => $count_missed_files,'modified' => $count_modified_files,'perm_changed' => $count_perm_changed_files);
}








# main symbol is called directly

sub main {
 	local @ARGV=@_;
	SetDebug(\@ARGV);
	printVersion(\@ARGV);
		
 	%opt_ctrl=(
		'h' => \$opt_help,
		'help' => \$opt_help,
		'F=s' => \$opt_file,
		'file=s' => $opt_file,
		'set_corrupted_invalid' => \$opt_modify
	);	
	
	my $usage= " [-h|--help] [-v|--version] [-F|--file <file name>]\n".
			   "\t\t[--set_corrupted_invalid]";
	my $usage_desc =	"\t-h --help\t\t\tshow this\n".
						"\t-v --version\t\t\tshow version\n".
						"\t-F --file <file name>\t\tcheck installed file\n".
						"\t--set_corrupted_invalid\t\tset inconsistent packages to invalid\n";
						
	SAPDB::Install::Getopt::Long::Configure ('no_pass_through','no_ignore_case');
	if(!GetOptions(%opt_ctrl) || $opt_help){
		print ("\n\nusage: ".$SAPDB::Install::Config{'ProgramName'}."$usage \n\n $usage_desc\n\n\n");
		die("\n\n\n");
	}

	my $write_access = (defined $opt_modify) ? 1 : 0;

	$SAPDB::Install::Values::diemsg = 'MaxDB install registry verification exited abnormally';

	if($write_access and $^O !~ /mswin/i and $> != 0){
		print2stderr("repair mode can start root user only!\n");
		diesoft($diemsg);
	}	

	local $log = SAPDB::Install::Log->new();
	$SAPDB::Install::Values::log=$log;
	local $cleaner = SAPDB::Install::Cleaner->new();
	$cleaner->SetRef(\$SAPDB::Install::Values::log);
	if(defined $write_access && $write_access){
		$log->LogName('verify');		
	}
	else{  
		$log->LogOff(1); # write no logfile
	}
	setSigHandler();
	my ($data_path,$prog_path)=readIndepPath();
 	-d $data_path || print2stderr("no data path found\n") && diesoft($diemsg);
 	-d "$data_path/config/install" || print2stderr("path \"$data_path/config/install\" not found\n") && diesoft($diemsg);
	local $instRegistry = SAPDB::Install::InstallRegistry->new("$data_path/config/install",$write_access,$log);
 	defined $instRegistry || print2stderr("cannot find install registry\n") && diesoft($diemsg); 	
 	$instRegistry->DenyWrite(0) if $write_access;
	$cleaner->SetRef(\$instRegistry);

	
	if($opt_file){
		my ($packname,$path,$absfile) = (SAPDB::Install::Diagnose::InstallRegistry::getPackageByFile({'file' => $opt_file, 'registry' => \$instRegistry}));	
		defined $packname or exit -1;
		my $hrRegistryData=$instRegistry->RegData()->{'HashRef_Packages'};
		my $dispname = $hrRegistryData->{$packname}->{$path}->{DispName};
		$dispname = $packname unless $dispname =~ /\S/;
		my $files = $hrRegistryData->{$packname}->{$path}->{FileList};
		print2stdout ("\nfile is part of package \"$dispname\" [$path]\n");
		print2stdout("checking file... ");
		my %sums;		
		if (ref($files) ne 'HASH'){
			print2stdout(" failed\n");	
			print2stderr("package has no file list\n");
  		}
		else{
			foreach my $file (keys(%$files)){
				my $tot_file = $path.'/'.$file;
				if($^O =~ /mwsin/i){
					normalizePath(\$tot_file);
				}
				else{
					my $resolved = resolveLinks($tot_file,10);
					defined $resolved and $tot_file = $resolved;		
				}
				$sums{$tot_file} = $files->{$file};
			}
			my ($sum_ok,$perm_ok,$user_ok,$group_ok) = (1,1,1,1);		
			
			my @errbuf;
			
			unless($sums{$absfile} eq MD5Sum($absfile)){
					$sum_ok = 0;
					push @errbuf, "file was modified\n";
			}
			
			unless($^O =~ /mswin/i){
				my $file_infos = $hrRegistryData->{$packname}->{$path}->{FileInfos}->{$sums{$absfile}}; 
				if(ref($file_infos) eq 'HASH'){
					my @statbuf = stat($absfile);
					unless($file_infos->{'mode'} == $statbuf[2]){
						$perm_ok = 0;
						push @errbuf,"mode was modified\n"; 
					}
					unless($file_infos->{'uid'} == $statbuf[4]){
						$perm_ok = 0;
						push @errbuf,"owner was modified\n"; 
					}
					unless($file_infos->{'gid'} == $statbuf[5]){
						$perm_ok = 0;
						push @errbuf,"group was modified\n"; 
					}
				}
				else{
					push @errbuff, "WRN: cannot check permissions: no such info in registry\n";		
				}
				
			}
			if($sum_ok && $perm_ok && $user_ok && $group_ok){
				print2stdout("ok\n");
			}
			else{
				print2stdout("failed\n");
			}
			
			foreach my $msg (@errbuf){
				print2stderr($msg);
			}

		
		}
		exit 0;	
	}
	
	
	#	
	# get all packages
	#
	
	local @packages=();

	foreach my $packagename ($instRegistry->getPackageNames()){
		my %versions = $instRegistry->getInstallPathes($packagename);
		foreach my $package_path (keys(%versions)){
			my $regpackobj = $instRegistry->getPackage($packagename,$package_path);
			defined $regpackobj or print2stderr("cannot get package $packagename in $package_path\n") and next;
			push @packages,$regpackobj;
		} 
	}

	$cleaner->SetRef(\@packages);
	
	
	#
	# counter vars
	#
	
	my $valid_packages = 0;
	my $invalid_packages = 0;
	my $bad_packages = 0;
	my $total_files = 0;
	my $missed_files = 0;
	my $modified_files = 0;
	my $perm_changed_files = 0;
	my $new_invalids = 0;
	 
	foreach my $regpackobj (@packages){
		if($regpackobj->Valid){
			$valid_packages++;
			my %rc = checkPackage(\$regpackobj);
			if($rc{'status'}){
				print2stdout("package data is consistent\n");
			}
			else{
				print2stdout("package data is inconsistent\n");
				$bad_packages++;
				$missed_files += $rc{'missed'};
				$modified_files += $rc{'modified'};
				$perm_changed_files += $rc{'perm_changed'};
				if($write_access){
					print2stdout("set package invalid (y/n)? ");
					if(readstdin() =~ /^y+$/i){
						$regpackobj->setValid(0);
						$new_invalids++;
						print2stdout("package is invalid now\n");
					}
				}
			}
			$total_files += $rc{'all'};
		}
		else{
			$invalid_packages++;
			print2stdout("\n\n".$regpackobj->DispName." in ".$regpackobj->Path." is invalid\n");
		}
	}

	# print summary
	$line = "VERIFICATION SUMMARY:";
	print2stdout("\n\n$line\n");
	print2stdout(('*' x length($line))."\n\n");

	my $table = [
		['INVALID PACKAGES:',$invalid_packages],
		['VALID PACKAGES:',$valid_packages],
		['INCONSISTENT PACKAGES:',$bad_packages],
		($write_access ? ['PACKAGES SET TO INVALID:',$new_invalids] : undef ),
		['TOTAL FILES:',$total_files],
		['MISSED FILES:',$missed_files],
		['MODIFIED FILES:',$modified_files],
		['FILES WITH MODIFIED PERMISSIONS:',$perm_changed_files]
				
	];

	printTable($table,' ' x 4);
	print2stdout("\n");
	WriteTrace();
	exit 0;
}	 	

1;