#!/usr/bin/perl
#
# $Header: //sapdb/V75/c_00/b_07/sys/src/install/perl/SAPDB/Install/InstallRegistry/Log.pm#1 $
# $DateTime: 2003/11/20 17:48:59 $
# $Change: 57359 $
#
# Desc: log to restore damaged InstallRegistry  


package SAPDB::Install::InstallRegistry::Log;

$VERSION = 1.01;

sub BEGIN {
      my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages=(
		'StdIO',
		'Values',
		'Misc',
		'Trace',
		'DataDumper',
		'FSUsage'
	);
	@EXPORT=('main');
	foreach my $package (@neededPackages){
	  	unless (defined $repo->Eval ("SAPDB::Install::$package", 1.01)) {
                print join ("\n", $repo->GetErr)."\n";
                die;
        	}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package"); 
	}
}


$DEBUG=0;

@ISA=qw(SAPDB::Install::Base);

# members of class 
my %members=(
	'FileName' => 'INSTREGLOG',
	'Path' => undef,	
	'Size' => 0x200000,
	'BLK_Size' => 0x2000,
	'FD' => undef,
	'ReadOnly' => undef,
	'Closed' => 0
);	


my %log_entry_types = (
	'BEFORE_IMAGE' => {
		'type' => 1,
		'pack_tmpl' => 'a*x', # regdata
		'data' => undef
	}, 
	'PACKAGE_DATA' => {
		'type' => 2,
		'pack_tmpl' => 'a*xa*xa*xa*x', # package path valuename value
		'data' => {
			'package_name' => undef,
			'package_path' => undef,
			'value_name' => undef,
			'value' => undef
		}
	},
	'INSTANCE_DATA' => {
		'type' => 3,
		'pack_tmpl' => 'a*xa*xa*x', # instance valuename value
		'data' => {
			'instance_name' => undef,
			'value_name' => undef,
			'value' => undef
		}
	} 
); 


sub open{
	my ($self) = @_;
	(defined $self->Path and -d $self->Path) or 
		$SAPDB::Install::Values::log->SetMsg("no path to install registry log given\n") and 
		return undef;
	if(-f $self->Path.'/'.$self->FileName){
		TraceMsg("install registry log file already exists\n",3,\$DEBUG);
		if($^O !~ /mswin/i and $> == 0){
			my @statbuf = stat($self->Path.'/'.$self->FileName);
			unless(0644 == ($statbuf[2] & 0777)){
				chmod(0644,$self->Path.'/'.$self->FileName) or
					print2stderr("cannot change permission of Install Registry Log: $!\n");
			}	
		}	
		my $open_mode = '+<';
		if(defined $self->ReadOnly and $self->ReadOnly){
			$open_mode = '';
		}
		
		open(IRLOG,$open_mode.$self->Path.'/'.$self->FileName) or 
			prin2stderr("cannot open install registry log: $!\n") and
			diesoft($diemsg);
	}
	else{
		if(defined $self->ReadOnly and $self->ReadOnly){
			TraceMsg("no install registry log found\n",3,\$DEBUG);	
			return undef;
		}
		TraceMsg("create install registry log",3,\$DEBUG);

		my ($total,$avail) = GetFSSize($self->Path.'/'.$self->FileName);
		my $fsname = GetFSName($self->Path.'/'.$self->FileName);

		if ($avail <= ($self->Size / 0x400)){ 
			print2stderr("InstallRegistry::Log: not enough space ($avail kb) left on filesystem $fsname ($total kb total) - need more than ".($self->Size / 0x100000)."mb\n");
			diesoft($SAPDB::Install::Values::diemsg);
		}
		else{
			$SAPDB::Install::Values::log->SetMsg("MSG: InstallRegistry::Log: space check ok: $avail kb on $fsname ($total kb total) available\n");
		}



		open(IRLOG,'+>'.$self->Path.'/'.$self->FileName) or 
			prin2stderr("cannot create install registry log: $!\n") and
			diesoft($diemsg);
		if($^O !~ /mswin/i and $> == 0){
			my @statbuf = stat($self->Path.'/'.$self->FileName);
			unless(0644 == ($statbuf[2] & 0777)){
				chmod(0644,$self->Path.'/'.$self->FileName) or
					print2stderr("cannot change permission of Install Registry Log: $!\n");
			}	
		}
		binmode(IRLOG);
		my $count = 0;
		while(1){
			$count > ($self->Size/$self->BLK_Size) and last;
			my $written = syswrite IRLOG, (pack('C',0) x $self->BLK_Size) , $self->BLK_Size;
			if($written != $self->BLK_Size){
				print2stderr("cannot reserve space\n");
				diesoft($diemsg);
			}
			$count++;
		}
		sysseek(IRLOG,0,0); 	
	}
	$self->FD(\*IRLOG);
	my @statbuf = stat ($self->FD);
	binmode($self->FD);
	if($statbuf[7] < $self->Size){
		TraceMsg($statbuf[7].' < '.$self->Size."\n",3,\$DEBUG);
		print2stderr("WRN: install registry log has not reached recommended size of ".($self->Size / 0x400)." kb\n");
	}
	return 1;	
}


sub new{
	my ($type,%args) = @_;
	my $self= SAPDB::Install::Base->new;
	foreach my $key (keys(%members)){
		$self->{$key}=$members{$key};
	}
	$self = bless $self,$type;
	foreach my $key (keys(%args)){
		if(exists $members{$key}){
			$self->{$key} = $args{$key};	
		}
		else{
			print2stderr("WRN: unknown member \"$key\" in class \"$type\"\n");
		} 
	}
	$self->open;
	return $self;
}


sub reset{
	my ($self,$before_image_struct) = @_;
	$self->Closed and TraceMsg("log device already closed\n",3,\$DEBUG) and return undef;

	if(defined $self->ReadOnly and $self->ReadOnly){
		TraceMsg("cannot reset log: read only mode\n",3,\$DEBUG);	
		return undef;
	}

	sysseek($self->FD,0,0) or print2stderr("cannot seek install registry log\n") and
		return undef;
	my $count = 0;
	my $before_image = struct2string($before_image_struct);
	unless(defined $before_image){
		TraceMsg("cannot reset install registry log: no valid before image given\n",3,\$DEBUG);	
		return undef;
	}
	while(1){
		$count > ($self->Size/$self->BLK_Size) and last;
		my $written = syswrite $self->FD, (pack('C',0) x $self->BLK_Size) , $self->BLK_Size;
		if($written != $self->BLK_Size){
			print2stderr("cannot reserve space\n");
			diesoft($diemsg);
		}
		$count++;
	}	
	sysseek($self->FD,0,0) or print2stderr("cannot seek install registry log\n") and
		return undef;
	my $len_before_image = length($before_image) + 1;
	my @entry_list = (
			${%{$log_entry_types{'BEFORE_IMAGE'}}}{'type'},
			$len_before_image,
			$before_image
	);
	my $total_len = $len_before_image + 5;
	my $written = syswrite $self->FD, (pack 'CLa*x',@entry_list),$total_len;
	unless($written == $total_len){
		TraceMsg("written = $written; expected = $total_len\n",3,\$DEBUG);
		print2stderr("cannot write into install registry log\n") and diesoft($diemsg);
	}
	return 1;
}	


sub setEntry{
	my ($self,$type,%data) = @_;

	$self->Closed and TraceMsg("log device already closed\n",3,\$DEBUG) and return undef;
	
	if(defined $self->ReadOnly and $self->ReadOnly){
		TraceMsg("cannot set log entry: read only mode\n",3,\$DEBUG);	
		return undef;
	}
	
	unless($type =~ /^INSTANCE_DATA$|^PACKAGE_DATA$/){
		TraceMsg("unkown log entry type \"$type\"\n",3,\$DEBUG);
		return undef;
	}
	if ($type =~ /PACKAGE_DATA/){
		TraceMsg("set $type entry ".$data{'package_name'}.
				' '.$data{'package_path'}.' '.$data{'value_name'}.
				' '.$data{'value'}."\n",5,\$DEBUG);
		my $total_length = length($data{'package_name'}) + 
					length($data{'package_path'}) + 
					length($data{'value_name'}) +
					length($data{'value'}) + 4;
		
		my $value_string = struct2string($data{'value'});
		
		my $total_length = length($data{'package_name'}) + 
					length($data{'package_path'}) + 
					length($data{'value_name'}) +
					length($value_string) + 4;
		
		my @entrylist = (
			${%{$log_entry_types{$type}}}{'type'},
			$total_length,	
			$data{'package_name'},
			$data{'package_path'},
			$data{'value_name'},
			$value_string
		);
		
		my $written = syswrite( 
					$self->FD,
					(pack('CL'.${%{$log_entry_types{$type}}}{'pack_tmpl'},@entrylist)),
					$total_length + 5); 
		if($written != ($total_length + 5)){
			TraceMsg("written = $written; expected = ".($total_length + 5)."\n",3,\$DEBUG); 
			print2stderr("cannot write into install registry log\n") and diesoft($diemsg);
		}

	}
	elsif ($type =~ /INSTANCE_DATA/){
		TraceMsg("set $type entry ".$data{'instance_name'}.
				' '.$data{'value_name'}.' '.$data{'value'}."\n",5,\$DEBUG);
		
		my $total_length = length($data{'instance_name'}) + 
					length($data{'value_name'}) +
					length($data{'value'}) + 3;
		
		
		my @entrylist = (
			${%{$log_entry_types{$type}}}{'type'},
			$total_length,	
			$data{'instance_name'},
			$data{'value_name'},
			$data{'value'}
		);
		
		my $written = syswrite( 
					$self->FD,
					(pack('CL'.${%{$log_entry_types{$type}}}{'pack_tmpl'},@entrylist)),
					$total_length + 5); 
		if($written != ($total_length + 5)){
			TraceMsg("written = $written; expected = ".($total_length + 5)."\n",3,\$DEBUG); 
			print2stderr("cannot write into install registry log\n") and diesoft($diemsg);
		}
	}
	return 1;
}

sub getEntry{
	my ($self) = @_;
	my $item;
	
	$self->Closed and TraceMsg("log device already closed\n",3,\$DEBUG) and return undef;

	my $read  = sysread ($self->FD,$item,1);
	
	unless($read == 1){
		TraceMsg("cannot read 1 byte (enty type)from install registry log: $!\n",3,\$DEBUG);
		return undef;
	}
	
	my ($type) = unpack ('C',$item);
	
	if($type == 0){
		TraceMsg("end of install registry log\n",3,\$DEBUG);		
		
		#
		# seek one byte back for writing new valid log entries
		# 1 byte caused by last log entry type read (should be null character)
		#	
		
		sysseek($self->FD,-1,1) and TraceMsg("seeked one byte back\n",3,\$DEBUG) or TraceMsg("cannot seek one byte back: $!\n",3,\$DEBUG);
			
		return undef; # end of log enties 
	}
	my $type_name = '';
	foreach my $type_str (keys(%log_entry_types)){
		if($type == ${%{$log_entry_types{$type_str}}}{'type'}){
			$type_name = $type_str;
			last;
		} 
	}
	$read  = sysread ($self->FD,$item,4);
	$read == 4 or print2stderr("cannot read 4 bytes from install regitsry log: $!\n");
	my ($length) = unpack('L',$item);
	$read  = sysread ($self->FD,$item,$length);
	
	unless($read == $length){
		TraceMsg("cannot read $length bytes (entry data) from install regitsry log: $!\n",3,\$DEBUG);
		return undef;
	}

	my $tmpl = ${%{$log_entry_types{$type_name}}}{'pack_tmpl'};
	$tmpl =~ s/x//g;
	$tmpl =~ s/a/A/g;
	my @data_list = split("\0",$item); 
	if($type_name eq 'BEFORE_IMAGE'){
		unless($#data_list == 0){
			TraceMsg("log entry type \"$type_name\" has wrong number of data fields\n",3,\$DEBUG);
			return undef;
		}
		my $data_string = $data_list[0];
		$data_string =~ s/\s*$//s;
		my $data = string2struct($data_string); 
		return $type_name,$data; 
	}
	elsif($type_name eq 'PACKAGE_DATA'){
		unless($#data_list == 3 or $#data_list == 1){
			TraceMsg("log entry type \"$type_name\" has wrong number of data fields\n",3,\$DEBUG);
			return undef;
		}
		my %data;
		TraceMsg("package_name = $data_list[0]\n",5,\$DEBUG);
		TraceMsg("package_path = $data_list[1]\n",5,\$DEBUG);
		TraceMsg("value_name = $data_list[2]\n",5,\$DEBUG);
		$data{'package_name'} = $data_list[0]; 
		$data{'package_path'} = $data_list[1]; 
		unless($#data_list == 1){
			$data{'value_name'} = $data_list[2]; 
			my $value_struct = string2struct($data_list[3]);
			unless($data_list[3] eq 'undef'){
				defined $value_struct or return undef;  
			}
			$data{'value'} = $value_struct; 
		}
		return ($type_name,\%data);
	}
	elsif($type_name eq 'INSTANCE_DATA'){
		unless($#data_list == 2 or $#data_list == 0){
			TraceMsg("log entry type \"$type_name\" has wrong number of data fields\n",3,\$DEBUG);
			return undef;
		}
		my %data;
		$data{'instance_name'} = $data_list[0];
		unless($#data_list == 0){
			$data{'value_name'} =   $data_list[1];
			$data{'value'} =   $data_list[2];
		}
		return ($type_name,\%data);
	}
	else{
		TraceMsg("unknown log entry type ($type)\n",3,\$DEBUG);
		return undef;
	}
}



sub restore{
	my ($self) = @_;
	
	$self->Closed and TraceMsg("log device already closed\n",3,\$DEBUG) and return undef;

	my ($type,$data) = $self->getEntry();
	
	unless($type == 'BEFORE_IMAGE'){
		print2stderr("invalid install registry log: no before image found\n");	
		return undef;
	}
	
	unless(defined $data){
		print2stderr("invalid install registry log: no valid data (undefined) inside before image\n");
		return undef;
	}
	
			
	unless(ref($data) eq 'HASH' and exists ${%$data}{'ID'} and ${%$data}{'ID'} eq 'SAP DB Package Registry' and exists ${%$data}{'HashRef_Packages'}){
		TraceMsg("TYPE = \"".ref($data).'"'.(exists ${%$data}{'ID'} ? " ID = ".${%$data}{'ID'}.' ' : ' ID dont exist ').(exists ${%$data}{'HashRef_Packages'} ? 'HashRef_Packages exists' : 'HashRef_Packages dont exist' )."\n",3,\$DEBUG);
		print2stderr("invalid install registry log: no valid data inside before image\n");
		return undef;
	}
	
	local *regdata = $data; # registry data structure from before image
			
	while(1){
		my ($type,$data) = $self->getEntry();
		defined $type or last;
		if($type eq 'PACKAGE_DATA'){
			if(defined ${%$data}{'value_name'} and ${%$data}{'value_name'} ne ''){
				TraceMsg('"'.${%$data}{'package_name'}.'" in "'.${%$data}{'package_path'}.'": "'.${%$data}{'value_name'}.'" = "'.${%$data}{'value'}."\"\n",5,\$DEBUG);
				${%{${%{${%{$regdata{'HashRef_Packages'}}}{${%$data}{'package_name'}}}}{${%$data}{'package_path'}}}}{${%$data}{'value_name'}} = ${%$data}{'value'};
			}
			else{
				TraceMsg('remove package "'.${%$data}{'package_name'}.'" in "'.${%$data}{'package_path'}."\"\n",5,\$DEBUG);
				delete ${%{${%{$regdata{'HashRef_Packages'}}}{${%$data}{'package_name'}}}}{${%$data}{'package_path'}};
				my @packages = keys(%{${%{$regdata{'HashRef_Packages'}}}{${%$data}{'package_name'}}});
				if($#packages == -1){
					TraceMsg('remove package tree for package '.${%$data}{'package_name'}."\n",5,\$DEBUG);
					delete ${%{$regdata{'HashRef_Packages'}}}{${%$data}{'package_name'}};
				}
				else{
					TraceMsg("dont remove package tree: ".($#packages + 1)." package(s) of this type left\n",5,\$DEBUG);
				}
			}
		}
		elsif($type eq 'INSTANCE_DATA'){
			if(defined ${%$data}{'value_name'} and ${%$data}{'value_name'} ne ''){
				TraceMsg('"'.${%$data}{'instance_name'}.'": "'.${%$data}{'value_name'}.'" = "'.${%$data}{'value'}."\"\n",5,\$DEBUG);
				${%{${%{$regdata{'InstanceData'}}}{${%$data}{'instance_name'}}}}{${%$data}{'value_name'}}  = ${%$data}{'value'};
			}
			else{
				TraceMsg('remove instance data of instance "'.${%$data}{'instance_name'}."\"\n",5,\$DEBUG);
				delete ${%{$regdata{'InstanceData'}}}{${%$data}{'instance_name'}};
				my @instances = keys (%{$regdata{'InstanceData'}});
				if($#instances == -1){
					TraceMsg("remove instance data tree\n",5,\$DEBUG);
					delete $regdata{'InstanceData'};
				}
				else{
					TraceMsg("dont remove instance data tree: ".($#instances + 1)." instance(s) left\n",5,\$DEBUG);
				}


			}
		}
	}
		
	return \%regdata;
}



sub DESTROY{
	my $self=shift;
	close($self->FD) unless $self->Closed;
	$self->SAPDB::Install::Base::DESTROY;
		
}






1;