#!/usr/bin/perl
#
# $Header: //sapdb/V75/c_00/b_07/sys/src/install/perl/SAPDB/Install/InstallRegistry.pm#1 $
# $DateTime: 2003/11/20 17:48:59 $
# $Change: 57359 $
#
# Desc: 

package SAPDB::Install::InstallRegistry;

$VERSION = 1.01;

sub BEGIN {
    my $repo = SAPDB::Install::Repository::GetCurrent ();
	my @neededPackages=(
		'DataDumper',
		'InstallRegistry::Log',
		'InstallRegistry::Package',
		'Base',
		'System',
		'FileLock',
		'StdIO',
		'Log',
		'FSUsage',
		'Trace',
		'Tools'
	);
	foreach my $package (@neededPackages) {
	  	unless (defined $repo->Eval ("SAPDB::Install::$package", 1.01)) {
                print join ("\n", $repo->GetErr)."\n";
                die;
        	}
		SAPDB::Install::Exporter::import ("SAPDB::Install::$package");
	  } 
}

@ISA=qw(SAPDB::Install::Base);


# members of class 
my %members=(
	FileName => 'INSTREG',		 # name of physical install registry file 
	InterfaceVersion =>'0.1',	 # current interface version 	
	ReservedSpace => 0x100000,   # number of bytes (1 mb) 
	RegPath => undef,			 # location of install registry file 	
	ReadOnly => undef,			 # readonly flag for install registry	
	refFD => undef,				 # reference to file descriptor
	Lock => undef,				 # file lock object
	RegData => undef,			 # hash reference to current data (in ram) of install registy 
	Log => undef,				 # log object (installation log file)  	
	Pos => undef,				 # data end position in file without reserved space 	
	DenyWrite => 1,				 # disable/enable writes into physicle registry file 	
	Killed => undef,			 # needed for uninstallation -> true: file descriptor already closed 
	InstRegLog => undef,		 # install registry log to recover damaged install registry	
	dirty => 0,					 # flag for noticing changes in registry data 
	AllFiles => undef			 # is a temporary list of all installed files 
);



# definition of instance data struct

my %InstanceData = (
		'TYPE' => undef,				# OLTP/LVC
		'MIGRATION_STRATEGY' => undef,	# needed to decide restarting db or not
		'STARTING_RELEASE' => undef,    # database release before update 
		'TARGET_RELEASE' => undef,		# database release after software update
		'APO_STARTING_RELEASE' => undef,# apo com release before software update
		'APO_TARGET_RELEASE' => undef,	# apo com release after software update
		'INSTANCE_CHECK_OK' => undef,
		'ADDITIONAL' => undef,			# additional instance data reference - dynamic data which structure known by instance upgrade only  
		'LOGPOS' => undef,				# current/expected log position of checked instance, needed to resume broken upgrades correctly
		'BACKUP_LOGPOS' => undef,		# log position of instance at beginning of first update call 
		'SOFTWARE_OK' => undef			# is set if software installation was successful -> begin with last step
);

unless($^O =~ /mswin/i){
	$InstanceData{'DB_PROCESS_OWNER_UID'} = undef;	 # not needed anymore
	$InstanceData{'DB_PROCESS_OWNER_GID'} = undef;	 # not needed anymore	
	$InstanceData{'DB_PROCESS_OWNER'} = undef;
} 


# definition of package registry structure
my %RegStruct = (
			ID => 'SAP DB Package Registry',		# string is used to check the registry file  
			Version => $members{'InterfaceVersion'},# string is used to check the compatibility of registry file 
			HashRef_Packages => undef,				# contains hashref to hash like $HashRef_Versions{$packagename}   
			InstanceData => undef					# contains collected instance knowledge if instance upgrade failed last time
);


# normalize windows path strings: lower case and slashes  
# return 0 if nothing changed else 1
sub normalize{
	my ($string_ref)=@_;
	$^O=~/mswin/i or return 0;
	local *path=$string_ref;
	my $rc=0;
	$path=~s/\\/\//g and $rc=1;
	$path=~tr/A-Z/a-z/ and $rc=1;
	return $rc;	
}



################################################
# mode:												
#												
#		0 or undef	-> all packages				
#		1			-> only valid packages		
#		2			-> only invalid packages	
#												
################################################


sub reserveSpace{
	my ($self)=@_;
	my $fd = $self->refFD;
	my $kb = $self->ReservedSpace / 0x400;
	
	$self->ReadOnly and print2stderr("InstallRegistry::reserveSpace(): readonly mode\n") and return 0; # read only mode
	fileno($fd) or print2stderr("InstallRegistry::reserveSpace(): invalid file handle\n") and return 0; # no valid file descriptor
	
	my $curpos = tell ($fd);
	#$self->Pos($curpos);
	if ($curpos >= $self->ReservedSpace){
		$self->Log->SetMsg("WRN: InstallRegistry::reserveSpace(): file >= $kb kb\n");
		return;
	}
	
	seek($fd,0,1) or print2stderr("InstallRegistry::reserveSpace(): cannot seek\n") and return; # reset eof error
	my $appendbytes = ($self->ReservedSpace) - $curpos;
	print $fd (('0' x $appendbytes)) or print2stderr("InstallRegistry::reserveSpace(): cannot write $appendbytes bytes\n") and return;
	my $len = tell($fd);
	if ($len != $self->ReservedSpace){
		print2stderr("cannot reserve whole needed space ($kb kb), $len bytes reserved\n");
		return;
	}
	$self->Log->SetMsg("MSG: InstallRegistry: reserved $kb kb disk space\n");	
}

# return list of all installed packages
sub getPackageNames{
	my $self=shift;
	my %hash=%{$self->RegData};
	exists $hash{'HashRef_Packages'} or return;
	my $HashRef_Packages = $hash{'HashRef_Packages'};
	defined $HashRef_Packages and ref($HashRef_Packages) ne 'HASH' and $self->Log->SetMsg("WRN: InstallRegisty::getPackageNames(): found wrong type - NO HASH\n") and return;
	return keys(%$HashRef_Packages);
}


sub getPackageData{
	my ($self,$packname,$instpath,$mode)=@_;
	my $hrRegistryData=${%{$self->RegData}}{'HashRef_Packages'};
	normalize(\$instpath);
	$self->existPackage($packname,$instpath,$mode) or return;
	return %{${%{${%$hrRegistryData}{$packname}}}{$instpath}};	
}


# return true if specified package exists in install registry
sub existPackage{
	my ($self,$packagename,$instpath,$mode)=@_;
	my @pathes = $self->getInstallPathes($packagename,$mode);
	my $found=0;
	foreach my $path (@pathes){
		normalize(\$instpath);
		$instpath eq $path and $found=1 and last;
	}
	$found and return 1;
	return 0;
}


# return hash of all install locations with version of specified kind of package 
sub getInstallPathes{
	my ($self,$packagename,$mode)=@_;
	my @packagenames = $self->getPackageNames(); 
	my $found = 0;
	foreach $name (@packagenames){
		$name eq $packagename and $found=1 and last;
	}
    $found or TraceMsg("package $packagename not found\n",3,\$DEBUG) and return; 
	my $hash_ref=${%{$self->RegData}}{'HashRef_Packages'};
	defined $hash_ref and ref($hash_ref) ne 'HASH' and $self->Log->SetMsg("WRN: InstallRegisty::getInstallPathes(): found wrong type - NO HASH\n") and return;
	$hash_ref = ${%$hash_ref}{$packagename};
	my %returnhash;
	foreach my $key (keys(%$hash_ref)){
		if($mode){
			if($mode == 1){
				exists ${%{${%$hash_ref}{$key}}}{'Valid'} or next;
				${%{${%$hash_ref}{$key}}}{'Valid'} == 1 or next;
			}
			elsif($mode == 2){
				${%{${%$hash_ref}{$key}}}{'Valid'} == 0 or next;
			}
			else{
				print2stderr("getInstallPathes(): wrong usage: mode = \"$mode\" not allowed, using default mode 0\n");
			}
		}
		$returnhash{$key}=${%{${%$hash_ref}{$key}}}{'Version'};
	}
	return %returnhash;
}




# remove specified package from install registry
sub removePackage{
	my ($self,$packagename,$path,$mode)=@_;
	normalize(\$path);
	$self->existPackage($packagename,$path,$mode) or $self->Log->SetMsg("WRN: InstallRegistry::removePackage(): cannot remove package \"$packagename\" in \"$path\" - package not found\n") and return;
	my %hPath = $self->getInstallPathes($packagename); 
	my @pathlist= keys(%hPath);
	my %dataroot=%{$self->RegData};
	if($#pathlist == 0){
		#only one existing installation -> can remove packageroot
		local *packages=$dataroot{'HashRef_Packages'};
		local *package = ${%{$packages{$packagename}}}{$path}; 
		%package = (); # mark as empty for regpackobjs
		delete $packages{$packagename};
	}
	else{
		my %packages=%{$dataroot{'HashRef_Packages'}};
		local *pathes = $packages{$packagename};
		local *package = $pathes{$path};
		%package = (); # mark as empty for regpackobjs 
		delete $pathes{$path};
	}
	$self->InstRegLog->setEntry('PACKAGE_DATA',('package_name' => $packagename,'package_path' => $path));
	$self->dirty(1);
	return 1;
}


# return an object of packagedata if exist
sub getPackage{
	my ($self,$packagename,$path,$noSELFOBJ,$mode)=@_;
	$path=~/^\s*$/ and $self->Log->SetMsg("ERR: InstallRegistry::getPackage(): empty path value\n") and return;
	normalize(\$path);
	$self->existPackage($packagename,$path,$mode) or $self->Log->SetMsg("ERR: InstallRegistry::getPackage(): cannot get package: don't exist!\n") and return;	
	my $data_ref=${%{${%{${%{$self->RegData}}{'HashRef_Packages'}}}{$packagename}}}{$path};
	my $package_obj=SAPDB::Install::InstallRegistry::Package->new($self,$packagename,$path,$data_ref,$noSELFOBJ);
	$package_obj->Registry($self);
	return $package_obj;	
}



# create a new package in registry and return the referencing object
sub newPackage{
	my ($self,$packagename,$path,$mode)=@_;
	if($mode == 2){
		#mode 2 -> dangerous / makes no sense -> forbidden
		$mode=0;
	}
	$packagename=~/^\s*$/ and print2stderr("ERR: InstallRegistry::newPackage(): cannot add package with empty package name\n")  and diesoft($SAPDB::Install::Values::diemsg);
	$self->existPackage($packagename,$path,$mode) and print2stderr("ERR: InstallRegistry::newPackage(): cannot add Package \"$packagename\" -> \"$path\": already exist!\n") and diesoft($SAPDB::Install::Values::diemsg);	
	$self->dirty(1);
	my %newHash;
	local *dataroot=$self->RegData;
	defined $dataroot{'HashRef_Packages'} or $dataroot{'HashRef_Packages'}=\%newHash;
	local *packageroot=$dataroot{'HashRef_Packages'};
	my %initHash;
	normalize(\$path);
	my %emptyHash=('Version' => undef); # init with a first member: able to detect removed package correctly 
	my $hashref_pathes=$packageroot{$packagename};
	if(ref($hashref_pathes) eq 'HASH'){
		local *pathes=$hashref_pathes;
		$pathes{$path}=\%emptyHash;	
	}
	else{
		$initHash{$path}=\%emptyHash;
		$packageroot{$packagename}=\%initHash;
	}
	# create packagename
	my $package_obj=SAPDB::Install::InstallRegistry::Package->new($self,$packagename,$path,\%emptyHash);
	$package_obj->Registry($self);
	return $package_obj; 	
}




# read registry file and initialize RegData
sub read{
	my $self=shift;
	$self->Killed and $self->Log->SetMsg("WRN: install registry not present - file descriptor already closed\n") and return 0;
 	my ($data,$pos) = readDump($self->refFD) or $self->Log->SetMsg("WRN: cannot read package DB\n") and return 0;
	ref($data) ne 'HASH' and $self->Log->SetMsg("WRN: Package Registry: readed dump is no HASH\n") and return 0; # have to be a hash reference
	my %hash=%$data;
	${%$data}{'ID'} ne 'SAP DB Package Registry' and $self->Log->SetMsg("WRN: Package Registry: ID not valid\n") and return 0; 	
	${%$data}{'Version'} ne $self->InterfaceVersion  and $self->Log->SetMsg("WRN: Package Registry: version don't match\n") and return 0; 	
	exists $hash{'HashRef_Packages'} or $self->Log->SetMsg("WRN: Package Registry: HashRef_Packages dont exist as key of HASH\n") and return 0; 	
	$self->RegData($data);
	$self->Log->SetMsg("MSG: read ".$self->PackageNum." PACKAGES\n");
	$self->Log->SetMsg("MSG: net registry size = ".$self->sizeOfDataStruct." bytes\n");
	$self->Pos($pos);
	$DEBUG && $self->showRegData;
	$self->dirty(0);
	return 1;
}



# special subs for storing instance data follows
# needed by SDBUPD
# data is set after instance check - to hold data for broken installation 
# data is removed after successful upgrade
 


sub existInstanceData{
	my ($self,$instance_name) = @_;
	exists ${%{${%{$self->RegData}}{'InstanceData'}}}{$instance_name} && TraceMsg("InstanceData for instance \"$instance_name\" exist\n",5,\$DEBUG) && return 1;
	TraceMsg("InstanceData for instance \"$instance_name\" dont exist\n",5,\$DEBUG);
	return 0;
}



sub removeInstanceData{
	my ($self,$instance_name) = @_;
	exists ${%{${%{$self->RegData}}{'InstanceData'}}}{$instance_name} || $self->Log->SetMsg("WRN: InstallRegistry::removeInstanceData(): no data for instance \"$instance_name\"\n") && return;
	local *instance_data = ${%{$self->RegData}}{'InstanceData'};
	TraceMsg("InstanceData deleted for instance \"$instance_name\"\n",3,\$DEBUG);
	$self->InstRegLog->setEntry('INSTANCE_DATA',('instance_name' => $instance_name));
	delete $instance_data{$instance_name};
	
	my @instances = keys(%instance_data);
	
	if($#instances == -1){
		#no instancedata left
		local *regdata = $self->RegData;
		delete $regdata{'InstanceData'};
	}
	
	$self->dirty(1);
	return 1;
}


sub getInstanceData{
	my ($self,$instance_name) = @_;
	exists ${%{${%{$self->RegData}}{'InstanceData'}}}{$instance_name} || $self->Log->SetMsg("MSG: InstallRegistry::getInstanceData(): no data for instance \"$instance_name\"\n") && return;
	return %{${%{${%{$self->RegData}}{'InstanceData'}}}{$instance_name}};
}


sub getExistingInstanceNames{
	my ($self) = @_;
	exists ${%{$self->RegData}}{'InstanceData'} or return undef;
	ref(${%{$self->RegData}}{'InstanceData'}) eq 'HASH' or return undef;
	return keys(%{${%{$self->RegData}}{'InstanceData'}});
}





sub setInstanceData{
	my ($self,$instance_name,$hrData) = @_;
	#exists ${%{${%{$self->RegData}}{'InstanceData'}}}{$instance_name} || print2stderr("InstallRegistry: cannot set new collected instance data: instance data already exist") && diesoft($SAPDB::Install::Values::diemsg);	
	unless (exists ${%{$self->RegData}}{'InstanceData'}){
		local *hashRegData = $self->RegData;
		my %emptyHash = ();
		$hashRegData{'InstanceData'} = \%emptyHash; 
	}
	local *instance_data = ${%{$self->RegData}}{'InstanceData'};
	my %newData = %InstanceData;
	defined ${%{${%{$self->RegData}}{'InstanceData'}}}{$instance_name} and %newData = %{${%{${%{$self->RegData}}{'InstanceData'}}}{$instance_name}};
	foreach my $key (keys(%$hrData)){
		if(exists $InstanceData{$key}){
			if($newData{$key} ne ${%$hrData}{$key}){
				defined ${%$hrData}{$key} and $newData{$key} = ${%$hrData}{$key} and TraceMsg("set $key to ".${%$hrData}{$key}." for instance \"$instance_name\"\n",4,\$DEBUG) and $self->dirty(1);# unable to unset values 
				$self->InstRegLog->setEntry('INSTANCE_DATA',('instance_name' => $instance_name,'value_name' => $key, 'value' => ${%$hrData}{$key}));
				TraceMsg("set new log entry\n",4,\$DEBUG);
			}
			else{
				TraceMsg("$key of $instance_name dont differ from registry data\n",4,\$DEBUG);
			}
		}
		else{
			print2stderr("WRN: InstallRegistry::setInstanceData(): ignore unknown data member: \"$key\"\n");
		}	
	}
	$instance_data{$instance_name} = \%newData;	
	return 1;	
}


sub new{
	my ($type,$path,$write_access,$log) = @_;
	my $create_new_registry = 0;
	$path and $members{RegPath}=$path;
	my $self= SAPDB::Install::Base->new;
	foreach my $key (keys(%members)){
		$self->{$key}=$members{$key};
	}
	$self->{'Log'}=$log;
	$self = bless $self,$type;
		
	if($write_access){
		TraceMsg("have write access\n",3,\$DEBUG);
		my ($total,$avail) = GetFSSize($self->RegPath.'/'.$self->FileName);
		my $fsname = GetFSName($self->RegPath.'/'.$self->FileName);

		if ($avail <= ($self->ReservedSpace / 0x400)){ 
			print2stderr("InstallRegistry: not enough space ($avail kb) left on filesystem $fsname ($total kb total) - need more than ".($self->ReservedSpace / 0x100000)."mb\n");
			diesoft($SAPDB::Install::Values::diemsg);
		}
		else{
			$log->SetMsg("MSG: InstallRegistry: space check ok: $avail kb on $fsname ($total kb total) available\n");
		}

	}
	else{
		TraceMsg("read only mode\n",3,\$DEBUG);
	}
	
	if(-f $self->RegPath.'/'.$self->FileName){
					
		# open registry
		TraceMsg("install registry file already exist\n",3,\$DEBUG);
		my $openmode = '';
		$self->ReadOnly(1);
		if($write_access){
			$openmode = '+<';
			if($^O !~ /mswin/i and $> == 0){
				my @statbuf = stat($self->RegPath.'/'.$self->FileName);
				unless(0644 == ($statbuf[2] & 0777)){
					chmod(0644,$self->RegPath.'/'.$self->FileName) or
						print2stderr("cannot change permission of InstallRegistry: $!\n");
				}	
			}
			$self->ReadOnly(0); 		
			$self->dirty(0);
		}
		open(INSTREG_FD,$openmode.$self->RegPath.'/'.$self->FileName) or print2stderr("cannot open install registry file\n") and diesoft($SAPDB::Install::Values::diemsg);
	}
	else{
		$write_access || print2stderr("install registry not found\n") && diesoft($SAPDB::Install::Values::diemsg);
		$create_new_registry = 1;
		$self->DenyWrite(0);
		
		TraceMsg("create new install registry file\n",3,\$DEBUG);
		
		-d $self->RegPath || print2stderr("install registry path \"".$self->RegPath."\" dont exist\n");
		
		open(INSTREG_FD,'>'.$self->RegPath.'/'.$self->FileName) or print2stderr("cannot create install registry file\n") and diesoft($SAPDB::Install::Values::diemsg);
		
		if($^O !~ /mswin/i and $> == 0){
			my @statbuf = stat($self->RegPath.'/'.$self->FileName);
			unless(0644 == ($statbuf[2] & 0777)){
				chmod(0644,$self->RegPath.'/'.$self->FileName) or
				print2stderr("cannot change permission of InstallRegistry: $!\n");
			}	
		}
		
		# initial InstallRegistry if no valid registry file exist
		my %newRegData=%RegStruct;
		$self->RegData(\%newRegData);
	}
	my $lock = SAPDB::Install::FileLock::new(fileno(INSTREG_FD),$write_access);
	my $rc = $lock->Test;
	if($rc == -1){
		print2stderr("cannot get lock status of install registry: ".$lock->GetError."\n");
		diesoft($SAPDB::Install::Values::diemsg);
	}
	elsif($rc > 0){
		my $unix_special;
		unless($^O=~/mswin/i){
			$unix_special=" (pid=$rc)";
		}
		print2stderr("install registry locked by another process$unix_special\n");
		diesoft($SAPDB::Install::Values::diemsg);
	}
	
	($lock->Lock == 0) or print2stderr("cannot lock install registry: ".$lock->GetError."\n") and diesoft($SAPDB::Install::Values::diemsg);
	$self->Log->SetMsg("MSG: install registry successfully locked\n");
	$self->Lock($lock);
	
	$self->refFD(\*INSTREG_FD);
	
	$self->InstRegLog(SAPDB::Install::InstallRegistry::Log->new(
			'Path' => $self->RegPath,'ReadOnly' => $self->ReadOnly));
	

	my $restored = 0;
	
	if($create_new_registry){
		$self->dirty(1);
		
		#
		# force write to avoid damaged install registry 
		#
		
		$self->dump(0);
	}
	else{
		$self->read();
		$self->dirty(0);
	}
	if(defined $self->RegData){
		unless($self->ReadOnly){
			$self->InstRegLog->reset($self->RegData);
		}
	}
	else{
		$self->Log->SetMsg("install registry damaged: try to restore from install registry log\n");
		my $regdata = $self->InstRegLog->restore();
		defined $regdata or print2stderr("install registry damaged\n") and diesoft($SAPDB::Install::Values::diemsg);
		$restored = 1;
		$self->RegData($regdata);
	}
	
	if($write_access){
		
		unless($create_new_registry || $restored){
			
			# remove oldest copies of install registry
			
			my $keep_num = 2; # number of old registry copies to keep, one will be addidionally created
			opendir(MYDH, $self->RegPath) or print2stderr("InstallRegistry::new(): cannot open directory ".$self->RegPath."\n");
			my $filename=$self->FileName;
			my @INSTREG_COPIES = sort grep { /^$filename\.\d+$/ && -f $self->RegPath."/$_" } readdir(MYDH);
			closedir(MYDH);
			if($#INSTREG_COPIES > ($keep_num -1)){
				for (my $i = 0;$i < ($#INSTREG_COPIES - ($keep_num -1));$i++){
					my $fullname = $self->RegPath.'/'.$INSTREG_COPIES[$i];
					unlink ($fullname) or print2stderr("InstallRegistry::new(): cannot remove file \"$fullname\"\n");
				}
			}
			
			# create a copy of install registry
			$filename = $self->RegPath.'/'.$self->FileName;
			my ($sec,$min,$hour,$day,$mon,$year)=localtime(time);
			$year+=1900;
			$mon++;
			$sec = "0$sec" if length($sec) == 1;
			$min = "0$min" if length($min) == 1;
			$hour = "0$hour" if length($hour) == 1;
			$day = "0$day" if length($day) == 1;
			$mon = "0$mon" if length($mon) == 1;
			my $timestamp="$year$mon$day$hour$min$sec";
			my $fd = $self->refFD;
			open(MYFD,">$filename.$timestamp") or print2stderr("cannot duplicate Install Registry\n") and diesoft($SAPDB::Install::Values::diemsg); 
			seek($fd,0,0)  or print2stderr("cannot duplicate Install Registry\n") and diesoft($SAPDB::Install::Values::diemsg); 
			while (<$fd>){
				print MYFD ($_)  or print2stderr("cannot duplicate Install Registry\n") and diesoft($SAPDB::Install::Values::diemsg); 
			}
			close(MYFD);
			#copyAscii($filename,"$filename.$timestamp") or print2stderr("cannot duplicate Install Registry\n") and diesoft($SAPDB::Install::Values::diemsg); 
		}
			
		$self->reserveSpace;   # reserve a couple of bytes for install registry in filesystem
	}

	return $self;
}

sub dump{
	my ($self,$truncate)=@_;
	$self->ReadOnly and $self->Log->SetMsg("WRN: package registry is in readonly mode\n") and return 0;
	$self->Killed and $self->Log->SetMsg("WRN: install registry not present - file descriptor already closed\n") and return 0;
	unless($self->dirty){
		$self->Log->SetMsg("MSG: dont need to write package registry\n");
		if($truncate){
			if($self->Pos > 0){
				seek($self->refFD,0,1); #reset eof error
				truncate($self->refFD,$self->Pos) or print2stderr ("cannot truncate registry file\n") and return 0;
				TraceMsg("file truncated at position: ".$self->Pos."\n",4,\$DEBUG);
			}
		}
		return 1;	
	}
	$self->DenyWrite and TraceMsg("writes on registry file denied\n",3,\$DEBUG) and return 0;
	$self->Log->SetMsg("MSG: write ".$self->PackageNum." packages\n");
	$self->Log->SetMsg("MSG: net install registry size = ".$self->sizeOfDataStruct." bytes\n");
	$DEBUG && $self->showRegData;
	my $bytes = dumpit($self->RegData,$self->refFD,$truncate);
	$self->Log->SetMsg("MSG: wrote install registry ($bytes bytes)\n");
	return 1;
}

sub DESTROY{
	my $self=shift;
	$self->dump(1); # param means final write -> truncate registry file to real size (else it has reserved size $self->ReservedSize)
	my $rc = $self->Lock->Unlock() unless ($self->Killed);
	(not $self->Killed and $rc == 0) && $self->Log->SetMsg("MSG: install registry successfully unlocked\n");
	close($self->refFD) unless ($self->Killed);
	$self->SAPDB::Install::Base::DESTROY;
}



# help debugging subs
#################################################


# return number of packages:
sub PackageNum{
	my($self) = @_;
	my $hrRegistryData=${%{$self->RegData}}{'HashRef_Packages'};
	my $count=0;
	foreach my $packname (keys(%$hrRegistryData)){
		my @pathes = keys(%{${%$hrRegistryData}{$packname}});
		$count += $#pathes +1;
	}
	return $count;
}

# show current content of registry data
sub showRegData{
	my($self) = @_;
	my $hrRegistryData=${%{$self->RegData}}{'HashRef_Packages'};
	foreach my $packname (keys(%$hrRegistryData)){
		print "Packagename = $packname\n";
		foreach my $path (keys(%{${%$hrRegistryData}{$packname}})){
			print "\n\n\nMAIN PATH: \t\t\t".$path."\n";
			print "SOFTWARE VERSION: \t\t".${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Version'}."\n";
			print "PACKAGE VERSION: \t\t".${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'PackageVersion'}."\n";
			print "VALID: \t\t\t\t$text\n";
			if(${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Mode'} =~ /\S/){
				print "BIT: \t\t\t\t".${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Mode'}." bit\n";
			}
			print "FILE OWNER: \t\t\t".${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'User'}."\n";
			print "FILE GROUP: \t\t\t".${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Group'}."\n";
			print "INSTALLATION DATE:\t\t".${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Date'}."\n";
			my $reqString = join ',',@{${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Require'}};
			print "REQUIRE:\t\t\t".$reqString."\n";
			$text='YES';
			$text='NO' unless(exists ${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'FileList'});
			print "FILELIST:\t\t\t$text\n";
			$text='YES';
			$text='NO' unless(exists ${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Script'});
			print "SCRIPT:\t\t\t\t$text\n";
			#print "PATHES:\n";
			my @pathes = @{${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Pathes'}};
			foreach my $path (@pathes){
				my $pathname=${%$path}{'name'};
				$pathname=~tr/[a-z]/[A-Z]/;
				print "$pathname:\t\t".${%$path}{'value'}."\n";
			}
		}
	}
}


sub RefreshAllFilesList{
	my ($self,$pkg_id)= @_;
	$self->AllFiles({}) unless(defined $self->AllFiles);
	local *list = $self->AllFiles;
	my $hrRegistryData=${%{$self->RegData}}{'HashRef_Packages'};
	#if(defined $pkg_id && ref($pkg_id) eq 'ARRAY' && $#{@$pkg_id} == 1){
	#	foreach my $file (keys(%{${%{${%{${%$hrRegistryData}{${@$pkg_id}[0]}}}{${@$pkg_id}[1]}}}{'FileList'}})){
	#				$list{$path.'/'.$file} = $pkg_id;
	#	}
	#}
	#else{
		# reset old list
		my @list = keys(%list);
		%list = () if $#list > -1;
		# generate new list
		foreach my $packname (keys(%$hrRegistryData)){
			foreach my $path (keys(%{${%$hrRegistryData}{$packname}})){
				foreach my $file (keys(%{${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'FileList'}})){
					$list{normalizePath($path.'/'.$file)} = [$packname,$path];
				}
			}
		}
	#}
	return 1;
}


sub CheckFilesOfNewPackage{
	my ($self,$package_name,$path,$files) = @_;
	unless(defined $self->AllFiles && ref($self->AllFiles) eq 'HASH'){
		$self->RefreshAllFilesList;
	}	
	my %return_value = ('state' => 1,'confilct_files' => undef);
	my %conflict_files; 
	foreach my $file (@$files){
		foreach my $installed_file (keys(%{$self->AllFiles})){
			if (normalizePath($installed_file) eq normalizePath($path.'/'.$file)){
				$conflict_files{$installed_file} = ${%{$self->AllFiles}}{$installed_file};
				last;
			}
		}
	}
	
	my @conflicts = keys(%conflict_files); 
	
	
	if($#conflicts > -1){
		$return_value{'state'} = 0;
		$return_value{'conflict_files'} = \%conflict_files;	
		return %return_value;
	}
	
	# add new files to AllFile struct
	
	local *allfiles = $self->AllFiles;
	foreach my $file (@$files){
		$allfiles{normalizePath($path.'/'.$file)} = [$package_name,$path];
	}
	return %return_value;		
}


sub RemoveFromAllFiles{
	my ($self,$path,$rfiles) = @_;
	local *allfiles = $self->AllFiles;
	ref($rfiles) eq 'ARRAY' or return undef;	 
	foreach my $file (@$rfiles){
		my $key = normalizePath($path.'/'.$file);
		exists $allfiles{$key} and delete $allfiles{$key}; 
	}
	return 1;
}



sub getSize{
	my ($ref)=@_;
	my $size = 0;	
	if(ref($ref) eq 'HASH'){
		foreach my $key (keys(%$ref)){
				$size += length($key);
				$size += getSize(${%$ref}{$key});			
		}
	}
	elsif(ref($ref) eq 'ARRAY'){
		foreach my $element (@$ref){
			$size += getSize($element);
		}

	}
	elsif(ref($ref) eq 'SCALAR'){
		$size += getSize($$ref);
	}
	elsif(ref($ref) eq 'REF'){
		$size += getSize($$ref);
	}
	else{
		$size += length($ref);
	}
	return $size;
}


sub sizeOfDataStruct{
	my ($self) = @_;
	return getSize($self->RegData);	
}

sub getPackageKeyName{
	my ($self,$display_name) = @_;
	my $hrRegistryData=${%{$self->RegData}}{'HashRef_Packages'};
	if(exists ${%$hrRegistryData}{$display_name}){
		return $display_name;  
	}
	
	foreach my $packname (keys(%$hrRegistryData)){
		foreach my $path (keys(%{${%$hrRegistryData}{$packname}})){
			exists ${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'} or next;
			if(${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'} eq $display_name){
				return $packname;	
			}
		}
	}
	return undef;
}



sub getDisplayNameTable{
	my ($self) = @_;
	my %table;
	my $hrRegistryData=${%{$self->RegData}}{'HashRef_Packages'};
	foreach my $packname (keys(%$hrRegistryData)){
		foreach my $path (keys(%{${%$hrRegistryData}{$packname}})){
			exists ${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'} or next;
			if(exists $table{${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'}}){
				if($table{${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'}} eq $packname){
					next;
				}
				else{
					print2stderr("display name \"".${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'}."\" isn't unique\n");
					print2stderr("match package key \"packname\" and \"".$table{${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'}}."\"\n");
					diesoft($diemsg);
				}
			}
			$table{${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'}} = $packname;
		}
	}
	return %table;
}





sub printPackageList{
	my ($self,$packagename) = @_; 
	my @table;
	my $hrRegistryData=${%{$self->RegData}}{'HashRef_Packages'};		
	if(defined $packagename){
		exists ${%$hrRegistryData}{$packagename} or print2stderr("package $packagename not found\n") and return undef;  
		foreach my $path (keys(%{${%$hrRegistryData}{$packagename}})){
		@table = (
				[
				$packagename,
				$path,
				${%{${%{${%$hrRegistryData}{$packagename}}}{$path}}}{'Version'},
				${%{${%{${%$hrRegistryData}{$packagename}}}{$path}}}{'Mode'} =~ /32|64/ ? ${%{${%{${%$hrRegistryData}{$packagename}}}{$path}}}{'Mode'}.' bit' : '',
				${%{${%{${%$hrRegistryData}{$packagename}}}{$path}}}{'Valid'} ? 'valid' : 'invalid'
				]
				);
		}
	}
	else{
		foreach my $packname (keys(%$hrRegistryData)){
			foreach my $path (keys(%{${%$hrRegistryData}{$packname}})){
				push @table, [
					defined ${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'} ? ${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'DispName'} : $packname,
					$path,
					${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Version'},
					${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Mode'} =~ /32|64/ ? ${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Mode'}.' bit' : '',
					${%{${%{${%$hrRegistryData}{$packname}}}{$path}}}{'Valid'} ? 'valid' : 'invalid'
					];
			}
		}
	}
	#$table == -1 and print2stderr("no package found\n") and return undef;		
	printTable(\@table,' ' x 4);
	return 1;
}



1;