package com.sap.security.api.util;

import java.util.Properties;
import java.util.HashMap;
import java.io.InputStream;

/**
 * Title:        User Management 60
 * Description:
 * Copyright:    Copyright (c) 2002
 * Company:      SAP Portals
 * @author d037363
 * @version 1.0
 */

public interface IUMParameters {

    /**
     * returns the String value of a property
     *
     * @param  property the property name
     *
     * @return  the string value of that property
     */
    public String get( String property );

    /**
     * returns the String value of a password property.
     * If the value starts with {base64} (e.g. {base64}abcdef==), the base64 decoded
     * value (UTF8-String) will be returned.
     * If the value starts with {encrypted}, the decrypted value will be returned.
     * Clear text passwords have no prefix or {text}.
     *
     * @param  property the property name
     *
     * @return  the string value of that property
     */
    public String getPassword( String property );

    /**
     *  same as {@link getPassword(String)} except that you already have a String
     *  and not a property name.
     *  @param passwd password that should be decrypted
     *  @return the decrypted password value
     */
    public String getPasswordDecode( String passwd );

    /**
     *  returns the String passwd encoded and with prefix as required for a property
     *  parameter
     *  @param passwd that should be encrypted
     *  @return encrypted password value with prefix {encrypted}
     */
    public String returnPasswordEncode(String passwd);


    /**
     * returns the String value of a property if this property exists, otherwise
     * the default value
     *
     * @param property the property name
     * @param dflt the default value
     *
     * @return the string value of that property
     */
    public String get( String property, String dflt );


    /**
     * returns the a number as a value for a property or thedeafutl, if the
     * property does not exist
     *
     * @param property the property name
     * @param dflt the default value
     *
     * @return a number as a property value
     */
    public int getNumber( String property, int dflt );


    /**
     * returns a boolean value as the value of a property or the default value,
     * if the property does not exist
     *
     * @param property the property name
     * @param dflt the default value
     *
     * @return a boolean value as value for the property
     */
    public boolean getBoolean( String property, boolean dflt );

    /**
     * returns a boolean value as the value of a property or false, if the property
     * does not exist
     *
     * @param property the property name
     *
     * @return a boolean value as value for the property
     */
    public boolean getBoolean( String property);

    /**
     * returns the name of a file as value of a property. If the property
     * does not exist the default value is used for
     * the lookup of the file. The plain filename without path is returned.
     * Does not check for the existence of the file.
     *
     * @param property the property name
     * @param dflt the default value
     *
     * @return file name with complete path, if path found otherwise plain filename.
     */
    public String getFile( String property, String dflt );

    /**
     * returns all properties
     *
     * @return all properties
     */
    public Properties getProperties();


    /**
     * this method initialites the <code>UMParameters</code> object with properties and a fileset
     * @param prop Property object that contains all properties
     * (e.g. from <code>sapum.properties</code>). Must not be <code>null</code>.
     * @param files This is a {@link java.util.HashMap} containing
     * <code>name/byte[]</code> pairs of files which can be
     * read with {@link getInputStream(String)}. Can be empty should not be <code>null</code>.
     */
    public void init(Properties prop, HashMap files);

//    public void addFile();


    /**
     * return stream for requested file. Caller must close stream after usage.
     * @param filename name for requested file (without path)
     * @return InputStream of file, or <code>null</code> if not found
     */
    public InputStream getInputStream(String filename);


    /**
     * Same as {@link getInputStream(String)}, but no trace is written.
     * This should be called only during initialization, if the
     * <code>InternalUMFactory</code> is not yet initialized.
     * @param filename name for requested file (without path)
     * @return InputStream of file, or <code>null</code> if not found
     */
    public InputStream getInputStream_noTrace(String filename);

    /**
     * add a file, so that {@link getInputStream(String)} can find it later
     * @param name filename (without path)
     * @param data <code>byte[]</code> of file data
     */
    public void addFile(String name, byte[] data);

    /**
     * list all known files which can be accessed by {@link getInputStream(String)}
     * @return string array of all filenames
     */
    public String[] listFiles();

    /**
     * change an existing property (there is a limited set of properties which can be changed!)
     * @param name property name
     * @param value new property value
     */
    public void changeProperty(String name, Object value);
}
