package com.sap.security.api.umap;

import java.net.HttpURLConnection;
import java.util.Map;
import java.util.Properties;
import java.io.IOException;

import javax.xml.soap.SOAPMessage;

import com.sap.security.api.IPrincipal;


/**
 *  Main interface for user mapping.
 * 
 */
public interface IUserMappingData
{
    /**
     *  Namespace used to store user mapping data.
     */ 
    public static final String USER_MAPPING_NAMESPACE = "$usermapping$";
    
    /**
     *  Namespace used to store inverse user mapping data.
     */ 
    public static final String USER_MAPPING_INVERSE_NAMESPACE = "$inverse_usermapping$";
    
    /**
     *  Logical attribute name used to store a user mapping for a SAP reference system
     *  for a user as plaintext (use case: attribute in LDAP server which is shared
     *  between ABAP system and UME, attribute contains ABAP user ID)
     */
    public static final String USER_MAPPING_REFSYS_ATTRIBUTE = "REFERENCE_SYSTEM_USER";
    
    /**
     *  Enrich Map with logon data.
     *  @param logonData contains the already
     *         available logonData. enrich
     *         will add the available user
     *         mapping data for the system
     *         identified by the string
     *         returned by {@link #getSystemId()}.
     */
    public void enrich (Map logonData)
        throws NoLogonDataAvailableException;

    /**
     *  Enriches an HttpURLConnection. Depending on
     *  the connection type this either adds
     *  the ticket or basic authentication data
     *  to the request.
     *  @param conn the request being enriched.
     */
    public void enrich (HttpURLConnection conn)
        throws NoLogonDataAvailableException;

    /**
     *  Enriches a jcoProperty object describing
     *  an RFC connection to an SAP backend
     *  system. Like in {@link #enrich(HttpURLConnection)},
     *  depending on the supported connect type this
     *  call either adds uid/pw, ticket or client
     *  certificate to the request.
     *  @param jcoProps Properties object containing
     *          the actual jco connection data
     */
    public void enrich (Properties jcoProps)
        throws NoLogonDataAvailableException;
    
    /**
     *  Enriches a SOAP message. This is done by adding
     *  either a SAPLOGONTICKET or a basic
     *  authentication header (only supported in conjunction with EP 6.0) to the http headers of
     *  the underlying http connection. If you want to 
     *  use the SAPLOGONTICKET, there is nothing left for
     *  you to do. If you want to use user id and password
     *  in a basic authentication header, you need to create
     *  this system in the EP6 system landscape and set the "logonMethod"
     *  property of this system to <code>UIDPW</code>.
     */ 
    public void enrich (SOAPMessage msg)
        throws NoLogonDataAvailableException;

    /**
     *  Returns the user this object was created for
     */
    public IPrincipal getPrincipal ();

    /**
     * Returns the system id this object was created
     * for.
     */
    public String getSystemId ();

    /**
     *  Used to store logon data for a specific user
     *  and a specific system.
     *  @param logonData hashmap containing the logon data.
     */
    public void storeLogonData (Map logonData)
        throws IOException;
        
    /**
     *  Indicates whether the mapping found is for
     *  the principal itself or for one of its parent
     *  principals, i.e. groups or roles.
     *  @return true if the mapping was found for
     *  the principal this object was created for. false
     *  if the mapping was found for a parent principal
     *  of the principal object this object was created
     *  for.
     */ 
    public boolean isMappingDirect ();
}
