package com.sap.security.api.umap;

import java.util.Map;

import com.sap.security.api.IPrincipal;
import com.sap.security.api.IUser;
import com.sap.security.api.UMException;
import com.sap.security.api.umap.system.ISystemLandscapeObject;

/**
 *  Provides an entry point to the user mapping functionality.
 *  In order to get mapping information for a specific user call {@link
 *  #getUserMappingData(String,IPrincipal,Map)}. In order to get R/3 user
 *  ids corresponding to an IUser object call {@link #getR3UserName(IUser,String,Map,boolean)}.
 *  In order to make a reverse lookup after a mapped user id, call
 *  {@link #getInverseMappingData(String,String,byte)}.
 * 
 */
public interface IUserMapping
{
	/**
	 *  logon method key for system attributes
	 */
    public static final String UMAP_SYSTEMATTRS_LOGONMETHOD="logonmethod";
    
    /**
     * @deprecated
     *  r3referencesystem key for system attributes
     */
    public static final String UMAP_SYSTEMATTRS_R3NAMEREF  ="r3referencesystem";
    
    /**
     *  usermappingfields key for system attributes
     */
    public static final String UMAP_USERMAPPING_FIELDS     ="usermappingfields";
    
    /**
     *  usermappingfield key for system attributes
     */
    public static final String UMAP_USERMAPPING_TYPE       ="usermappingtype";
    
    /**
     *  SystemType key for system attributes
     */
    public static final String UMAP_SYSTEM_TYPE            ="SystemType";
    
	/**
	 *  prefix for system ids in usermapping
	 */
	public static final String UMAP_EP6_ALIAS_PREFIX       ="ep6_sl_alias";

    /**
     *  flag to identify a system as an Enterprise Portal 6.0 system landscape alias
     */
    public static final byte   UMAP_SYSTEM_TYPE_EP6_ALIAS  =1;

	/**
	 * @deprecated
	 */    
    public static final String [] UMAP_SYSTEM_FIELDS =
        new String [] { UMAP_SYSTEMATTRS_LOGONMETHOD,
                        UMAP_SYSTEMATTRS_R3NAMEREF,
                        UMAP_USERMAPPING_FIELDS,
                        UMAP_USERMAPPING_TYPE,
                        UMAP_SYSTEM_TYPE
        };
                        
    /**
     *  for internal use
     */
    public static final String UMAP_ENCRYPTION_KEYTYPE_ALIAS="ume.umap.encryption.keytype";
                            
	/**
	 *  for internal use
	 */
    public static final String UMAP_ENCRYPTION_ALGO_ALIAS= "ume.umap.encryption.algo";
                        
	/**
	 *  for internal use
	 */
    public static final String UMAP_R3_MASTER = "ume.r3.mastersystem";
                        
	/**
	 *  for internal use
	 */
    public static final String UMAP_R3_MASTER_UID_MODE = "ume.r3.mastersystem.uid.mode";
    
    
    /**
     *  Gets an {@link IUserMappingData} object to perform user mapping activities.
     *  @param system a system object. Can be null, if only ticket-related enrich calls
     *          (e.g. {@link IUserMappingEnrich#enrich(HttpURLConnection)},
     *          {@link IUserMappingEnrich#enrich(Properties)}) are to be performed.
     *  @param pri Principal the user mapping is to be carried out for. Commonly, this is
     *          a IUser object.
     *  @return an IUserMappingData object.
     */
    public IUserMappingData getUserMappingData (ISystemLandscapeObject system, IPrincipal pri);

    /**
     *  do not use this function anymore. Use {@link #getUserMappingData(ISystemLandscapeObject,IPrincipal)}
     *  instead. 
     */    
    public IUserMappingData getUserMappingData (String      sysid,
                                                IPrincipal  pri,
                                                Map         sysAttrBag);

    /**
     *  This function creates a {@link ISystemLandscapeObject} from sysid and calls
     *  {@link #getR3UserName (IUser,ISystemLandscapeObject,boolean)}. Please see
     *  there for a complete documentation.
     *  @param pri IUser the SAP user name is searched for
     *  @param sysid system identifier. The function {@link ISystemLandscapeObject#getSystemByAlias(String)}
     *  will be called in order to get a {@link ISystemLandscapeObject} object to pass to
     *  {@link #getR3UserName (IUser,ISystemLandscapeObject,boolean)}.
     *  @param sysAttrBag will be ignored
     *  @param bGenerateId see {@link #getR3UserName (IUser,ISystemLandscapeObject,boolean)}.
     *  @throws UMException if the underlying user management operations fail with an Exception.
     */
    public String getR3UserName (IUser pri, String sysid, Map sysAttrBag, boolean bGenerateId)
        throws UMException;
        
    /**
     *  This function creates a {@link ISystemLandscapeObject} from sysid and calls
     *  {@link #getInverseMappingData (String,ISystemLandscapeObject)}.
     *  @param sysid system identifier. The function {@link ISystemLandscapeObject#getSystemByAlias(String)}
     *  will be called in order to get a {@link ISystemLandscapeObject} object to pass to
     *  {@link #getR3UserName (IUser,ISystemLandscapeObject,boolean)}.
     *  @param userid user id in the remote system
     *  @param you <b>must</b> pass {@link #UMAP_SYSTEM_TYPE_EP6_ALIAS} here.
     *  @throws UMException if the underlying user management operations fail with an Exception.
     *  @throws NoLogonDataAvailableException if on of the underlying user mapping operations fail.
     *  @throws MultipleHitException thrown if more than one user object was
     *             found with a mapped user = userid for system sysid. In order
     *             to get the string array of users found call
     *             {@link MultipleHitException#getUserNames()}. 
     */
    public String getInverseMappingData (String sysid, String userid, byte system_type)
        throws NoLogonDataAvailableException, UMException;

    /**
     *      Tells the caller the R/3 user id of the provided IUser object.
     *      <br>
     *      The return value is as follows:
     *      <table><tr><td colspan="2">
     *      case 1: logonmethod (sysid) == SAPLOGONTICKET:
     *      </td></tr>
     *      <tr><td>&nbsp;&nbsp;&nbsp;&nbsp;</td><td>
     *            <table>
     *            <tr><td>Value of ume.r3.mastersystem.uid.mode</td><td></td>Meaning</tr>
     *            <tr>
     *               <td valign="top" align="left">0</td>
     *               <td>the function returns a user id if one is maintained, 
     *                   otherwise null</td>
     *            </tr>
     *            <tr>
     *               <td valign="top" align="left">1</td>
     *               <td>the function returns a user id if one is maintained,
     *                   otherwise the logon id.</td>
     *            </tr>
     *            <tr>
     *               <td valign="top" align="left">2</td>
     *               <td>the function returns a user id if one is maintained,
     *                   otherwise one is generated and stored in the user mapping (unless bGenerateId is false)</td>
     *            </tr>
     *            </table>
     *          </td>
     *      </tr>
     *      <tr><td colspan="2">
     *      case 2: logonmethod (sysid) == UIDPW:</td></tr>
     *      <tr><td>&nbsp;&nbsp;&nbsp;&nbsp;</td><td>user id of sysid or null.</td></tr>
     *      </table>
     *      In case of logon method <i>X509CERT</i> null is returned since
     *      in this case the logon id is maintained internally within R/3 and not known to UME.
     * 
     *      @param user user id of the user the mapped id is queried for
     *      @param system system that the mapped id is queried for. You can
     *      @param bGenerateId controls the generation of R/3 userids (please see explanation above).
     *      @return user id of the R/3 system
     */  
    public String getR3UserName (IUser user, ISystemLandscapeObject system, boolean bGenerateId)
        throws UMException;

    /**
     *  Searches for IUser objects which have maintained a user mapping
     *  for the specified system with the given user id. This function performs
     *  a lookup in the custom attribute table for user objects which have a 
     *  mapping maintained for the given system with the given mapped user id.
     *  Even if no mapping is maintained, a non-null value is returned if the
     *  following three conditions apply:
     *  <ul>
     *    <li>The logon method of <i>system</i> is SAPLOGONTICKET
     *    <li>The value of the parameter ume.r3.mastersystem.uid.mode is 1
     *    <li>A user with logon uid <i>mappedUser</i> exists in UME
     *  </ul> 
     *  In this case the unique id of the UME user with logon uid <i>mappedUser</i>
     *  is returned. In other words: Like {@link #getR3UserName(IUser,ISystemLandscapeObject,boolean)}
     *  the identity mapping should always work (in this case in the opposite direction)
     *  without addition user interaction. If this behaviour is not convenient, it can
     *  be switched off with the parameter ume.r3.mastersystem.uid.mode=0. <p>
     *  @param mappedUser the mapped user id the user searched for has maintained.
     *  @param system system object that identifies the system in question. If null
     *         the system identified by the parameter ume.r3.mastersystem is used.
     *         If no system landscape is installed (e.g. if UME running in a non-portal
     *         environment) then the call behaves as if no user mapping was available
     *         for a system with logon method SAPLOGONMETHOD.
     *  @exception NoLogonDataAvailableException thrown if the search over all
     *             user objects returned no result
     *  @exception MultipleHitException thrown if more than one user object was
     *             found with a mapped user = userid for system sysid. In order
     *             to get the string array of users found call
     *             {@link MultipleHitException#getUserNames()}.
     *  @return   returns the unique id of a user that has maintained a specified
     *             mapping. To get the corresponding IUser object call
     *             {@link com.sap.security.api.IUserFactory#getUser(String)}
     */  
    public String getInverseMappingData (String                 mappedUser,
                                         ISystemLandscapeObject system)
        throws NoLogonDataAvailableException, UMException;
    
    /**
     *  Call for user mapping data optimized for mass calls. The logic is identical to the one
     *  of {@link #getInverseMappingData(String,ISystemLandscapeObject)} except that the search
     *  is performed for a set of users simultaneously. 
     *  @param mappedUsers array of mapped users
     *  @param system system the reverse mapping is to be searched for
     *  @return result map. In order to get a UME user that is mapped to mappedUsers[idx], for instance,
     *          call map.get (mappedUsers[idx]). If the result is non-null it can be casted to IUser.
     */    
    public Map getInverseMappingData (String [] mappedUsers, ISystemLandscapeObject system)
        throws NoLogonDataAvailableException, UMException;
}
