package com.sap.security.api;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;

import com.sap.engine.lib.logging.LoggingHelper;
import com.sap.security.api.acl.IAclManager;
import com.sap.security.api.logon.IAnonymousUserFactory;
import com.sap.security.api.logon.ILoginConstants;
import com.sap.security.api.logon.ILogonAuthentication;
import com.sap.security.api.logon.ISecurityPolicyFactory;
import com.sap.security.api.srvUser.IServiceUserFactory;
import com.sap.security.api.ticket.TicketVerifier;
import com.sap.security.api.umap.IUserMapping;
import com.sap.security.api.umap.system.ISystemLandscapeWrapper;
import com.sap.security.api.util.IUMFileIO;
import com.sap.security.api.util.IUMParameters;
import com.sap.tc.logging.Category;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * The <code>UMFactory</code> class provides functionality to get access to implementations
 * of factory classes. Applications access all the functionality through this
 * class.
 *
 * The <code>UMFactory</code> can be initialized in 2 ways: either the function
 * {@link #initialize(Properties, HashMap)}
 * is called or the first call to
 * {@link #getInstance()} initializes the factory.
 */

public class UMFactory {

	private static Location loc = Location.getLocation(UMFactory.class);
	private static Category cat = Category.getCategory(LoggingHelper.SYS_SECURITY, "Usermanagement");
	
	private static final String PROPFILENAME = "sapum.properties";
	private static final String CFGPATH = "ume.cfg.path"; //TODO: Delete
	private static final String EMPTY_IMPL = "empty";

	private static boolean is_initialized = false;
	/** Description of the Field */
	public static final String VERSIONSTRING =
		"$Id: //shared_tc/com.sapall.security/630_SP_REL/src/_api/java/com/sap/security/api/UMFactory.java#19 $ from $DateTime: 2005/02/22 15:21:26 $ ($Change: 18120 $)";
	private static UMFactory _instance;
	private static IUMFileIO _fileio = null;
//	private static ClassLoader _loader = null; //deprecated
	private IAuthentication _authfactory = null;
	private ILogonAuthentication _logonAuthfactory = null;
	private IGroupFactory _groupFactory = null;
	private IUMParameters _properties = null;
	private IRoleFactory _roleFactory = null;
	private IUserAccountFactory _userAccountFactory = null;
	private IUserFactory _userFactory = null;
	private IPrincipalFactory _principalFactory = null;
	private HashMap _aclManagers = new HashMap();
	private IUserMapping _umap = null;
	private IServiceUserFactory _serviceUserFactory = null;
	private IAnonymousUserFactory _anonymousUserFactory = null;
	private ISecurityPolicyFactory _securityPolicyFactory = null;
	private String _databaseVendor = null;
	private boolean _ticketVerifierInitialized = false;
	private ArrayList _systemLandscapeWrappers = null;

	final static Class[] _interfacesForDummy =
		new Class[] {
			ILogonAuthentication.class,
			IGroupFactory.class,
			IRoleFactory.class,
			IUserAccountFactory.class,
			IUserFactory.class,
			IPrincipalFactory.class,
			IServiceUserFactory.class,
			IAnonymousUserFactory.class,
			ISecurityPolicyFactory.class };

	// use system security manager if there is one!
	private static SecurityManager _securitymanager = null;

	private static Object pcdRoleDataSourceClass = null;

	private static Object pcdLoggerClass = null;

	//~ Constructors ..........................................................

	private UMFactory(Properties prop, HashMap fileset, IUMFileIO fileio) {
		// first try to access property file
		// we must load the parameter class dynamically
		// to build the api
		// author: Marc Noe
		try {
			Class myClass = Class.forName("com.sap.security.core.util.imp.UMParameters", true, getmyClassLoader());

			IUMParameters params = (IUMParameters) myClass.newInstance();
			params.init(prop, fileset);
			this._properties = params;
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("initUMFactory failed.");
		}

		_fileio = fileio;
		is_initialized = true;
	}

	//~ Methods ...............................................................

	/**
	 *
	 * getLogonAuthenticator provides access to an implementation of ILogonAuthentication
	 * @return ILogonAuthentication object used for extended authentication handling.
	 *  For more details see {@link com.sap.security.api.logon.ILogonAuthentication}
	 *
	 **/
	public synchronized static ILogonAuthentication getLogonAuthenticator() {
		if (UMFactory.getInstance()._authfactory == null && UMFactory.getInstance()._logonAuthfactory == null) {
			// initialize AuthFactory
			String className =
				UMFactory.getInstance()._properties.get(
					"ume.logonAuthenticationFactory",
					"com.sap.security.core.logon.imp.JUMAuthenticator");

			UMFactory.getInstance()._logonAuthfactory = (ILogonAuthentication) initializeClass(className);
			UMFactory.getInstance()._authfactory = (IAuthentication) UMFactory.getInstance()._logonAuthfactory;
		}

		return UMFactory.getInstance()._logonAuthfactory;
	}

	/**
	 *
	 * getAuthenticator provides access to an implementation of IAuthentication
	 * @return IAuthentication object used for authentication handling.
		 *  For more details see {@link com.sap.security.api.IAuthentication}
	 **/
	public synchronized static IAuthentication getAuthenticator() {
		if (UMFactory.getInstance()._authfactory == null && UMFactory.getInstance()._logonAuthfactory == null) {
			// initialize AuthFactory
			String className =
				UMFactory.getInstance()._properties.get(
					"ume.authenticationFactory",
					"com.sap.security.core.logon.imp.JUMAuthenticator");

			UMFactory.getInstance()._authfactory = (IAuthentication) initializeClass(className);
			UMFactory.getInstance()._logonAuthfactory = (ILogonAuthentication) UMFactory.getInstance()._authfactory;
		}

		return UMFactory.getInstance()._authfactory;
	}

	/**
	 *  Method without functionality.
	 *  @deprecated applications should not use this method.
	 */
	public synchronized static void setClassLoader(ClassLoader loader) {
//		_loader = loader;
	}

	/**
	 * Gets the class loader to load all the factories.
	 * <p>This is the class loader set with setClassLoader() or the
	 * UMFactory's class loader, if not defined
	 * @return the class loader to load all the factories
	 */
	private final static ClassLoader getmyClassLoader() {
//		ClassLoader l = _loader;
//		return l != null ? l : UMFactory.class.getClassLoader();
		return UMFactory.class.getClassLoader();
	}

//	/**
//	 *  Gets the class loader that the core classes are loaded with.
//	 *  @deprecated applications should not use this method.
//	 *  <p>Only released for internal use
//	 */
//	public synchronized static ClassLoader getClassLoader() {
//
//		SecurityManager sm = UMFactory.getSecurityManager();
//		if (sm != null) {
//			//Permission p = new ProtectedCallPermission(SM_CLASSNAME, SM_METH_GETCLASSLOADER);
//			Permission p = new RuntimePermission("getClassLoader");
//			sm.checkPermission(p);
//		}
//		return _loader;
//	}

	/**
	 *  Sets the security manager that is used to protect the API.
	 *  The security manager can only be set once. More attempts to set
	 *  a security manager result in an IllegalStateException.
	 *  An IllegalStateException is also thrown if there is a system
	 *  security manager and this method is called.
	 *  If the SecurityManagerFactory is visible for the UMFactory, this
	 *  this method must be called before the UMFactory is initialized,
	 *  because in this case during intialization a security manager is set if there is
	 *  neither a system security manager nor this method was called.
	 *  @param securitymanager security manager to be used
	 *  @exception IllegalStateException in case this method has already
	 *             been called before or there is a system security manager
	 */
	public synchronized static void setSecurityManager(SecurityManager securitymanager) {
		if (UMFactory._securitymanager != null)
			throw new IllegalStateException("securitymanager already set!");

		UMFactory._securitymanager = securitymanager;
	}

	/**
	 * A method in the usermanagement that wants to check whether the caller is
	 * allowed to call it should call this method instead of
	 * System.getSecurityManager() to get a security manager to perfom the
	 * checkPermission call.
	 * @return SecurityManager object, if a security manager was set using
	 * 			method setSecurityManager or if s system security manager
	 * 			exists.<br>
	 * 			null otherwise.
	 */
	public static SecurityManager getSecurityManager() {
		return UMFactory._securitymanager;
	}

	/**
	 * Returns an implementation of IUserFactory.
	 * This method should be called to get the
	 * user factory for all user related operations.
	 *
	 * @return UserFactory object
	 * @deprecated : use {@link #getUserFactory()} instead
	 */
	public static IUserFactory getDefaultFactory() {

		return UMFactory.getUserFactory();
	}

	/** Gets the global user mapping object.
	 *  @return IUserMapping object used for handling user mapping operations.
	 *
	 */
	public synchronized static IUserMapping getUserMapping() {
		if (getInstance()._umap == null) {
			String className =
				UMFactory.getInstance()._properties.get("userMapping", "com.sap.security.core.umap.imp.UserMapping");

			UMFactory.getInstance()._umap = (IUserMapping) instantiateClass(className);
		}
		return UMFactory.getInstance()._umap;
	}

	/*******
	 * Returns an implementation of IGroupFactory.
	 * This method should be called to get the
	 * group factory for all group related operations.
	 *
	 * @return IGroupFactory object used for handling group operations
	 *******/
	public synchronized static IGroupFactory getGroupFactory() {
		if (UMFactory.getInstance()._groupFactory == null) {
			// initialize groupFactory
			String className =
				UMFactory.getInstance()._properties.get("groupFactory", "com.sap.security.core.imp.GroupFactory");

			UMFactory.getInstance()._groupFactory = (IGroupFactory) initializeClass(className);

		}

		return UMFactory.getInstance()._groupFactory;
	}

	public static IUMFileIO getUMFileIO() {
		return _fileio;
	}

	private static byte[] getByteArray(InputStream tmp) {
		String mn = "getByteArray(InputStream tmp)";
		byte[] ret;
		if (tmp == null)
			return null;
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		int val;
		try {
			while ((val = tmp.read()) != -1)
				bout.write(val);
			tmp.close();
			ret = bout.toByteArray();
			bout.close();
		} catch (IOException e) {
			LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, e);
			ret = null;
		}
		return ret;
	}

	// filter out .bak and directories
	private static class NoBak implements FilenameFilter {
		public boolean accept(File dir, String name) {
			File f = new File(dir, name);
			return !(name.endsWith(".bak") || f.isDirectory());
		}
	}

	//TODO: Delete
	// return all configuration files from ume.cfg.path directory
	private static String[] scandir() {
		NoBak filter = new NoBak();
		File dir = new File(System.getProperty(CFGPATH));
		return dir.list(filter);
	}

	private static class FileSystemIO implements IUMFileIO {

		private String path = "";
		public FileSystemIO(String p) {
			if (p != null && !p.trim().equals(""))
				path = p.trim() + File.separator;
		}

		public void writeFile(String name, InputStream in) throws IOException {
			try {
				ByteArrayOutputStream tmp = new ByteArrayOutputStream();
				int val;
				while ((val = in.read()) != -1) {
					tmp.write(val);
				}
				FileOutputStream fo = new FileOutputStream(path + name);
				fo.write(tmp.toByteArray());
				fo.close();
				getProperties().addFile(name, tmp.toByteArray());
				tmp.close();
			} catch (FileNotFoundException e) {
				throw new IOException("File not found");
			}
		}

		public InputStream readFile(String name) throws IOException {
			try {
				return new FileInputStream(path + name);
			} catch (FileNotFoundException e) {
				throw new IOException("File not found");
			}
		}

		public String[] getFiles() throws IOException {
			File dir = new File(path);
			if (dir == null) {
				throw new IOException("Path " + path + " is wrong");
			}
			return dir.list();
		}

		public void deleteFile(String name) throws IOException {
			if (name == null) {
				throw new IOException("File could not be deleted");
			} else {
				File toDelete = new File(path + name);
				boolean deleted = toDelete.delete();
				if (deleted != true) {
					throw new IOException("File could not be deleted");
				}
			}
		}

	}

	//TODO delete
	// this function is called if the UMFactory is not initialized
	// on the first getInstance() call e.g. initialize() was not called before
	// It loads the configuration from the directory given by ume.cfg.path
	private static UMFactory loadPropAndFiles() {
		String mn = "loadPropAndFiles()";
		Properties props = new Properties();
		String path = System.getProperty(CFGPATH);
		String filename = path + File.separator + PROPFILENAME;
		try {
			FileInputStream f = new FileInputStream(filename);
			if (f != null) {
				props.load(f);
				f.close();
			}
			//        propFile = filename;
		} catch (FileNotFoundException e) {
			LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, e);
			System.err.println("No property file at: " + path);
			return null;
		} catch (IOException e) {
			LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, e);
			System.err.println("No property file at: " + path);
			return null;
		}
		HashMap files = new HashMap();
		String[] xmlnames = scandir();
		for (int i = 0; i < xmlnames.length; i++) {
			//        System.out.println("FILE= " + xmlnames[i]);
			if (xmlnames[i].equals(PROPFILENAME))
				continue;
			File orig = new File(path + File.separator + xmlnames[i]);
			try {
				InputStream tmp = new FileInputStream(orig);
				if (tmp != null) {
					files.put(xmlnames[i], getByteArray(tmp));
					tmp.close();
				}
			} catch (FileNotFoundException e) {
				LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, e);
			} catch (IOException e2) {
				LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, e2);
			}
		}
		return new UMFactory(props, files, new FileSystemIO(path));
	}

	/**
	 * Returns the instance of UMFactory.
	 * If the <code>UMFactory</code> is not already initialized this function throws
	 * an <code>IllegalStateException</code>.
	 * Note: 	UMFactory will be initialized by EP6 Portal or J2EE Engine 630.
	 * 			If UMFactory should be used standalone, it has to be initialized explicitly with
	 *                             the method initialize(String umeCfgPath)
	 * 			
	 * @return    Instance of <code>UMFactory</code>
	 * @exception java.lang.IllegalStateException If the <code>UMFactory</code> is not 
	 * 			  already initialized.
	 */
	public static UMFactory getInstance() {
		if (_instance == null) {
			String path = System.getProperty(CFGPATH); //TODO Delete
			if (path == null || path.equals(""))
				throw new IllegalStateException(
					"Could not initialize UMFactory! Please set property '"
						+ CFGPATH
						+ "' or call UMFactory.initialize() first");
			else {
				_instance = loadPropAndFiles();
				if (_instance == null) {
					throw new IllegalStateException(
						"Could not initialize UMFactory! Please set property '"
							+ CFGPATH
							+ "' or call UMFactory.initialize() first");
				}
			}
		}
		return _instance;
	}

	/**
	 * Get access to <code>UMParameters</code> interface
	 * @return API for accessing <code>IUMParameters</code>
	 */
	public static synchronized IUMParameters getProperties() {
		return UMFactory.getInstance()._properties;
	}

	/**
	 * Returns an implementation of IRoleFactory.
	 * This method should be called to get the
	 * role factory for all role related operations.
	 *
	 * @return IRoleFactory object used for handling role operations
	 **/
	public synchronized static IRoleFactory getRoleFactory() {
		if (UMFactory.getInstance()._roleFactory == null) {
			// initialize roleFactory
			String className =
				UMFactory.getInstance()._properties.get("roleFactory", "com.sap.security.core.imp.RoleFactory");

			UMFactory.getInstance()._roleFactory = (IRoleFactory) initializeClass(className);
		}

		return UMFactory.getInstance()._roleFactory;
	}

	/**
	 * Returns an implementation of IUserAccountFactory.
	 * This method should be called to get the
	 * user account factory for all user account related operations.
	 *
	 * @return IUserAccountFactory object used for handling user account operations
	 **/
	public synchronized static IUserAccountFactory getUserAccountFactory() {
		if (UMFactory.getInstance()._userAccountFactory == null) {
			// initialize userAccountFactory
			String className =
				UMFactory.getInstance()._properties.get(
					"userAccountFactory",
					"com.sap.security.core.imp.UserAccountFactory");

			UMFactory.getInstance()._userAccountFactory = (IUserAccountFactory) initializeClass(className);
		}

		return UMFactory.getInstance()._userAccountFactory;
	}

	/**
	 * Returns an implementation of IPrincipalFactory.
	 * This method should be called to get the
	 * principal factory for all principal related operations.
	 *
	 * @return IPrincipalFactory object used for handling principal operations
	 **/
	public synchronized static IPrincipalFactory getPrincipalFactory() {
		if (UMFactory.getInstance()._principalFactory == null) {
			// initialize userFactory
			String className =
				UMFactory.getInstance()._properties.get(
					"principalFactory",
					"com.sap.security.core.imp.PrincipalFactory");
			UMFactory.getInstance()._principalFactory = (IPrincipalFactory) initializeClass(className);
		}

		return UMFactory.getInstance()._principalFactory;
	}

	/**
	 * Returns an implementation of IUserFactory.
	 * This method should be called to get the
	 * user factory for all user related operations.
	 *
	 * @return IUserFactory object used for handling user operations
	 **/
	public synchronized static IUserFactory getUserFactory() {
		if (UMFactory.getInstance()._userFactory == null) {
			// initialize userFactory
			String className =
				UMFactory.getInstance()._properties.get("userFactory", "com.sap.security.core.imp.UserFactory");
			UMFactory.getInstance()._userFactory = (IUserFactory) initializeClass(className);
		}

		return UMFactory.getInstance()._userFactory;
	}

	/**
	 * Returns an implementation of IServiceUserFactory.
	 * This method should be called to get the service
	 * user factory for all service user related operations.
	 * Only used for internal use.
	 *
	 * @return IServiceUserFactory object used for handling user operations
	 * <p>Only released for internal use
	 **/
	public synchronized static IServiceUserFactory getServiceUserFactory() {
		if (UMFactory.getInstance()._serviceUserFactory == null) {
			// initialize userFactory
			String className =
				UMFactory.getInstance()._properties.get(
					"serviceUserFactory",
					"com.sap.security.core.srvUser.imp.ServiceUserFactory");
			UMFactory.getInstance()._serviceUserFactory = (IServiceUserFactory) initializeClass(className);
		}

		return UMFactory.getInstance()._serviceUserFactory;
	}

	/**
	 * Gets the default Access Control List (ACL) Manager.
	 * @return IAclManager object used for handling Access Control Lists
	 * For further details check com.sap.security.api.acl.IAclManager
	 */
	public synchronized static IAclManager getAclManager() {
		return getAclManager("default");
	}

	/**
	 * Gets an application specific Access Control List (ACL) Manager.
	 * @return IAclManager object used for handling Access Control Lists
	 * For further details check com.sap.security.api.acl.IAclManager
	 */
	public synchronized static IAclManager getAclManager(String applicationId) {
		// try to get ACL manager from HashMap
		IAclManager aclManager = (IAclManager) UMFactory.getInstance()._aclManagers.get(applicationId);
		if (aclManager == null) {
			// initialize new aclManager
			String className =
				UMFactory.getInstance()._properties.get("ume.acl.manager", "com.sap.security.core.acl.imp.AclManager");
			aclManager = (IAclManager) initializeClass(className);
			// call init method on AclManager via reflection
			try {
				Method gsm = aclManager.getClass().getMethod("init", new Class[] { Class.forName("java.lang.String")});
				gsm.invoke(aclManager, new Object[] { applicationId });
			} catch (Exception e) {
				e.printStackTrace();
				System.out.println(" Unable to init class: " + className);
				throw new RuntimeException(" Unable to init class: " + className + e);
			}
			// put ACL manager in HashMap
			UMFactory.getInstance()._aclManagers.put(applicationId, aclManager);
		}
		return aclManager;
	}

	/**
	 * Returns an array of all used Access Control List (ACL) Managers.
	 * @return String[]   applicationIDs of used ACL managers
	 */
	public synchronized static String[] getAllAclManagers() {
		int size = UMFactory.getInstance()._aclManagers.size();
		String[] aclManagerIDs = new String[size];
		if (size > 0) {
			Iterator it = UMFactory.getInstance()._aclManagers.keySet().iterator();
			int i = 0;
			while (it.hasNext()) {
				aclManagerIDs[i++] = (String) it.next();
			}
		}
		return aclManagerIDs;
	}

	/**
	 * Initialize the UMFactory manually.
	 * For standalone issues.
	 * @param umeCfgPath	String object that contains the path
	 * 						to the sapum.properties file and the additional xml files
	 */
	public static synchronized void initialize(String umeCfgPath) {
		if (_instance == null) {
			String path = umeCfgPath;
			if (path == null || path.equals(""))
				throw new IllegalArgumentException("Can not initialize UMFactory! Given path is not correct");
			else {
				_instance = loadPropAndFiles(path);
				if (_instance == null) {
					throw new IllegalStateException("Can not initialize UMFactory! Given path is not correct");
				}
			}
		}
	}

	private static UMFactory loadPropAndFiles(String umeCfgPath) {
		String mn = "loadPropAndFiles(String umeCfgPath)";
		Properties props = new Properties();
		String path = umeCfgPath;
		String filename = path + File.separator + PROPFILENAME;
		try {
			FileInputStream f = new FileInputStream(filename);
			if (f != null) {
				props.load(f);
				f.close();
			}
			//			  propFile = filename;
		} catch (FileNotFoundException e) {
			LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, e);
			System.err.println("No property file at: " + path);
			return null;
		} catch (IOException e) {
			LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, e);
			System.err.println("No property file at: " + path);
			return null;
		}
		HashMap files = new HashMap();
		String[] xmlnames = scandir(umeCfgPath);
		for (int i = 0; i < xmlnames.length; i++) {
			//			  System.out.println("FILE= " + xmlnames[i]);
			if (xmlnames[i].equals(PROPFILENAME))
				continue;
			File orig = new File(path + File.separator + xmlnames[i]);
			try {
				InputStream tmp = new FileInputStream(orig);
				if (tmp != null) {
					files.put(xmlnames[i], getByteArray(tmp));
					tmp.close();
				}
			} catch (FileNotFoundException e) {
				LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, e);
			} catch (IOException e2) {
				LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, e2);
			}
		}
		return new UMFactory(props, files, new FileSystemIO(path));
	}

	/**
	 * @param umeCfgPath
	 * @return
	 */
	private static String[] scandir(String umeCfgPath) {
		NoBak filter = new NoBak();
		File dir = new File(umeCfgPath);
		return dir.list(filter);
	}

	/**
	   * initialize the UMFactory manually.
	   * Called from Portal UserMangementService or WebAS 6.30 (or any other application
	   * that does all the initialization by itself (e.g. read properties and configuration
	   * files)
	   * @param prop Property object that contains all <code>sapum.properties</code> (it should consider
	   *        the search path that is defined). Must not be <code>null</code>.
	   * @param files This is a {@link java.util.HashMap} containing <code>name/byte[]</code> pairs of files which can be
	   *         read with {@link com.sap.security.api.util.IUMParameters#getInputStream(String)}.
	   *         <code>name</code> should be plain names without path. Can be empty, should not be <code>null</code>.
	   * @param fileio Implements {@link com.sap.security.api.util.IUMFileIO} for reading/writing files
	   */
	public static synchronized void initialize(Properties prop, HashMap files, IUMFileIO fileio) {
		//        System.out.println("initialize called");
		if (_instance == null) {
			synchronized (UMFactory.class) {
				if (_instance == null) {
					_instance = new UMFactory(prop, files, fileio);
					_instance.initializeSecurityManager();
					_instance.initializeSecStore();
				}
			}
		} else {
			// @TODO just do nothing?
			//System.err.println("WARNING! UMFactory is initialized again.");
			if (prop != null) {
				reinitialize(prop);
			}
			// throw new UMRuntimeException("UMFactory is already initialized, this can't be done twice!");
		}
	}

	//reinitialize some properties of the UMFactory
	private static void reinitialize(Properties props) {

		if (props != null && props.size() > 0) {
			IUMParameters params = getProperties();

			Enumeration newKeys = props.keys();
			while (newKeys.hasMoreElements() && params != null) {
				String key = (String) newKeys.nextElement();
				params.changeProperty(key, props.getProperty(key));
			}
		}
	}

	/**
	 * Gets the database vendor
	 * @return String specifying the database vendor
	 * <p>Only released for internal use
	 *
	 */
	public synchronized static String getDatabaseVendor() {
		String mn = "getDatabaseVendor()";
		if (UMFactory.getInstance()._databaseVendor == null) {
			String vendor = null;
			Object o = instantiateClass("com.sap.security.core.InternalUMFactory");
			if (o != null) {
				try {
					Class oClass = o.getClass();
					Method oMethod = null;
					if (oClass != null) {
						oMethod = oClass.getDeclaredMethod("getDatabaseVendor", null);
					}
					if (oMethod != null) {
						vendor = (String) oMethod.invoke(null, null);
					}
				} catch (Exception e) {
					LoggingHelper.traceThrowable(Severity.WARNING, loc, mn, e);
				}
			}
			if (vendor == null) {
				UMFactory.getInstance()._databaseVendor = "Not accessible";
			} else {
				UMFactory.getInstance()._databaseVendor = vendor;
			}
		}
		return UMFactory.getInstance()._databaseVendor;
	}

	// get a security if there none was set by environment and there
	// is no system security manager set
	private void initializeSecurityManager() {
		String mn = "initializeSecurityManager()";
		UMFactory._securitymanager = System.getSecurityManager();
		if (UMFactory._securitymanager == null) {
			class doGetSecurityManagerFromFactory implements PrivilegedExceptionAction {
				public Object run() throws Exception {
					Class smf =
						Class.forName("com.sap.security.core.codesecurity.securitymanager.SecurityManagerFactory");
					Method gsm = smf.getMethod("getSecurityManager", null);
					return gsm.invoke(null, null);
				}
			}

			try {
				UMFactory._securitymanager =
					(SecurityManager) AccessController.doPrivileged(new doGetSecurityManagerFromFactory());
			} catch (PrivilegedActionException pae) {
				Exception e = pae.getException();
				LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, pae);
				if (e instanceof ClassNotFoundException) {
					// that is okay, UM will run without security manager
				} else {
					e.printStackTrace();
					throw new RuntimeException("error installing security manager");
				}
			}
		}
	}

	private boolean initializeTicket() {
		String mn = "initializeTicket()";
		final IUMParameters conf = _properties;

		//
		String strIss = conf.get(ILoginConstants.SSOTICKET_ISSUER_NAME, "");
		String strClient = conf.get(ILoginConstants.SSOTICKET_ISSUER_CLIENT, "000");
		String strValid = conf.get(ILoginConstants.SSOTICKET_LIFETIME);
		//        boolean includeCert            = conf.getBoolean(ILoginConstants.SSOTICKET_INCLUDE_CERT, false);
		//        String  storetype              = conf.get(ILoginConstants.SSOTICKET_KEYSTORE_TYPE);
		int iValid = 0;
		int iValidMin = 0;
		String tt;

		boolean ticketSecureFlag = conf.getBoolean(ILoginConstants.SSOTICKET_SECURE, false);

		// Parse ticket life time
		if (strValid == null) {
			strValid = "60";
		}

		int idx = strValid.indexOf((int) ':');

		try {
			if (idx > 0) {
				iValid = Integer.parseInt(strValid.substring(0, idx));
				iValidMin = Integer.parseInt(strValid.substring(idx + 1));
			} else {
				iValid = Integer.parseInt(strValid);
			}
		} catch (NumberFormatException nfe) {
			LoggingHelper.traceThrowable(Severity.DEBUG, loc, mn, nfe);
			iValid = 60;
		}
        
        boolean bStandalone = conf.getBoolean (ILoginConstants.SSOTICKET_STANDALONE, false);
        char[] d = { 'o' };
        Class[] args_clazz = null;
        String keystore=null;
        String password=null;
        String alias = null; 
        
        if (bStandalone) {
            keystore = conf.get (ILoginConstants.SSOTICKET_KEYSTORE, "ticketKeyStore");
            password = conf.get (ILoginConstants.SSOTICKET_KEYSTORE_PASSWORD, "");
            alias    = conf.get (ILoginConstants.SSOTICKET_KEYALIAS, "testkey");
                
            args_clazz = new Class [] {
                              InputStream.class /* keystore     */
                            , d.getClass()      /* keystore pwd */
                            , String.class      /* alias        */
                            , String.class      /* store type   */     
                            , String.class      /* issuer       */
                            , String.class      /* client       */
                            , Integer.TYPE      /* valid min    */
                            , Integer.TYPE      /* valid hours  */
                            , Boolean.TYPE      /* include cert */
            };            
            
        }
        else {
            args_clazz = new Class [] { String.class /* issuer       */
                            , String.class           /* client       */
                            , Integer.TYPE           /* valid min    */
                            , Integer.TYPE           /* valid hours  */
                            , Boolean.TYPE           /* include cert */
            };
        }
        
        try {
            Class clazz = Class.forName ("com.sap.security.core.ticket.imp.BufferingTicket", false, getmyClassLoader());


            Method init = clazz.getMethod("init", args_clazz);
            //System.out.println("ticketkeystorepwvalue = " + new String(ticketKeyStorePwValue));
            Object[] args = null;
            
            if (bStandalone) {
                args = new Object [] {
                                 UMFactory.getProperties().getInputStream (keystore),
                                 password.length()==0? new char [] { 't', 'e', 's', 't' }:password.toCharArray (),
                                 alias,
                                 null,
                                 strIss,
                                 strClient,
                                 new Integer(iValid),
                                 new Integer(iValidMin),
                                 new Boolean(ticketSecureFlag)};
            }
            else {
                args = new Object [] { strIss, strClient, new Integer(iValid), new Integer(iValidMin), new Boolean(ticketSecureFlag)}; 
            }
                

            Boolean result = (Boolean)init.invoke(null, args);
            return result.booleanValue();
        } catch (Exception e) {
            System.out.println("Unable to initialize ticket: ");
            e.printStackTrace();
            throw new RuntimeException("Initialization failed.");
        }
        
	}

	private void initializeSecStore() {
	}

	/**
	 * Instantiates a class, casts to {@link IConfigurable} and calls the method
	 * initialize with the loaded properties.
	 * @param className the name of the class
	 * @author guenther.wannenmacher@sap.com
	 * @todo What happens if cast fails?
	 */
	private static Object initializeClass(String className) {

		Object obj = null;

		try {
			obj = instantiateClass(className);
			IConfigurable conf = (IConfigurable) obj;
			conf.initialize(getInstance()._properties.getProperties());
		} catch (Exception e) {

			e.printStackTrace();
			System.out.println(" Unable to initialize class: " + className + e);
			throw new UMRuntimeException(e, " Unable to initialize class: " + className + e);
		}

		return obj;
	}

	/**
	 * Instantiates a class.
	 * @param className the name of the class
	 * @author guenther.wannenmacher@sap.com
	 * @todo What happens if instantiating fails?
	 */
	private static Object instantiateClass(String className) {

		Object obj = null;
		try {
			if (className.equals(UMFactory.EMPTY_IMPL)) {
				return GenericDummyImplFactory.getDummyInstance(_interfacesForDummy, getmyClassLoader());
			}
			Class c = Class.forName(className, true, getmyClassLoader());
			obj = c.newInstance();

		} catch (Exception e) {
			e.printStackTrace();
			System.out.println(" Unable to instantiate class: " + className + e);
			throw new UMRuntimeException(e, " Unable to instantiate class: " + className + e);
		}

		return obj;
	}

	/**
	 * getTicketVerifier provides access to an object which can be used for verifing
	 * Tickets
	 * @return TicketVerifier object used for ticket handling.
		 *  For more details see {@link com.sap.security.api.ticket.TicketVerifier}
	 **/
	public TicketVerifier getTicketVerifier() {
		String mn = "getTicketVerifier()";
		synchronized (this) {
			if (!_ticketVerifierInitialized) {
				_ticketVerifierInitialized = this.initializeTicket();
			}
		}

		String className =
			UMFactory.getInstance()._properties.get(
				"TicketVerifier",
				"com.sap.security.core.ticket.imp.BufferingTicket");

		TicketVerifier tv = (TicketVerifier) UMFactory.instantiateClass(className);
		try {
			Method m = tv.getClass().getMethod("setMode", new Class[] { Integer.TYPE });
			Field f = tv.getClass().getField("MODE_VERIFY");

			// tv.setMode (Ticket.MODE_VERIFY);
			m.invoke(tv, new Object[] { f.get(tv)});

			m = tv.getClass().getMethod("setEnforceVerify", new Class[] { Boolean.TYPE });

			// tv.setEnforceVerify (false);
			m.invoke(tv, new Object[] { new Boolean(false)});
		} catch (Exception e) {
			LoggingHelper.traceThrowable(Severity.ERROR, loc, mn, e);
			e.printStackTrace();
		}

		return tv;
	}

	public synchronized static void addSystemLandscapeWrapper(ISystemLandscapeWrapper slw) {
		if (null == getInstance()._systemLandscapeWrappers) {
			getInstance()._systemLandscapeWrappers = new ArrayList();
		}

		getInstance()._systemLandscapeWrappers.add(slw);
	}

	public synchronized static ArrayList getSystemLandscapeWrappers() {
		ArrayList copy = new ArrayList();

		if (getInstance()._systemLandscapeWrappers != null)
			copy.addAll(getInstance()._systemLandscapeWrappers);

		return copy;
	}

	public synchronized static void removeSystemLandscapeWrapper(ISystemLandscapeWrapper slw) {
		ArrayList al = getInstance()._systemLandscapeWrappers;
		if (al != null) {
			for (int i = 0; i < al.size(); i++) {
				if (al.get(i).equals(slw)) {
					al.remove(i);
					break;
				}
			}
		}
	}

	/**
	 * Gets the anonymous user factory for retrieving anonymous user.
	 * @return IAnonymousUserFactory factory handling anonymous user objects.
	 * For further details refer to {com.sap.security.api.logon.IAnonymousUserFactory}
	 */
	public synchronized static IAnonymousUserFactory getAnonymousUserFactory() {
		if (UMFactory.getInstance()._anonymousUserFactory == null) {
			// initialize AnonymousUser
			try {
				Class c = Class.forName("com.sap.security.core.logon.imp.AnonymousUser", true, getmyClassLoader());

				Class[] args = new Class[] {
				};
				Method m = c.getMethod("getInstance", args);
				UMFactory.getInstance()._anonymousUserFactory = (IAnonymousUserFactory) m.invoke(null, new Object[] {
				});
			} catch (Exception ex) {
				// Throw an error in order to really stop further processing.
				ex.printStackTrace();
				throw new RuntimeException(ex.toString());
			}
		}

		return UMFactory.getInstance()._anonymousUserFactory;
	}

	/**
	 * getSecurityPolicy provides access to the security policy object
	 * @return ISecurityPolicy object used for security policy handling.
		 *  For more details see {@link com.sap.security.api.ISecurityPolicy}
	 **/
	public synchronized static ISecurityPolicy getSecurityPolicy() {
		if (UMFactory.getInstance()._securityPolicyFactory == null) {
			// initialize AnonymousUser
			try {
				Class c =
					Class.forName("com.sap.security.core.logon.imp.SecurityPolicyFactory", true, getmyClassLoader());

				Class[] args = new Class[] {
				};
				Method m = c.getMethod("getInstance", args);
				UMFactory.getInstance()._securityPolicyFactory = (ISecurityPolicyFactory) m.invoke(null, new Object[] {
				});
			} catch (Exception ex) {
				// Throw an error in order to really stop further processing.
				ex.printStackTrace();
				throw new RuntimeException(ex.toString());
			}
		}

		return UMFactory.getInstance()._securityPolicyFactory.getSecurityPolicy();
	}

	/**
	 * setDataSourceClass sets a dataSourceClass which is used for
	 * PCD role handling
	 * <p>NOTE: only released for internal use
	 **/
	public static synchronized void setDataSourceClass(Object dataSourceClass) {
		pcdRoleDataSourceClass = dataSourceClass;
	}

	/**
	 * getDataSourceClass gets a dataSourceClass which is used for
	 * PCD role handling
	 * <p>NOTE: only released for internal use
	 **/
	public static synchronized Object getDataSourceClass() {
		return pcdRoleDataSourceClass;
	}

	/***
	 * If Portal Logger Wrapper is used, UMFactory provides access to core classes
	 * <p>NOTE: only released for internal use
	 * @deprecated
	 **/
	public static void setLoggerClass(Object loggerClass) {
		pcdLoggerClass = loggerClass;
	}

	/***
	 * If Portal Logger Wrapper is used, UMFactory provides access to logger object
	 * <p>NOTE: only released for internal use
	 * @deprecated
	 **/
	public static Object getLoggerClass() {
		return pcdLoggerClass;
	}

	/***
	 * isInitialized provides information about the state of UMFactory.
	 * @return true if UMFactory is already initialized and configured,
	 * false otherwise
	 **/
	public static boolean isInitialized() {
		return is_initialized;
	}
}
