package com.sap.security.api;

import java.util.Iterator;

/**
 * Class PrincipalIterator is used to change the return type of 
 * iterators returned by this API (e.g. IUser.getRoles(...))
 * 
 * The following return types are available:<p>
 * <ul>
 * <li><pre>PrincipalIterator.ITERATOR_TYPE_PRINCIPALS</pre> (default) Returns IPrincipal objects</li>
 * <li><pre>PrincipalIterator.ITERATOR_TYPE_UNIQUEIDS_CHECKED</pre> Returns unique ids (String) of existing principals</li>
 * <li><pre>PrincipalIterator.ITERATOR_TYPE_UNIQUEIDS_NOT_CHECKED</pre> Returns unique ids (String) of principals without checking their existence</li>
 * </ul>
 * 
 * Example:<p>
 * <pre>Iterator it = user.getRoles(true);
 * PrincipalIterator pIterator = new PrincipalIterator(it, PrincipalIterator.ITERATOR_TYPE_PRINCIPALS);
 * while (pIterator.hasNext())
 * {
 * 	IPrincipal principal = (IPrincipal)pIterator.next();
 * }</pre>
 * 		
 */
public class PrincipalIterator implements java.util.Iterator
{
	protected static final String VERSIONSTRING = "$Id: //shared_tc/com.sapall.security/630_SP_REL/src/_api/java/com/sap/security/api/PrincipalIterator.java#3 $ from $DateTime: 2004/03/23 14:21:06 $ ($Change: 15115 $)";
			
	public static int ITERATOR_TYPE_UNIQUEIDS_CHECKED 		= 0;	
	public static int ITERATOR_TYPE_UNIQUEIDS_NOT_CHECKED 	= 1;
	public static int ITERATOR_TYPE_PRINCIPALS 				= 2;
	
	protected int 			    mType;	
	private Iterator 	mIterator;

	protected void setType(int type)
	{
		mType = type;
	}		
	
	protected PrincipalIterator()
	{
		//set the default
		mType = ITERATOR_TYPE_UNIQUEIDS_CHECKED;
	}

	/***
	 * Constructor of PrincipalIterator where the return type can be expicitely specified.
	 * 
	 * @param iterator: the original iterator, which was returned by UME objects
	 * @param type: the type which should be returned by the next()-method
	 ***/
	public PrincipalIterator(Iterator iterator, int type)
	{
		if (iterator == null)
		{
			throw new IllegalArgumentException("Given iterator must not be null.");
		}
		if (iterator instanceof PrincipalIterator)
		{
			mIterator = iterator;	
			((PrincipalIterator)mIterator).setType(type);
		}
		else
		{
			throw new IllegalArgumentException("Only itorators returned by UME API are allowed.");
		}
	}

	/***
	 * Constructor of PrincipalIterator.
	 * A PrincipalIterator which was instanciated by this constructor will return the default
	 * type (IPrincipal objects where the existence is checked).
	 * 
	 * @param iterator: the original iterator, which was returned by UME objects
	 ***/
	public PrincipalIterator(Iterator iterator)
	{
		this(iterator, ITERATOR_TYPE_PRINCIPALS);
	}
	
	/***
	 * Checks whether there are more objects.
	 ***/
	public boolean hasNext() 
	{
		return mIterator.hasNext(); 
	}
	
	/***
	 * Returns the next object.
	 * @throws NoSuchElementException if there are no more objects to return.
	 ***/
	public Object next() 
	{
		return mIterator.next(); 
	}

	/***
	 * This method is not supported.
	 * @throws UnsupportedOperationException 
	 ***/
	public void remove() 
	{
		throw new UnsupportedOperationException("Removing elements is not supported.");
	}
}
