package com.sap.security.api;


/**
 *  Title: UME4 Security Policy
 *  Description: Security Policy settings. 
 *  Note: It is not possible to change the current security policy settings
 *        with the setter methods at runtime. If you want to change these
 *        settings, change the configuration and restart the server.
 *
 *@author	Guenther Wannenmacher
 *@version    1.0
 */
public interface ISecurityPolicy {
    /**
     *  Check the validity of the password against the password policy
     *  Set the logonID of the user before with the method setUserName().
     *
     *@param  pass  password to check
     *@return       boolean is the password valid
     * @deprecated please use isPasswordValid(String pass, String logonId) 
     */
    public boolean isPasswordValid(String pass) throws InvalidPasswordException;

    /**
     *  Check the validity of the password against the password policy
     *
     *@param  pass    password to check
     *@param  logoId  logonID to check against the password.
     *@return       boolean is the password valid
     */
    public boolean isPasswordValid(String pass, String logonId) throws InvalidPasswordException;


    /**
     *  Check the validity of the password against the password policy.
     *  Set the logonID of the user before with the method setUserName().
     *
     *@param  pass  password to check
     *@return the error message why the password is not valid or <code>null</code>
     *@deprecated please use validatePassword(String pass, String uid)
     */
    public String validatePassword(String pass);

    /**
     *  Check the validity of the password against the password policy.
     *
     *@param  pass  password to check
     *@return the error message why the password is not valid or <code>null</code>
     */
    public String validatePassword(String pass, String uid);

    /**
     * Generate a password for this security policy.
     * @return  a valid password
     */
    public String generatePassword();

    /**
     *  Gets the Number of Allowed Logon Attempts (AllowedLogonAttempts) attribute
     *  of the ISecurityPolicy object
     *
     *@return    The AllowedLogonAttempts value
     */
    public int getLockAfterInvalidAttempts();

    /**
     *  Sets the Number of Allowed Logon Attempts (AllowedLogonAttempts) attribute
     *  of the ISecurityPolicy object
     *
     *@param  count  The new AllowedLogonAttempts value
     */
    public void setLockAfterInvalidAttempts(int count);

    /**
     *  Gets the AutoUnlockTime(in seconds) attribute of the ISecurityPolicy object
     *
     *@return    The AutoUnlockTime value
     */
    public int getAutoUnlockTime();

    /**
     *  Sets the AutoUnlockTime(in seconds) attribute of the ISecurityPolicy object
     *
     *@param  time  The new AutoUnlockTime value
     */
    public void setAutoUnlockTime(int time);

    /**
     *  Gets the Password Minimum Length(PasswordMinLength) attribute of the
     *  ISecurityPolicy object
     *
     *@return    The PasswordMinLength value
     */
    public int getPasswordMinLength();

    /**
     *  Sets the Password Minimum Length(PasswordMinLength) attribute of the
     *  ISecurityPolicy object
     *
     *@param  length  The new PasswordMinLength value
     */
    public void setPasswordMinLength(int length);

    /**
     *  Gets the Password Maximum Length(PasswordMaxLength) attribute of the
     *  ISecurityPolicy object
     *
     *@return    The PasswordMaxLength value
     */
    public int getPasswordMaxLength();

    /**
     *  Sets the Password Maximum Length(PasswordMaxLength) attribute of the
     *  ISecurityPolicy object
     *
     *@param  length  The new PasswordMaxLength value
     */
    public void setPasswordMaxLength(int length);

    /**
     *  Return a boolean indicating whether the password need to be different from
     *  the user id(uid).
     *
     *@return    a boolean indicating whether the password need to be different
     *      from the user id(uid).
     */
    public boolean getUseridInPasswordAllowed();

    /**
     *  Sets the boolean indicating whether the password need to be different from
     *  the user id(uid).
     *
     *@param  pwnur  a boolean indicating whether the password need to be different
     *      from the user id(uid).
     */
    public void setUseridInPasswordAllowed(boolean pwnur);

    /**
     *  Return a boolean indicating whether the password need to be different from
     *  the old password.
     *
     *@return    a boolean indicating whether the password need to be different
     *      from old password.
     */
    public boolean getOldInNewAllowed();

    /**
     *  Sets the boolean indicating whether the password need to be different from
     *  the old password.
     *
     *@param allow   a boolean indicating whether the password need to be different
     *      from old password.
     */
    public void setOldInNewAllowed(boolean allow);

    /**
     *  Return a number indicating how many capical and lower case letters the password
     * must contain.
     *
     *@return    number of required both capical and lower case letters.
     */
    public int getPasswordMixCaseRequired();

    /**
     * Sets the number indicating how many capical and lower case letters the password
     * must contain.
     *
     *@param  number  the number of both capical and lower case letters.
     */
    public void setPasswordMixCaseRequired(int number);

	/**
	 *  Return a number indicating how many capical and lower case letters the logon id
	 * must contain.
	 *
	 *@return    number of required both capical and lower case letters.
	 */
	public int getLogonIdLowerCaseRequired();

	/**
	 * Sets the number indicating how many capical and lower case letters the logon id
	 * must contain.
	 *
	 *@param  number  the number of both capical and lower case letters.
	 */
	public void setLogonIdLowerCaseRequired(int number);

    /**
     * Return the number indicating how many alphabets and numeric values the password
     * must contain.
     *
     *@return    number of required both alphabets and numeric values.
     */
    public int getPasswordAlphaNumericRequired();

    /**
     * Sets the number indicating how many alphabets and numeric values the password
     * must contain.
     *
     *@param  number  the number of required both alphabets and numeric values.
     */
    public void setPasswordAlphaNumericRequired(int number);

	/**
	 * Return the number indicating how many alphabets and numeric values the logon id
	 * must contain.
	 *
	 *@return    number of required both alphabets and numeric values.
	 */
	public int getLogonIdNumericDigitsRequired();

	/**
	 * Sets the number indicating how many alphabets and numeric values the logon id
	 * must contain.
	 *
	 *@param  number  the number of required both alphabets and numeric values.
	 */
	public void setLogonIdNumericDigitsRequired(int number);

	/**
	 * Return the number indicating how many special characters the password
	 * must contain.
	 *
	 *@return    a number indicating whether the password has to include both
	 *      alphabets and numeric values
	 *  Return the boolean indicating whether the password has to include special
	 *  character
	 */
	public int getPasswordSpecialCharRequired();

	/**
	 * Sets the number indicating how many special characters the password
	 * must contain.
	 *
	 *@param  number  the number of required special characters.
	 *      
	 */
	public void setPasswordSpecialCharRequired(int number);

    /**
     * Return the numbers of days the password is set to expired. If the return value
     * is 0, that means the password does not expired.
     *
     *@return    The numbers of days the password is set to expired
     */
    public int getPasswordExpiredDays();

    /**
     *  Sets the numbers of days the password is to be expired. If the value is set
     * to be 0, that means the password will not expired.
     *
     *@param  days  The numbers of days the password is set to expired
     */
    public void setPasswordExpiredDays(int days);

    /**
     *  Return the boolean value indicating whether the password is allowed to
     *  be changed.
     *
     *@return    A boolean value indicating whether the password is allowed to
     *      be changed
     */
    public boolean getPasswordChangeAllowed();

    /**
     *  Sets the boolean value indicating whether the password is allowed to be changed.
     *
     *@param  pca  The boolean value indicating whether the password is allowed to
     *      changed
     */
    public void setPasswordChangeAllowed(boolean pca);

	/**
	 *  Return the boolean value indicating whether the password is enforced to
	 *  change at the first logon.
	 *
	 *@return    A boolean value indicating whether the password is enforced to
	 *     change
	 */
	public boolean getPasswordChangeRequired();

	/**
	 *  Sets the boolean value indicating whether the password is enforced to change.
	 *
	 *@param  pca  The boolean value indicating whether the password is enforced to 
	 *      change
	 */
	public void setPasswordChangeRequired(boolean pca);
	
    /**
     *  Return the integer value indicating the minimum length of a logon id.
     *
     *@return    The integer value indicating the minimum length of a logon id.
     */
    public int getLogonIdMinLength();

    /**
     *  Sets the integer value indicating the minimum length of a logon id.
     *
     *@param  length  integer value indicating the minimum length of a logon id.
     */
    public void setLogonIdMinLength(int length);

    /**
     *  Return the integer value indicating the maximum length of a logon id.
     *
     *@return    The integer value indicating the maximum length of a logon id.
     */
    public int getLogonIdMaxLength();

    /**
     *  Sets the integer value indicating the maximum length of a logon id.
     *
     *@param  length  integer value indicating the maximum length of a logon id.
     */
    public void setLogonIdMaxLength(int length);

	/**
	 *  Return a number indicating how many special characters the logon id
	 * must contain.
	 *
	 *@return    number of required special characters.
	 */
	public int getLogonIdSpecialCharRequired();

	/**
	 * Sets the number indicating how many special characters the logon id
	 * must contain.
	 *
	 *@param  number  the number of special characters.
	 */
	public void setLogonIdSpecialCharRequired(int number);


    /**
     *  Check the validity of the logon id against the security policy
     *
     *@param  	logonId  logonid to check
     *@return  	boolean true if the logon id is valid
     *@throws	InvalidLogonIdException if the logon id is invalid
     */
    public boolean isLogonIdValid(String logonId) throws InvalidLogonIdException;

	/**
	 * Generate a logon id for this security policy.
	 * @return  a valid logon id
	 * @throws 	InvalidLogonIdException if security policy configuration
	 * 			cannot be fullfilled
	 */
	public String generateLogonId() throws InvalidLogonIdException;

	/**
	 *@deprecated
	 */
	public boolean getCertLogonRequired();

	/**
	 *@deprecated
	 */
	public void setCertLogonRequired(boolean clr);

	/**
	 *@deprecated
	 */
	public int getCookieLifeTime();

	/**
	 *@deprecated
	 */
	public void setCookieLifeTime(int lifeTime);

	/**
	 *@deprecated
	 */
	public void setUserName(String id);
}
