package com.sap.security.api;

import java.io.Serializable;
import java.util.Vector;


/**
 * Class AttributeList is used to define specific attributes which can be used 
 * to populate an IPrincipal object. If you know in advance which 
 * attributes you want to access use this class to define the used attributes.
 * You will have better performance and less communication with 
 * the server if you specify the desired attributes.
 * 
 * Note: Reuse of a AttributeList object which was already used to get a principal
 *       from a factory is not supported. The factory might change the content of
 *       the passed AttributeList object. 
 */

public class AttributeList implements Serializable {

    public static final String VERSIONSTRING = "$Id: //shared_tc/com.sapall.security/630_SP_REL/src/_api/java/com/sap/security/api/AttributeList.java#9 $ from $DateTime: 2004/06/22 13:34:35 $ ($Change: 16497 $)";


/***
 * TYPE_UNKNOWN	is returned by {@link #getAttributeType(String, String)} if the type of the values of an
 * attribute cannot be determined
 ***/
    public static final int TYPE_UNKNOWN = 0;
/***
 * TYPE_STRING	is returned by {@link #getAttributeType(String, String)} if the type of the values of an
 * attribute is of type String
 ***/
    public static final int TYPE_STRING  = 1;
/***
 * TYPE_BLOB	is returned by {@link #getAttributeType(String, String)} if the type of the values of an
 * attribute is of type byte[]
 ***/
    public static final int TYPE_BLOB    = 2;

    protected Vector attributes;
    
    protected boolean mCheckSizeLimit;

	/***
	 * Default constructor of AttributeList. An empty instance of AttributeList is ignored during 
	 * population of an IPrincipal object. You have to use {@link #addAttribute(String, String)}
	 * to add attributes
	 ***/
    public AttributeList()
    {
		this (true);
    }

	/***
	 * Constructor of AttributeList. An empty instance of AttributeList is ignored during 
	 * population of an IPrincipal object. 
	 * You have to use {@link #addAttribute(String, String)}to add attributes
	 * @param checkSizeLimit specifies whether the size limit of 25 attributes should be checked
	 ***/
	public AttributeList(boolean checkSizeLimit)
	{
		attributes = new Vector();
		mCheckSizeLimit = checkSizeLimit;
	}

/***
 * used to compare instances of AttributeList
 * @param obj: object which should be compared with this instance
 * @return true if objects are identical, false otherwise
 ***/
    public boolean equals(Object obj)
    {
        if (!(obj instanceof AttributeList))
        {
            return false;
        }
        AttributeList pa = (AttributeList)obj;
        if (this.getSize() != pa.getSize())
        {
            return false;
        }
        boolean allOk = true;
        AttributeWrapper aw = null;
        for (int i=0; (i< pa.getSize()) && allOk; i++)
        {
            aw = new AttributeWrapper(pa.getNameSpaceOfAttributeAt(i), pa.getAttributeNameOfAttributeAt(i));
            allOk = allOk && (attributes.contains(aw));
        }
        return allOk;
    }

/***
 * returns the type of the attribute of this instance of AttributeList.
 * @param index: index of attribute in AttributeList
 * @return following constants: TYPE_UNKNOWN, TYPE_STRING, TYPE_BLOB
 ***/
    public int getAttributeTypeOfAttributeAt(int index)
    {
        return ((AttributeWrapper)attributes.elementAt(index)).getType();
    }

/***
 * returns the type of the attribute of this instance of AttributeList.
 * @param nameSpace: namespace of the attribute
 * @param attributeName: name of the attribute
 * @return following constants: TYPE_UNKNOWN, TYPE_STRING, TYPE_BLOB
 ***/
    public int getAttributeType(String nameSpace, String attributeName)
    {
        boolean found = false;
        int size = attributes.size();
        String aName;
        String aNamespace;
        int attributeType = TYPE_UNKNOWN;
        
        for (int i=0; (i<size) && !found; i++)
        {
            aName = ((AttributeWrapper)attributes.elementAt(i)).getAttributeName();
            aNamespace = ((AttributeWrapper)attributes.elementAt(i)).getNameSpace();
            if (aName.equals(attributeName) && aNamespace.equals(nameSpace))
            {
                found = true;
                attributeType = ((AttributeWrapper)attributes.elementAt(i)).getType();
            }    
        }
        return attributeType;
    }

	/***
	 * Removes an attribute from this instance of AttributeList with following properties
	 * @param nameSpace: namespace of the attribute
	 * @param attributeName: name of the attribute
	 ***/
		public void removeAttribute(String nameSpace, String attributeName)
		{
			int size = attributes.size();
			int index = -1;
			AttributeWrapper dummy;
			for (int i=0; (i<size) && (index != -1); i++)
			{
				dummy = (AttributeWrapper)attributes.elementAt(i);
				if (dummy.getAttributeName().equals(attributeName) && dummy.getNameSpace().equals(nameSpace))
				{
					index = i;
				}
			}
			if(index != -1)
			{
				attributes.remove(index);
			}
		}
	 
 	/***
 	 * Returns whether an attribute is contained in this instance of AttributeList with following properties
 	 * @param nameSpace: namespace of the attribute
	 * @param attributeName: name of the attribute
 	 * @param type: the attribute type
 	 * @return boolean: true if this instance contains this attribute, otherwise false
 	 ***/
 		public boolean containsAttribute(String nameSpace, String attributeName, int type)
 		{
 			int size = attributes.size();
 			boolean found = false;
 			AttributeWrapper dummy;
 			for (int i=0; (i<size) && !found; i++)
 			{
 				dummy = (AttributeWrapper)attributes.elementAt(i);
 				found = dummy.getAttributeName().equals(attributeName) 
 						&& dummy.getNameSpace().equals(nameSpace)
 						&& ((type == TYPE_UNKNOWN) || (type == dummy.getType()) || (dummy.getType() == TYPE_UNKNOWN));
 			}
 			return found;
 		}
	 
	 	public boolean containsAttribute(String nameSpace, String attributeName)
	 	{
	 		return this.containsAttribute(nameSpace, attributeName, TYPE_UNKNOWN);
	 	}


/***
 * Add an attribute to this instance of AttributeList with following properties
 * 
 * Note: A AttributeList can only contain 25 attributes.
 * 
 * @param nameSpace: namespace of the attribute
 * @param attributeName: name of the attribute
 * @throws UMRuntimeException if the AttributeList already contains 25 attributes.
 ***/
    public void addAttribute(String nameSpace, String attributeName)
    {
        this.addAttribute(nameSpace, attributeName, TYPE_UNKNOWN);
    }

/***
 * Add an attribute to this instance of AttributeList with following properties
 * @param nameSpace: namespace of the attribute
 * @param attributeName: name of the attribute
 * @param type of the attribute. Following constants are allowed:
 *  TYPE_UNKNOWN, TYPE_STRING, TYPE_BLOB
 * @throws UMRuntimeException if the AttributeList already contains 25 attributes.
 ***/
    public void addAttribute(String nameSpace, String attributeName, int type)
    {
    	if (mCheckSizeLimit && (attributes.size() == 25))
    	{
    		throw new UMRuntimeException("This AttributeList contains already 25 attributes.");
    	}
        if ((nameSpace == null) || (attributeName == null))
        {
            throw new UMRuntimeException("Can't add attribute when namespace or attribute name is null!");
        }
        if (!this.containsAttribute(nameSpace,attributeName, type))
        {
			AttributeWrapper aw = new AttributeWrapper(nameSpace,attributeName, type);
			attributes.add(aw);
        }
    }
    
/***
 * Add attributes to this instance of AttributeList 
 * @param populateAttributes: instance of another AttributeList which is used to copy
 * the attributes into this instance
 * @throws UMRuntimeException if the AttributeList already contains 25 attributes.
 ***/
    public void addAttributeList(AttributeList populateAttributes)
    {
        if (populateAttributes != null)
        {
            int size = populateAttributes.getSize();
            for (int i=0; i<size; i++)
            {
                this.addAttribute( populateAttributes.getNameSpaceOfAttributeAt(i),
                                    populateAttributes.getAttributeNameOfAttributeAt(i),
                                    populateAttributes.getAttributeTypeOfAttributeAt(i) );
            }
        }
    }
    
    private AttributeWrapper getAttributeAt(int i)
    {
    	return (AttributeWrapper)attributes.elementAt(i);
    }
    
	/***
	 * Returns a whether this AttributeList is a subset of the given AttributeList
	 * 
	 * @return true if this AttributeList is a subset of the given AttributeList or false if this AttributeList is empty or the passed AttributeList is null
	 ***/
    public boolean isSubsetOf(AttributeList populateAttributes)
    {
		if ((populateAttributes == null) || (this.getSize() == 0))
		{
			return false;
		}
		int size = this.getSize();
		boolean allFound = true;
		for (int i=0; (i<size) && (allFound); i++)
		{
			if ( !populateAttributes.contains(this.getAttributeAt(i)) )
			{
				allFound = false;
			}
		}
		return allFound;
    }
    
    private boolean contains(AttributeWrapper attrWrapper)
    {
    	return attributes.contains(attrWrapper);
    }
    
	/***
	 * Returns a new instance of AttributeList which contains all attributes which are
	 * contained in the passed AttributeList, but not contained in this instance. 
	 * @return Object: the new AttributeList instance
	 ***/
    public AttributeList getNotContainedAttributes(AttributeList populateAttributes)
    {
		if (populateAttributes == null)
		{
			return new AttributeList();
		} 
		AttributeList attrList = new AttributeList();
		int size = populateAttributes.getSize();
		for (int i=0; i<size; i++)
		{
			if (!attributes.contains( populateAttributes.getAttributeAt(i) ))
			{
				attrList.addAttribute(  populateAttributes.getNameSpaceOfAttributeAt(i),
										populateAttributes.getAttributeNameOfAttributeAt(i),
										populateAttributes.getAttributeTypeOfAttributeAt(i) );
			}
		}
		return attrList;
    }
    
	/***
	 * Returns a new instance of AttributeList which contains the same data as this instance. 
	 * @return Object: the new AttributeList instance
	 ***/
    public Object clone()
    {
    	AttributeList attrList = new AttributeList(mCheckSizeLimit);
    	attrList.addAttributeList(this);
    	return attrList;
    }

/***
 * Returns the number of components in this attributeList
 ***/
    public int getSize()
    {
        return attributes.size();
    }

/***
 * Returns the namespace of an attribute at a given index in this attributeList
 * @return String: name of attribute's namespace
 ***/
    public String getNameSpaceOfAttributeAt(int index)
    {
        return ((AttributeWrapper)attributes.elementAt(index)).getNameSpace();
    }

/***
 * Returns the name of an attribute at a given index in this attributeList
 * @param given index in attributeList
 * @return String: name of attribute at given index
 ***/
    public String getAttributeNameOfAttributeAt(int index)
    {
        return ((AttributeWrapper)attributes.elementAt(index)).getAttributeName();
    }

/***
 * Private internally used class to administrate attributeList
 ***/
    private class AttributeWrapper implements Serializable
    {
        private String mNameSpace;
        private String mAttributeName;
        private int    mType;

        public AttributeWrapper(String nameSpace, String attributeName)
        {
            mNameSpace     = nameSpace;
            mAttributeName = attributeName;
            mType          = TYPE_UNKNOWN;
        }

        public AttributeWrapper(String nameSpace, String attributeName, int type)
        {
            mNameSpace     = nameSpace;
            mAttributeName = attributeName;
            mType          = type;
        }

        public boolean equals(Object obj)
        {
            if (!(obj instanceof AttributeWrapper))
            {
                return false;
            }
            AttributeWrapper aw = (AttributeWrapper)obj;
            return (aw.getNameSpace().equals(mNameSpace) && aw.getAttributeName().equals(mAttributeName) && ((aw.getType() == mType) || (aw.getType()==TYPE_UNKNOWN) || (mType==TYPE_UNKNOWN) ));
        }
        
        public int getType()
        {
            return mType;
        }

        public String getNameSpace()
        {
            return mNameSpace;
        }

        public String getAttributeName()
        {
            return mAttributeName;
        }

        public String toString()
        {
            StringBuffer res = new StringBuffer();
			res.append("\"");
            res.append(mNameSpace);
            res.append("\"|->\"");
            res.append(mAttributeName);
            res.append("\" (type: ");
            switch( mType)
            {
                case TYPE_STRING:
                      res.append("string");
                break;

                case TYPE_BLOB:
                      res.append("blob");
                break;

                case TYPE_UNKNOWN:
                      res.append("unknown");
                break;
            }
            res.append(")");
            return res.toString();
        }

    }

/***
 * Returns a string representation of this AttributeList.
 * Containing the String representation of each element.
 ***/
    public String toString()
    {
        StringBuffer res = new StringBuffer();
		res.append("***************************************************************************\n");
		res.append("* ");
		if (mCheckSizeLimit)
		{
			res.append("Size limit (25) enabled.");
		}
		else
		{
			res.append("No size limit.");
		}
		res.append("\n");
		res.append("* ");
		res.append(this.getClass().getName());
		res.append("\n");
        res.append("*\n");
        for (int i=0; i<attributes.size(); i++)
        {
            res.append("* ");
            res.append( ((AttributeWrapper)attributes.get(i)).toString() );
            res.append("\n");
        }
		res.append("***************************************************************************\n");
        return res.toString();
    }

}

