/*
 * Created on Jun 27, 2003
 *
 */
package com.sap.caf.km.ejb.svc.idxsearch.index;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import com.sap.caf.km.ejb.svc.idxsearch.bo.BODbHelper;
import com.sap.caf.km.ejb.svc.idxsearch.common.IIndex;
import com.sap.caf.km.ejb.svc.idxsearch.util.ConnectionManager;
import com.sap.caf.km.ejb.svc.idxsearch.util.IDbConst;

/**
 * @author viachaslau_kudzinau@epam.com
 */
public final class KMIndexHelper extends BODbHelper {
	public KMIndexHelper(ConnectionManager manager) {
		super(manager);
	}

	public IIndex assignIndex(String BOName, String indexId, int indexType) throws SQLException {
		Connection conn = m_manager.getConnection();
		int iUpdateType = 0; // 0 for insert, 1 for update
		try {
			BigDecimal boId = getBusinessObjectId(conn, BOName);
			String sql;
			PreparedStatement pstmt;
			pstmt = conn.prepareStatement(IDbConst.INDEX_EXISTS);
			try {
				pstmt.setBigDecimal(1, boId);
				pstmt.setInt(2, indexType);
				ResultSet rs = pstmt.executeQuery();

				if (indexType == IIndex.TYPE_KM_BO_GLOBAL || !rs.next()) {
					sql = IDbConst.INDEX_INSERT;
				}
				else {
					iUpdateType = 1;
					sql = IDbConst.INDEX_UPDATE;
				}
			}
			finally {
				pstmt.close();
			}
			pstmt = conn.prepareStatement(sql);
			try {
				Timestamp creationDate = new Timestamp(System.currentTimeMillis());

				pstmt.setString(1, indexId);
				pstmt.setTimestamp(2, creationDate);
				pstmt.setBigDecimal(3, boId);
				pstmt.setInt(4, indexType);

				if (iUpdateType == 0) {
					BigDecimal key = getNextInSequence(conn, IDbConst.CAF_RT_KEY_SEQUENCE);
					pstmt.setBigDecimal(5, key);
				}

				pstmt.executeUpdate();
				return new KMIndex(indexId, creationDate, BOName, indexType, IIndex.STATE_UNDEFINED);
			}
			finally {
				pstmt.close();
			}
		}
		catch (SQLException ex) {
			throw ex;
		}
		finally {
			conn.close();
		}
	}

	public void unassignIndex(String BOName, int indexType) throws SQLException {
		Connection conn = m_manager.getConnection();
		try {
			BigDecimal boId = getBusinessObjectId(conn, BOName);
			PreparedStatement pstmt = conn.prepareStatement(IDbConst.INDEX_DELETE);
			try {
				pstmt.setBigDecimal(1, boId);
				pstmt.setInt(2, indexType);
				pstmt.executeUpdate();
			}
			finally {
				pstmt.close();
			}
			deleteBusinessObject(conn, BOName);
		}
		catch (SQLException ex) {
			throw ex;
		}
		finally {
			conn.close();
		}
	}

	public void unassignIndex(String BOName, String indexId, int indexType) throws SQLException {
		Connection conn = m_manager.getConnection();
		try {
			BigDecimal boId = getBusinessObjectId(conn, BOName);
			PreparedStatement pstmt = conn.prepareStatement(IDbConst.INDEX_DEL_BY_NAME_AND_TYPE);
			try {
				pstmt.setBigDecimal(1, boId);
				pstmt.setString(2, indexId);
				pstmt.setInt(3, indexType);
				pstmt.execute();
			}
			finally {
				pstmt.close();
			}
			deleteBusinessObject(conn, BOName);
		}
		catch (SQLException ex) {
			throw ex;
		}
		finally {
			conn.close();
		}
	}

	public IIndex[] getAllIndexes() throws SQLException {
		Connection conn = m_manager.getConnection();
		try {
			PreparedStatement pstmt = conn.prepareStatement(IDbConst.INDEX_SELECT);
			try {
				ResultSet rs = pstmt.executeQuery();
				ArrayList indexes = new ArrayList();
				String BOName;
				String indexId;
				int indexType;
				Timestamp reindexDate;
				while (rs.next()) {
					BOName = rs.getString(IDbConst.BO_NAME);
					indexId = rs.getString(IDbConst.INDEX_NAME);
					reindexDate = rs.getTimestamp(IDbConst.REINDEX_DATE);
					indexType = rs.getInt(IDbConst.INDEX_TYPE);
					indexes.add(new KMIndex(indexId, reindexDate, BOName, indexType, IIndex.STATE_UNDEFINED));
				}
				return (IIndex[]) indexes.toArray(new IIndex[indexes.size()]);
			}
			finally {
				pstmt.close();
			}
		}
		finally {
			conn.close();
		}
	}

	public void updateIndexDate(String BOName, int indexType) throws SQLException {
		Connection conn = m_manager.getConnection();
		try {
			BigDecimal boId = getBusinessObjectId(conn, BOName);
			PreparedStatement pstmt = conn.prepareStatement(IDbConst.INDEX_UPDATE_DATE);
			try {
				Timestamp reindexDate = new Timestamp(System.currentTimeMillis());
				pstmt.setTimestamp(1, reindexDate);
				pstmt.setBigDecimal(2, boId);
				pstmt.setInt(3, indexType);
				pstmt.executeUpdate();
			}
			finally {
				pstmt.close();
			}
		}
		catch (SQLException ex) {
			throw ex;
		}
		finally {
			conn.close();
		}
	}

	/**
	 * @param remoteIdxs
	 * @param localIdxs
	 * @return
	 */
	public IIndex[] merge(IIndex[] remoteIndexes, IIndex[] localIndexes) throws SQLException {
		IIndex remote, local;
		String remoteIndexId, BOName;
		for (int i = 0; i < remoteIndexes.length; i++) {
			remote = remoteIndexes[i];
			remoteIndexId = remote.getName();
			remote.setType(IIndex.TYPE_KM_BO_GLOBAL);
			boolean merged = false;

			for (int j = 0; j < localIndexes.length; j++) {
				local = localIndexes[j];
				if (local == null) {
					continue;
				}
				if (remoteIndexId.equals(local.getName())) {
					local.setState(remoteIndexes[i].getState());
					remoteIndexes[i] = local;
					localIndexes[j] = null;
					break;
				}
			}
		}
		// add unused local indexes to remote
		ArrayList remoteList = new ArrayList(Arrays.asList(remoteIndexes));
		for (int i = 0; i < localIndexes.length; i++) {
			local = localIndexes[i];
			if (local != null && local.getType() == IIndex.TYPE_KM_BO_GLOBAL && local.getBOName() != null) {
				local.setType(IIndex.TYPE_UNDEFINED);
				remoteList.add(local);
				localIndexes[i] = null;
			}
		}

		remoteIndexes = (IIndex[]) remoteList.toArray(new IIndex[remoteList.size()]);
		deleteUnused(localIndexes);
		return remoteIndexes;
	}

	protected void deleteUnused(IIndex[] indexes) throws SQLException {
		Connection conn = m_manager.getConnection();
		try {
			PreparedStatement pstmt = conn.prepareStatement(IDbConst.INDEX_DEL_BY_NAME);
			try {
				IIndex index;
				for (int i = 0; i < indexes.length; i++) {
					index = indexes[i];
					if (index != null) {
						pstmt.setString(1, index.getName());
						pstmt.execute();
					}
				}
				deleteBusinessObjects(conn);
			}
			finally {
				pstmt.close();
			}
		}
		catch (SQLException ex) {
			throw ex;
		}
		finally {
			conn.close();
		}
	}

	public List getIndexId(String BOName, int type) throws SQLException {
		Connection conn = m_manager.getConnection();
		try {
			PreparedStatement pstmt;
			pstmt = conn.prepareStatement(IDbConst.INDEX_SEL_BY_BO_NAME_AND_TYPE);

			pstmt.setString(1, BOName);
			pstmt.setInt(2, type);

			try {
				ResultSet rs = pstmt.executeQuery();
				List indices = new ArrayList();
				while (rs.next()) {
					indices.add(rs.getString(IDbConst.INDEX_NAME));
				}
				return indices;
			}
			finally {
				pstmt.close();
			}
		}
		finally {
			conn.close();
		}
	}

	public boolean checkIndexExist(String BOName, int type) throws SQLException {
		List coll = getIndexId(BOName, type);
		return coll != null && coll.size() > 0;
	}

	public boolean checkAtLeastOneIndexExist(Collection BONames, int type) throws SQLException {
		List coll;
		String BOName;
		for (Iterator i = BONames.iterator(); i.hasNext();) {
			BOName = (String) i.next();
			coll = getIndexId(BOName, type);
			if (coll != null && coll.size() > 0) {
				return true;
			}
		}
		return false;
	}

	public IIndex bless(IIndex remote) throws SQLException {
		Connection conn = m_manager.getConnection();
		try {
			PreparedStatement pstmt;
			pstmt = conn.prepareStatement(IDbConst.INDEX_SEL_BY_NAME);
			String indexName = remote.getName();
			pstmt.setString(1, indexName);
			try {
				ResultSet rs = pstmt.executeQuery();
				if (rs.next()) {
					remote.setBOName(rs.getString(IDbConst.BO_NAME));
					remote.setType(rs.getInt(IDbConst.INDEX_TYPE));
					remote.setCreationDate(rs.getTimestamp(IDbConst.REINDEX_DATE));
				}
				return remote;
			}
			finally {
				pstmt.close();
			}
		}
		finally {
			conn.close();
		}
	}

}
