package com.sap.caf.km.ejb.svc.idxsearch.common;

import java.io.Serializable;
import java.util.Date;

/**
 * The interface defines operands, options and operations for search condition. <br>
 * Notes: <br> 
 * Two functions to define the taxonomy search are added to the methods of the 
 * KM interface IQueryEntry. The rest is identical except some unimportant methods 
 */
public interface ISearchEntry extends Serializable {
	
	/**
	 * Activates the taxonomy search. This parameter must be interpreted first!
	 * Defaults is <code>false</code>
	 * 
	 * @param taxSearch 	The value is <code>true<code> if the search must be done in the corresponding 
	 *  taxonomy folders
	 */
	public void setTaxonomySearch(boolean taxSearch);
	
	/**
	 * Tests "TaxonomySearch" property
	 * @return <code>true</code> if taxonomy search is enabled
	 */
	public boolean isTaxonomySearch();
	
	/** Sets value as <code>String</code>
	 * @param stringValue	The value to be set
	 */  
	public void setValueAsString(String stringValue);
	public String getValueAsString();
	
	/** Sets value as <code>int</code>
	 * @param intValue	The value to be set
	 */  
	public void setValueAsInt(int intValue);
	public int getValueAsInt();

	/** Sets value as <code>Date</code>
	 * @param dateValue	The value to be set
	 */
	public void setValueAsDate(Date dateValue);
	public Date getValueAsDate();

	/** Defines first <b>int</b> range value. 
	 *  Result should be between <code>[intRangeValue1, intRangeValue2]</code>
	 *  @param intValue1 Start range value
	 */	
	public void setRangeValue1stAsInt(int intValue1);
	public int getRangeValue1stAsInt();

	/** Defines second <b>int</b> range value. 
	 *  Result should be between <code>[intRangeValue1, intRangeValue2]</code>
	 *  @param intValue2 End range value
	 */	
	public void setRangeValue2ndAsInt(int intValue2);
	public int getRangeValue2ndAsInt();
	
	/** Defines first <b>Date</b> range value. 
	 *  Result should be between <code>[dateRangeValue1, dateRangeValue2]</code>
	 *  @param dateValue1 Start range value
	 */	
	public void setRangeValue1stAsDate(Date dateValue1);
	public Date getRangeValue1stAsDate();

	/** Defines second <b>Date</b> range value. 
	 *  Result should be between <code>[dateRangeValue1, dateRangeValue2]</code>
	 *  @param dateValue1 Start range value
	 */	
	public void setRangeValue2ndAsDate(Date dateValue2);
	public Date getRangeValue2ndAsDate();

	/** Sets "Row Type" property
	 *  @param rowType A value of Row Type, see static attributes ROW_TYPE_xxx in KM API
	 */	
	public void setRowType(String rowType);

	/** Gets "Row Type" property
	 *  @return Row Type value as defined in KM API 
	 */
	public String getRowType();

	/** Sets "Term Weight" property
	 *  @param weight The value of term weight in range <code>[0.0, 1.0]</code>
	 */	                
	public void setTermWeight(float weight);
	public float getTermWeight();

	/** Sets a term action.  The term action defines how the search is executed (fuzzy, exact, etc.) 
	 *  The method must only be called when the row type is ROW_TYPE_ATTRIBUTE or ROW_TYPE_TERM. 
	 *  @param termAction A value of Term Action, seee static attributes TERM_ACTION_xxx from KM API
	 */	                   
	public void setTermAction(byte termAction);

	/** Gets the value of "Term Action" property
	 * @return the value as defined in static attributes TERM_ACTION_xxx in KM API
	 */
	public byte getTermAction();

	/** Sets a property type. 
	 *  The property type defines the kind of property to search on (string, date, integer, ...).
	 *  It must only be called when the row type is ROW_TYPE_ATTRIBUTE. 
	 *  The default value is PROPERTY_TYPE_STRING.
	 *  @param propertyType The value of Property Ttype, see static attributes PROPERTY_TYPE_xxx in KM API
	 */
	public void setPropertyType(byte propertyType);
	public byte getPropertyType();

	/** 
	 * Defines attribute the search has to be executed on, if an attribute search is executed.
	 * Property namespace must be also specified.
	 * @param propertyName The name of the property being searched
	 * @see #setPropertyNamespace(String nameSpace);
	 */	             
	public void setPropertyName(String propertyName);
	public String getPropertyName();

	/** 
	 * Defines attribute namespace for attribute the search has to be executed on, 
	 * if an attribute search is executed. Property name must be also specified.
	 * @param propertyName The name of the property being searched
	 * @see #setPropertyName(String nameSpace);
	 */	             	
	public void setPropertyNamespace(String nameSpace);
	public String getPropertyNamespace();	

	/** Sets the property operator defining the context of the entry value. 
	 *  It is only needed when the entry type is "Row". See KM API <code>IQueryEntry.ROW_TYPE_ATTRIBUTE</code>. 
	 *  Use constants beginning with <code>IQueryEntry.PROPERTY_OPERATOR_</code>. 
	 *  Default is IQueryEntry.PROPERTY_OPERATOR_EQUAL.
	 *  @param propertyOperator The value of the property operator
	 */	
	public void setPropertyOperator(byte propertyOperator);
	public byte getPropertyOperator();
	
	/**
	 * Sets "Fuzziness" property. Must be used for fuzzy search. 
	 * If no fuzziness is set the default fuzziness for the index is used. 
	 * The default fuzziness can be set via the index administration
	 * @param fuzziness 	The value is between 0.0 and 1.0 where 1.0 is exact search
	 */	
	public void setFuzziness(float fuzziness);

	/**
	 * Gets "Fuzziness" property, that has been set for fuzzy search
	 * @return fuzziness
	 */
	public float getFuzziness();
	
	/**
	 * Get value type.
	 * Valid values: <code>IKMSearchEntry.VALUE_TYPE_INT, IKMSearchEntry.VALUE_TYPE_STRING, 
	 * IKMSearchEntry.VALUE_TYPE_UNDEFINED</code>
	 * 
	 * @return Value type
	 */
	public int getValueType();
	
	/**
	 * Set value type. Valid values: <code>IKMSearchEntry.VALUE_TYPE_INT, IKMSearchEntry.VALUE_TYPE_STRING, 
	 * IKMSearchEntry.VALUE_TYPE_UNDEFINED</code>
	 * 
	 * @param type Value type
	 */
	public void setValueType(int type);

	/** linguistic search */
	public static final byte TERM_ACTION_LINGUISTIC        = 1;
	/** fuzzy search */
	public static final byte TERM_ACTION_FUZZY             = 2;
	/** near with search */
	public static final byte TERM_ACTION_NEAR_WIDTH        = 3;
	/** exact search */
	public static final byte TERM_ACTION_EXACT             = 4;
	/** soundex search */
	public static final byte TERM_ACTION_SOUNDEX           = 5;

	/** if query entry is an attribute entry you can specifiy the property type*/
	public static final byte PROPERTY_TYPE_USE_META_MODEL = 0;
	/** if query entry is an attribute entry you can specifiy the property type*/
	public static final byte PROPERTY_TYPE_STRING = 1;
	/** if query entry is an attribute entry you can specifiy the property type*/
	public static final byte PROPERTY_TYPE_DATE = 2;
	/** if query entry is an attribute entry you can specifiy the property type*/
	public static final byte PROPERTY_TYPE_INTEGER = 3;
	/** if query entry is an attribute entry you can specifiy the property type*/
	public static final byte PROPERTY_TYPE_TEXT = 4;

	/** specifies that query entry is a term entry */
	public static final String ROW_TYPE_TERM      = "TERM";
	/** specifies that query entry is an attribute entry */
	public static final String ROW_TYPE_ATTRIBUTE = "ATTRIBUTE";
	/** specifies that query entry is an operator entry */
	public static final String ROW_TYPE_OPERATOR  = "OPERATOR";
	/** specifies that query entry is an open bracket entry */
	public static final String ROW_TYPE_BRACKET_OPEN  = "BRACKET_OPEN";
	/** specifies that query entry is an close bracket entry */
	public static final String ROW_TYPE_BRACKET_CLOSE  = "BRACKET_CLOSE";

	/** if query entry is an operator entry the value can be OPERATOR_AND, OPERATOR_OR or OPERATOR_NOT */
	public static final String OPERATOR_AND = "AND";
	/** if query entry is an operator entry the value can be OPERATOR_AND, OPERATOR_OR or OPERATOR_NOT*/
	public static final String OPERATOR_OR = "OR";
	/** if query entry is an operator entry the value can be OPERATOR_AND, OPERATOR_OR or OPERATOR_NOT */
	public static final String OPERATOR_NOT = "NOT";

	/** Property value should be equal to the value of the query entry */
	public static final byte PROPERTY_OPERATOR_EQUAL = 0;
	/** Property value should not be equal to the value of the query entry */
	public static final byte PROPERTY_OPERATOR_NOT = 1;
	/** Property value should be between value 1 and value 2 of the query entry */
	public static final byte PROPERTY_OPERATOR_BETWEEN = 2;
	/** Property value should be greater than the value of the query entry */
	public static final byte PROPERTY_OPERATOR_GREATER = 3;
	/** Property value should be less than the value of the query entry */
	public static final byte PROPERTY_OPERATOR_LESS = 4;
	/** Property value should be greater or equal the value of the query entry */
	public static final byte PROPERTY_OPERATOR_GREATER_OR_EQUAL = 5;
	/** Property value should be less or equal the value of the query entry */
	public static final byte PROPERTY_OPERATOR_LESS_OR_EQUAL = 6;
	
	/** type of entry's value or entry's ranges */
	public static final int VALUE_TYPE_INT = 1;
	
	/** type of entry's value or entry's ranges */
	public static final int VALUE_TYPE_DATE = 2;
	
	/** type of entry's value or entry's ranges */
	public static final int VALUE_TYPE_STRING = 3;
	
	/** type of entry's value or entry's ranges */
	public static final int VALUE_TYPE_UNDEFINED = 0;
	
	/** Namespace for custom properties */
	public static final String NAMESPACE_CUSTOM_PROPERTY = "http://sapportals.com/xmlns/cm/custom";	
}