package com.sap.caf.km.ejb.svc.idxsearch.common;

/**
 * This interface provides KM index administration functionality on CAF side.
 * 
 * @version 1.0
 * @author Viachaslau_Kudzinau@epam.com 
 * @author Aliaksandr_Pranko@epam.com
 */
public interface IIndexSearchIndexSvc {
	
	/**
	* This method checks if specified index exists. 
	* 
	* @param indexId 				KM index id 
	* @return 					<b>True<b/> if index exists otherwise <b>False</b>
	* @exception IndexException 	if something goes wrong 
	*/	
	public boolean existsIndex(String indexId) throws IndexException;
	
	/**
	 * This method gets index by it KM ID.
	 * 
	 * @param indexId				The index id
	 * @return 						Instance of index object
	 * @throws IndexException
	 */
	public IIndex getIndex(String indexId) throws IndexException;

	/**
	 * This method clears index content
	 * 
	 * @param indexId				The index id
	 * @return 						Instance of index object
	 * @throws IndexException
	 */
	public void clearIndex(String indexId) throws IndexException;
	
	/**
	 * This method returns list of all KM indexes.
	 * 
	 * @return	List of all KM indexes
	 * @throws  IndexException
	 */
	public IIndex[] getAllIndexes() throws IndexException;

	/**
	 * Creates an index for documents contained in one of the folders given
	 * Required to create a CAF business object index: All BO instances are persisted
	 * under one folder.
	 * 
	 * @param indexId				The index Id
	 * @param folderRids			Folders' RIDs to be indexed
	 * @throws IndexException
	 */
	public void createFolderIndex(String indexId, String[] folderRids) throws IndexException;

	/**
	 * The method creates an index for EP BO (tasks and documents) related to CAF 
	 * business objects  by attachment service or via relation service and has 
	 * to invoke a first initial indexing, this means that all EP tasks and documents 
	 * related to CAF BO would be found and attached to specified index.<br>
	 * Such index should be created for each CAF BO type.<br>
	 * The type of CAF BO for which related index should be created come in from index name.
	 * The index name pattern is: CAF_<i>BOName</i>_Index_Related.  
	 * 
	 * @see #reindexRelated(String)
	 * @param indexId				The KM index Id.
	 * @param mofId					GUID of CAF BO type for which index will be created  
	 * @throws IndexException
	 */
	public void createRelatedIndex(String indexId, String mofId) throws IndexException;
	
	/**
	 * The method deletes specified index.
	 * 
	 * @param indexId				The KM index id
	 * @throws IndexException	
	 */		
	public void deleteIndex(String indexId) throws IndexException;

	/**
	 * The method adds a list of objects to the indexing queue.
	 * 
	 * @param indexId				The KM index id
	 * @param objectsRids			The array of objects' RIDs 
	 * @throws IndexException
	 */
	public void indexObjects(String indexId, String[] objectsRids) throws IndexException;
	
	/**
	 * The method removes a list of objects from indexing queue. 
	 * Required if object is deleted
	 * 
	 * @param indexId				The KM index id
	 * @param objectsRids			The array of objects' RIDs
	 * @throws IndexException
	 */
	public void deindexObjects(String indexId, String[] objectsRids) throws IndexException;
	
	/**
	 * This method performs reindexing of specified index.
	 * 
	 * @param indexId				The KM index id
	 * @throws IndexException		If something goes wrong
	 */	
	public void reindex(String indexId) throws IndexException;
	
	/**
	 * This method performs recreation of index which is created for EP BO (task & documents) 
	 * related to CAF business objects by attachment service or via relation service.	 
	 * The type of CAF BO for which related index should be created come in from index name.
	 * The index name pattern is: CAF_<i>BOName</i>_Index_Related.  
	 * 
	 * @see #createRelatedIndex(String)
	 * @param indexId				The KM index id
	 * @param mofId					GUID of CAF BO type for which index will be rebuilded  
	 * @throws IndexException		If something goes wrong
	 */
	public void reindexRelatedIndex(String indexId, String mofId) throws IndexException;

	
	/**
	 * The method updates specified index for new and changed objects.
	 * 
	 * @param indexId				The KM index id
	 * @throws IndexException		If something goes wrong
	 */
	public void deltaReindex(String indexId) throws IndexException;	
}
