package com.sap.caf.km.ejb.svc.idxsearch.common;

/**
 * This interface provides KM classification and taxonomy functionality on CAF side.
 * 
 * @version 1.0
 * @author Viachaslau_Kudzinau@epam.com 
 * @author Aliaksandr_Pranko@epam.com
 */
public interface IIndexSearchClassifySvc {
		
	/**
	 * The method creates a folder in the repository /taxonomy in relation
	 * to an index (uses the same method in KM API)
	 * 
	 * @param indexID	The KM Index Id
	 * @param taxName   The taxonomy name
	 */
	public void addTaxonomy(String indexID, String taxName) throws ClassifyException;
	
	/**
	 * The method deletes the taxonomy from the repository and from the index
	 * (uses the same method in KM API)
	 *  
	 * @param indexID	The KM Index Id
	 * @param taxName   The taxonomy name
	 */	
	public void removeTaxonomy(String indexID, String taxName) throws ClassifyException;

	/**
	 * Assign resources to a category in a certain taxonomy
	 * (see first parameter of categoryRID):
	 * 1) check, if the resourceLink is in toClassify folder
	 * 2) if yes –-> move the resourceLink to the category
	 * 3) if not --> find resourceLink in taxonomy and copy the link to the new category
	 *  
	 * @param resourceRIDList	List of ResourceRID for classification
	 * @param categoryRID		Category RID
	 */
	public void classifyResources(String[] resourceRIDList, String categoryRID) throws ClassifyException;

	/**
	 * Assign resources to a category in a certain taxonomy
	 * (see first parameter of categoryRID):
	 * 1) check, if the resourceLink is in toClassify folder
	 * 2) if yes –-> move the resourceLink to the category
	 * 3) if not --> find resourceLink in taxonomy and copy the link to the new category
	 *  
	 * @param resourceRID		Resource for classification
	 * @param categoryRID		Category RID
	 */
	public void classifyResource(String resourceRID, String categoryRID) throws ClassifyException;

	/**
	 * Assign resources to a category in a certain taxonomy
	 * (see first parameter of categoryRID):
	 * 1) check, if the resourceLink is in toClassify folder
	 * 2) if yes –-> move the resourceLink to the first category and copy the resourceLink to the other categories
	 * 3) if not --> find resourceLink in taxonomy and copy the link to the new category
	 * 
	 * @param resourceRID
	 * @param cate
	 */
	public void classifyResource(String resourceRID, String[] categoryRIDList) throws ClassifyException;

	/**
	 * Gets the prefix of the used taxonomy repository
	 * 
	 * @return	The prefix of the used taxonomy repository
	 */
	public String getTaxonomyRepositoryPrefix() throws ClassifyException; 	

	/**
	 * The method deletes the resourceLink from the categories.
	 * If the resource is not categorized anymore, the resourceLink 
	 * must be moved to the special folder “toClassify” at last
	 * 	 
	 * @param categoryRIDList	List of categories to remove from
	 * @param resourceRID		Resource Id to remove
	 */
	public void removeFromCategories(String[] categoryRIDList, String resourceRID) throws ClassifyException;
 	
	/**
	 * The method returns all categories for an object in a certain taxonomy.
	 * The result String will be composed of the different levels
	 * like: /taxonomyName/level1/level2/category1
	 * 
	 * @param taxName		The name of taxonomy
	 * @param resourceRID	The object identifier on KM side	
	 */
	public ICategory[] getObjectClassification(String taxName, String resourceRID) throws ClassifyException;
	
	/**
	 * The method returns all categories for an object in all assigned taxonomies.
	 * The result ICategory is an array, containing all necessary
	 * informations like taxonomy name and categoryRID (represents
	 * the path to the categorized folder)
	 * The categoryRID is composed of the different levels
	 * like: /taxonomyName/level1/level2/category1
	 *
	 * @param resourceRID	The object identifier on KM side
	 * @return 				Array of categories
	 */
	public ICategory[] getObjectClassification(String resourceRID) throws ClassifyException;
	
	/**
	 * The method gets the special folder name. 
	 * 
	 * @param folderId		The special folder id
	 * @return		The special folder name
	 * @throws ClassifyException
	 */
	public String getSpecialFolderName(int folderId) throws ClassifyException;
	
	/**
	 * The method gets the array of taxonomyes in which specified resource is included.
	 * 
	 * 
	 * @param resourceRid		The KM resource RID
	 * @return			The array of taxonomyes
	 * @throws ClassifyException
	 */
	public ITaxonomy[] getObjectTaxonomy(String resourceRid) throws ClassifyException;


	/**
	 * The method creates categories in specified taxonomy
	 * 
	 * @param taxonomyRid		The taxonomy RID
	 * @param categoryArray		The Array of categories for creation
	 * @throws ClassifyException
	 */
	public void createCategories(String[] categoryArray) throws ClassifyException;

	/**
	 * The method creates categories in specified taxonomy
	 * 
	 * @param taxonomyRid		The taxonomy RID
	 * @param categoryArray		The Array of categories for creation
	 * @throws ClassifyException
	 */	
	public void deleteCategories(String[] categoryArray) throws ClassifyException;

	/**
	 * The method set up description for specified categories
	 * 
	 * @param taxonomyRid		The taxonomy RID
	 * @param categoryArray		The Array of categories for creation
	 * @throws ClassifyException
	 */		
	public void setCategoryDescription(String description, String[] categoryArray) throws ClassifyException;


	/**
	 * The method returns the array of unclassified objects
	 * 
	 * @param taxonomyRid
	 * @return
	 * @throws ClassifyException
	 */
	public String[] getUnclassifiedObjects(String taxonomyRid) throws ClassifyException;
		
		
	public String[] getChildren(String categoryRid, boolean includeOnlyFolders) throws ClassifyException;
	
	public boolean hasChildren(String categoryRid) throws ClassifyException;
		
	/**
	 * ID of "to classify" special folder
	 */
	public static final int TO_CLASSIFY_FOLDER = 0;
	
	/**
	 * ID of "never to classify" special folder
	 */
	public static final int NEVER_TO_CLASSIFY_FOLDER = 1;	
}
