package com.sap.caf.km.ejb.svc.idxqueue;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import javax.ejb.MessageDrivenBean;
import javax.ejb.MessageDrivenContext;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.naming.Context;
import javax.naming.InitialContext;

import com.sap.caf.km.ejb.svc.idxsearch.IIndexSearchIndex;
import com.sap.caf.km.ejb.svc.idxsearch.bean.IndexSearchLocalHome;
import com.sap.caf.metamodel.Application;
import com.sap.caf.metamodel.DataObject;
import com.sap.caf.rt.exception.CAFBaseException;
import com.sap.caf.rt.metamodel.MetaModel;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * 
 * @author D040882
 *
 * IndexQueueBean acts as an subscriber to changes of business objects. For this, it is implemented as
 * MessageDrivenBean which is notified by the PublisherBean of caf runtime. It is used to update the index
 * of the business object in trex if a business object was created, changed or removed. Additionally, if the
 * business object is persisted in km, the related index of the business object.
 */
public class IndexQueueBean implements MessageDrivenBean, MessageListener {
  
  private static final String APPLICATION = IndexQueueBean.class.getName();
  private static final String jARMReqPrefix = "CAF:RT:oal:"; 
  private static final String JARM_REQUEST =  jARMReqPrefix + APPLICATION;    
  private static final Location location = Location.getLocation(APPLICATION);

  private MessageDrivenContext myContext;
  private final static String jndiIndexSearchBean = "localejbs/sap.com/caf~com.sap.caf.runtime/IndexSearchBean";
  private final static String separator = "%$%";
  private final static String CHANGE = "BUSINESS_OBJECT_CHANGED";
  private final static String REMOVE = "BUSINESS_OBJECT_REMOVED";
  private final static String CREATE = "BUSINESS_OBJECT_CREATED";

  /**
   * Is called by jms container if a business object had been changed. msg contains all necessary information about
   * the business object to update the required index.
   */
  public void onMessage(Message msg) {
    String method = JARM_REQUEST + ":onMessage(Message)";
    enter(method, new Object[] { msg });
    try {
      //extract all information which is required to update the index 
      Map map = new HashMap();
      Enumeration enum = msg.getPropertyNames();
      while (enum.hasMoreElements()) {
        String name = (String) enum.nextElement();
        String value = msg.getStringProperty(name);

        map.put(name, value);
      }

      Collection col = new ArrayList();
      String usedby = (String) map.get("USEDBY");
      StringTokenizer tok2 = new StringTokenizer(usedby, ";");
      while (tok2.hasMoreTokens()) {
        String tmp = tok2.nextToken();
        col.add(tmp);
      }

      //call the appropriate method indicated by the notification information
      if (msg.getStringProperty("ACTION").equals(CHANGE)) {
        put((String) map.get("RID"), (String) map.get("OBJTYPE"), col, 1);
      }
      else if (msg.getStringProperty("ACTION").equals(CREATE)) {
        put((String) map.get("RID"), (String) map.get("OBJTYPE"), col, 1);
      }
      else if (msg.getStringProperty("ACTION").equals(REMOVE)) {
        put((String) map.get("RID"), (String) map.get("OBJTYPE"), col, 0);
      }
    }
    catch (Exception ex) {
      CAFBaseException indexqEx = new CAFBaseException(ex);
      log(indexqEx, method, null);
    }
    finally {
      exit(method, null);
    }
  }

  public void ejbRemove() {
  }

  public void setMessageDrivenContext(MessageDrivenContext context) {
    myContext = context;
  }

  public void ejbCreate() {
  }

  /**
   * Updates the index of the business object in trex.
   * @param rid km rid of the object that has to be updated
   * @param objType type of the business object that had been changed (e.g. "sap.com/xapps.xpd/Idea")
   * @param usedBy rids of all objects that have a reference to the changed business object. is only usefull if
   *         parameter rid points to a km business object
   * @param bReindex indicates if the object has to be added to/updated in the index or if it has to be removed from the index 
   * @throws BOException
   */
  private synchronized void put(String rid, String objType, Collection usedBy, int bReindex) throws CAFBaseException {
    String method = JARM_REQUEST + ":put(String, String, Collection, int)";
    enter(method, new Object[] { rid, objType, usedBy, new Integer(bReindex)});
    try {
      IIndexSearchIndex bean = null;
      Context ctx = new InitialContext();
      IndexSearchLocalHome home = (IndexSearchLocalHome) ctx.lookup(jndiIndexSearchBean);
      bean = home.create();

      if (bReindex == 1) {
        //object has to be reindex in its own index 
        bean.indexObject(objType, rid);
      }
      else if (bReindex == 0) {
        //object has to be removed from its own index 
        bean.deindexObject(objType, rid);
      }

      if ((usedBy != null) && (usedBy.size() > 0)) {
        updateIndexByRelatedObject(usedBy, rid, bReindex, bean);
      }
    }
    catch (Exception ex) {
      CAFBaseException indexqEx = new CAFBaseException(ex);
      log(indexqEx, method, null);
      throw indexqEx;
    }
    finally {
      exit(method, null);
    }
  }

  /**
   * Updates the related index of a business object. For this, the business object has to be persisted in km.
   * @param usedBy rids of all objects which have a reference to the changed km business object 
   * @param objRid rid of the km object which had been changed
   * @param bReindex indicates if the object has to be added to/updated in the index(true) or if it has to be 
   *          removed from the index
   * @param bean local interface of the IndexSearchBean which calls the required trex functionality 
   * @throws BOException
   */
  private void updateIndexByRelatedObject(Collection usedBy, String objRid, int bReindex, IIndexSearchIndex bean)
    throws CAFBaseException {
    String method = JARM_REQUEST + ":updateIndexByRelatedObject(Collection, String, int, IIndexSearchIndex)";
    enter(method, new Object[] { usedBy, objRid, new Integer(bReindex), bean });
    try {
      // evaluate a set of BOs in whose related indices objRid will be added
      Set objNames = new HashSet();
      MetaModel model = new MetaModel();
      String rid, sMofId, fullBOName;
      int k, m;
      for (Iterator i = usedBy.iterator(); i.hasNext();) {
        rid = (String) i.next();
        if (rid.length() == 0)
          continue;
        // extract MofId
        k = rid.indexOf('/', 1);
        if (k == -1)
          continue;
        m = rid.indexOf('/', ++k);
        if (m == -1)
          continue;
        sMofId = rid.substring(k, m);
        // compose full BO name
        if (!objNames.contains(sMofId)) {
                  DataObject dataObj = model.getDataObjectByGUID(sMofId);
                  Application appl = dataObj.getApplication();
                  fullBOName = appl.getProviderName()+"/"+appl.getName()+"/"+dataObj.getName();
                  objNames.add(fullBOName);
        }
      }
      if (objNames.isEmpty()) {
        return;
      }

      for (Iterator i = objNames.iterator(); i.hasNext();) {
        fullBOName = (String) i.next();

        if (bReindex == 1) {
          //object has to be reindex in its own index 
          bean.indexRelatedObject(fullBOName, objRid);
        }
        else if (bReindex == 0) {
          //object has to be removed from its own index 
          bean.deindexRelatedObject(fullBOName, objRid);
        }
      }
    }
    catch (Exception ex) {
      CAFBaseException indexqEx = new CAFBaseException(ex);
      log(indexqEx, method, null);
      throw indexqEx;
    }
    finally {
      exit(method, null);
    }
  }

  protected void log(CAFBaseException e, String method, Object[] args) {
    CAFPublicLogger.logThrowable(Severity.ERROR, CAFPublicLogger.categoryCAF, getLocation(), method, e.getMessage(), e);
  }

  protected void info(String method, String message, Object[] args) {
    CAFPublicLogger.logThrowable(Severity.INFO, CAFPublicLogger.categoryCAF, getLocation(), method, message, null);
  }

  protected void enter(String method, Object[] args) {
    CAFPublicLogger.entering(null, getJARMRequest(), method, getLocation(), args, CAFPublicLogger.LEVEL_MEDIUM);
  }

  protected void exit(String method, Object result) {
    CAFPublicLogger.exiting(null, getJARMRequest(), method, getLocation(), result, CAFPublicLogger.LEVEL_MEDIUM);
  }

  protected Location getLocation() {
    return location;
  }

  protected String getJARMRequest() {
    return JARM_REQUEST;
  }
  

}