package com.sap.caf.km.da;

/**
 * @author viachaslau_kudzinau@epam.com
 */
public final class RidUtils {

	/**
	 * Gets name part of a rid. 
	 * If there is no '/' chars in the rid return rid itself
	 * @param rid 	A rid
	 * @return Name part of the rid
	 */
	public static String getName(String rid) {
		int i = rid.lastIndexOf(PATH_SEPARATOR);
		return (i==-1? rid: rid.substring(i+1));
	}

	/**
	 * Gets path part of a rid. 
	 * If there is no '/' chars in the rid return rid itself
	 * @param rid 	A rid
	 * @return Path part of the rid
	 */
	public static String getPath(String rid) {
		int i = rid.lastIndexOf(PATH_SEPARATOR);
		return (i==-1? rid: rid.substring(0, i));
	}
	

	/**
	 * Tests whether given Rid has BO root as its parent 
	 * @param rId A Rid to be tested 
	 * @return true if BO root folder is a prefix of given rid 
	 */
	public static boolean isUnderBORootFolder(String rId) {
		return getBORootFolder(KMDataAccessHelper.getDocRepositoryRid(), rId)!=null;	
	}
	
	/**
	 * Tests whether given Rid has Temp folder as its parent 
	 * @param rId A Rid to be tested 
	 * @return true if Temp folder is a prefix of given rid 
	 */
	public static boolean isUnderTempFolder(String rId) {
		return rId.startsWith(KMDataAccessHelper.getTempFolderRid());
	}
	
	/**
	 * Gets BO root folder part from specified rid.
	 * @param docRootRid	CAF documents root repository rid
	 * @param rid		A rid to be processed
	 * @return BO root part, <code>null</code> rid has invalid format
	 */
	public static String getBORootFolder(String docRootRid, String rid) {
		int i;
		// find doc repository part
		i = rid.indexOf(docRootRid);
		if (i!=0) {
			return null;
		}
		// find full BO name part
		i = rid.indexOf(PATH_SEPARATOR, i+docRootRid.length()+1);
		if (i==-1) {
			return null;
		}
		// find BO key part having short BO name as a prefix
		i = rid.indexOf(PATH_SEPARATOR, i+1);
		if (i==-1) {
			return null; // we assume that it has to be one level after at least
		}
		return rid.substring(0, i);
	}
	
	/** 
	 * Merges pathname given as an array of strings
	 * @param pathname A pathname to be merged
	 * @return merged pathname as string with delimiter <code>STR_DIVIDER</code>
	 */
	public static String mergePathname(String[] pathname, int count) {
		if (pathname == null || pathname.length == 0 || count < 0 || count > pathname.length) {
			return null;
		}
		StringBuffer sb = new StringBuffer((pathname[0].length() + 1) * count);
		for (int i = 0; i < count; i++) {
			if (pathname[i] != null && pathname[i].length() > 0) {
				sb.append(pathname[i]).append(PATH_SEPARATOR);
			}
		}

		if (sb.length() > 0) {
			sb.setLength(sb.length() - 1);
		}

		return sb.toString();
	}
	
	/**
	 * Splits pathname into three parts: root path, extra path information and file name
	 * @param rootPath	A root path which should be present at the beginning of <code>pathname</code>
	 * @param pathname	A pathname to be splitted
	 * @return	An array of three parts: root path, extra path and file name
	 */
	public static String[] splitPathname(String rootPath, String pathname) {
		int i = pathname.indexOf(rootPath);
		if (i != 0) {
			return null;
		}
		int j = rootPath.length();
		String[] res = new String[3];
		// gets root path
		res[0] = pathname.substring(i, j);
		pathname = pathname.substring(j + 1);
		// gets extra path info
		j = pathname.lastIndexOf(PATH_SEPARATOR);
		res[1] = (j == -1 ? null : pathname.substring(0, j));
		// gets file name
		res[2] = (j == -1 ? pathname : pathname.substring(j + 1));

		return res;
	}

	/**
	 * Transforms source root folder path and source file pathname into
	 * resulted root folder and file name moving extra path information to resulted root folder name 
	 * i.e.
	 * operands: "/documents/temp" and "countries/US/states/NY/report.doc" will be transformed into
	 * results: "/documents/temp/countries/US/states/NY" and "report.doc"
	 *  
	 * @param folderPath 	 Source root folder
	 * @param filePathname Source pathname probably containing extra path information
	 * @return An array of two elements: 0 is a root folder and 1 is a file name
	 */
	public static String[] normalizePathname(String folderPath, String filePathname) {
		if (folderPath == null || filePathname == null) {
			throw new RuntimeException("Method parameters can not be null!");
		}
		int i = filePathname.lastIndexOf(PATH_SEPARATOR);
		if (i > 0) {
			folderPath = folderPath + PATH_SEPARATOR + filePathname.substring(0, i);
			filePathname = filePathname.substring(i + 1);
		}
		return new String[] { folderPath, filePathname };
	}

	public final static char PATH_SEPARATOR = '/';
}
