package com.sap.caf.km.da;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import com.sap.caf.rt.bol.util.UserContext;
import com.sap.caf.km.ejb.data.util.KMNodeException;
import com.sap.caf.km.ejb.data.util.NodeContent;
import com.sap.caf.km.ejb.data.util.NodeHeader;
import com.sap.caf.km.proxies.data.kmnode.types.KMNodeHeader;
import com.sap.caf.rt.bol.IDependentObject;
import com.sap.caf.rt.bol.da.DataAccessFactory;
import com.sap.caf.rt.bol.da.IDataAccessService;
import com.sap.caf.rt.bol.pk.PrimaryKeyFactory;
import com.sap.caf.rt.bol.util.IntQueryFilter;
import com.sap.caf.rt.bol.util.QueryFilter;
import com.sap.caf.rt.exception.CAFFindException;
import com.sap.caf.rt.exception.DataAccessException;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

public class DocumentDataAccessService implements IDataAccessService {
	
	private UserContext userContext = null;
	private static DocumentDataAccessService service;
	
	private static final String APPLICATION = DocumentDataAccessService.class.getName();
	private static final String jARMReqPrefix = "CAF:RT:oal:"; 
	private static final String JARM_REQUEST =  jARMReqPrefix + APPLICATION;		
	private static final Location location = Location.getLocation(APPLICATION);	
		
	private IDataAccessService jdoDataAccessService=null;
	
	//Name of DocumentContent Object
	private String m_sDocContentName = "sap.com/caf.core/DocumentContent";
	
	private DocumentDataAccessService() throws DataAccessException {
		String method=JARM_REQUEST + ":activate()";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, new Object[0], CAFPublicLogger.LEVEL_MEDIUM);

		try {
			jdoDataAccessService = DataAccessFactory.getDataAccessService(DataAccessFactory.DATASOURCE_LOCAL);			
		}
		catch(Exception ex) {
			CAFPublicLogger.logThrowable(Severity.ERROR, CAFPublicLogger.categoryCAF, 
											location, method, "Cannot get instance of document proxy bean", ex);
			throw new DataAccessException("Cannot get instance of document proxy bean", new Object[0]);
		} finally {
			CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, new Object[0], CAFPublicLogger.LEVEL_MEDIUM);
		}
	}
	
	/**
	 * Get instance of DocumentDataAccessService
	 *  
	 * @return DocumentDataAccessService
	 * @throws DataAccessException
	 */
	public static DocumentDataAccessService getInstance() throws DataAccessException {
		String method = JARM_REQUEST + ":getInstance()";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);					
		if (service == null)
			service = new DocumentDataAccessService();
		CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, new Object[] {service}, CAFPublicLogger.LEVEL_MEDIUM);			
		return service;
	}
	
	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#destroy()
	 */
	public void destroy() {
		String method = JARM_REQUEST + ":destroy()";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, new Object[0], CAFPublicLogger.LEVEL_DETAIL);		
		jdoDataAccessService = null;		
		CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_DETAIL);		
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#create(com.sap.caf.rt.bol.IDependentObject)
	 */
	public String create(IDependentObject arg0) throws DataAccessException {
		String method = JARM_REQUEST + ":create(IDependentObject)";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, new Object[] { arg0.getKey()}, CAFPublicLogger.LEVEL_MEDIUM);

		try {
				//store into JDO
				jdoDataAccessService.create(arg0);

				//store into KM
				KMDataAccessHelper.getDocumentProxyService()
					.createDocumentHeader((String)arg0.getProperty("documentId"),
											(String)arg0.getProperty("parentFolder"),
											(String)arg0.getProperty("title"),
											(String)arg0.getProperty("description"),
											(String)arg0.getProperty("link"),
											(Collection)arg0.getProperty("relatedObjects"));
			
				//store document content
				if (m_sDocContentName.equals(arg0.getObjectType())) {
					uploadDocumentContent(arg0);
				}
				
				return arg0.getKey();											
		} catch (Exception ex) {
				CAFPublicLogger.LOC_CAF.catching(ex);
				throw new DataAccessException(ex);	
		} finally {
				CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
		}		
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#remove(com.sap.caf.rt.bol.IDependentObject)
	 */
	public void remove(IDependentObject arg0) throws DataAccessException {
		String method = JARM_REQUEST + ":remove(IDependentObject)";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, new Object[] { arg0}, CAFPublicLogger.LEVEL_MEDIUM);
		
		try {
			String documentId = (String)arg0.getProperty("documentId");
			String parentFolder = (String)arg0.getProperty("parentFolder");
			
			//Load Document if only KEY specified
			if (documentId==null && parentFolder==null) {
				arg0 = (IDependentObject)load(arg0.getPK(), arg0.getClass());
				documentId = (String)arg0.getProperty("documentId");
				parentFolder = (String)arg0.getProperty("parentFolder");				
			}
			
			//delete from KM
			KMDataAccessHelper.getDocumentProxyService()
				.deleteDocument(documentId, parentFolder);
			
			jdoDataAccessService.remove(arg0);	
		}
		catch(Exception ex) {
			location.catching(ex);
			throw new DataAccessException(ex);		
		} finally {
			CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
		}	
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#findByPrimaryKey(java.lang.Object, java.lang.Class)
	 */
	public Object findByPrimaryKey(Object arg0, Class arg1) 
		throws DataAccessException {
		String method = JARM_REQUEST + ":findByPrimaryKey(Object, Class)";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, new Object[] { arg0, arg1}, CAFPublicLogger.LEVEL_MEDIUM);
					
		try {
			IDependentObject obj = (IDependentObject) jdoDataAccessService.findByPrimaryKey(arg0, arg1);		
			return loadObjectFromKM(obj);	
		} catch (Exception ex) {
			location.catching(ex);
		} finally {
			CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM); 
		}
		return null;
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#query(java.lang.Class)
	 */
	public Collection query(Class arg0, String operationName) throws DataAccessException {
		return loadObjectsFromKM( jdoDataAccessService.query(arg0, operationName) );
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#query(java.lang.Class, java.lang.String, java.lang.String, java.util.Map)
	 */
	public Collection query(Class arg0, String arg1, String arg2, Map arg3, String operationName)
		throws DataAccessException {
		return loadObjectsFromKM( jdoDataAccessService.query(arg0, arg1, arg2, arg3, operationName) );
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#query(java.lang.Class, com.sap.caf.rt.bol.util.IntQueryFilter)
	 */
	public Collection query(Class arg0, IntQueryFilter arg1, String operationName)
		throws DataAccessException {
		return loadObjectsFromKM( jdoDataAccessService.query(arg0, arg1, operationName) );
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#query(java.lang.Class, com.sap.caf.rt.bol.util.IntQueryFilter[])
	 * 
	 * The specified search is implemented for parentFolder/documentId 
	 * and parentFolder/documentTitle IntQueryFilter pairs
	 * 
	 * For parentFolder/documentId pair search in CAF JDO is performed and if nothing
	 * is found then search on KM is performed. If document is found on KM it 
	 * will be created in CAF JDO <br>
	 * For parentFolder/documentTitle pair search in KM is performed then
	 * for each found document method performs search in CAF JDO by documentId and 
	 * parentFolder attributes. If representation of KM document isn't found in 
	 * CAF JDO it will be created otherwice found Document KEY is used 
	 */
	public Collection query(Class arg0, IntQueryFilter[] arg1, String operationName) throws DataAccessException {
		
		if (arg1.length>2) {
			return loadObjectsFromKM( jdoDataAccessService.query(arg0, arg1, operationName) );
		}
		
		Collection result=null;
		String documentId=null;
		String parentFolder=null;
		String documentTitle=null;
		
		for (int i=0; i<arg1.length; i++) {
			arg1[i].getJdoMapFilter();
			String attr =  arg1[i].attribute;			
			if ("documentId".equals(attr)) {
				documentId = arg1[i].valueLow;
			} else if ("parentFolder".equals(attr)) {
				parentFolder = arg1[i].valueLow;
			} else if ("title".equals(attr)) {
				documentTitle = arg1[i].valueLow;
			} else 
					break;		
		}
		
		if (documentId!=null && parentFolder!=null) {
		//search for document using documentId and parentFolder
			
			//search for document in JDO
			result = loadObjectsFromKM( jdoDataAccessService.query(arg0, arg1, operationName) );
			
			//if nothing is found try to search Document on KM
			if (result.size()==0) {
				try {
						IDependentObject doc = createDependentObject(arg0, documentId, parentFolder, null);
						
						doc = (IDependentObject)loadObjectFromKM(doc);

						//create document in JDO						
						if (doc!=null) {							
							createInJDO(doc);						
						}
											
						result.add(doc);
				} catch (Exception ex) {
					location.throwing(ex);
				}
			}
		} else if (documentTitle!=null && parentFolder!=null) {
		//search for document using parentFolder and document title			
			
			try {
				result = new ArrayList();
				Collection km_result = KMDataAccessHelper.getDocumentProxyService().findDocument(parentFolder, documentTitle);
			
				//check is document exists in JDO
				Iterator i=km_result.iterator();
				while (i.hasNext()) {
					IDependentObject doc = createDependentObjectFromHeader(arg0, (KMNodeHeader)i.next());
					documentId = (String)doc.getProperty("documentId");
								
					//check if such document is already created in JDO				
					Collection jdo_result = 
						jdoDataAccessService.query(arg0, createQueryToJDOSearch(documentId, parentFolder), operationName);
				
					if (jdo_result.size()>0) {
						result.add( loadObjectFromKM(
										(IDependentObject)jdo_result.iterator().next() ));
					} else {
						createInJDO(doc);
						result.add( loadObjectFromKM(doc) );
					}
				}				
				
			} catch (Exception ex) {
				location.throwing(ex);
			}
		} else {
			//Not specified search
			result = loadObjectsFromKM( jdoDataAccessService.query(arg0, arg1, operationName) );
		}
		
		return result;
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#query(java.lang.String, java.lang.Class, com.sap.caf.rt.bol.util.IntQueryFilter[])
	 */
	public Collection query(String arg0, Class arg1, IntQueryFilter[] arg2, String operationName)
		throws DataAccessException {
		return loadObjectsFromKM( jdoDataAccessService.query(arg0, arg1, arg2, operationName) );
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#load(java.lang.Object, java.lang.Class)
	 */
	public Object load(Object arg0, Class arg1) throws DataAccessException {
		String method = JARM_REQUEST + ":load(String, Class)";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, new Object[] { arg0}, CAFPublicLogger.LEVEL_MEDIUM);
		try {
			IDependentObject obj = (IDependentObject)jdoDataAccessService.load(arg0, arg1);
			
			//create new instance of Class in order to get instance that not registered in JDO manager
			IDependentObject newObj = createDependentObject(arg1, 
															obj.getProperty("documentId"),
															obj.getProperty("parentFolder"),
															obj.getProperty("key"));
			 
			//load document from km			
			newObj = loadObjectFromKM(newObj);
						
			return newObj;
			
		} catch(Exception ex) {
				location.throwing(ex);
				throw new DataAccessException(ex);		
		} finally {
				CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
		}			
	}

	/**
	 * The method loads document from JDO only without quiries to KM 
	 */
	public Object loadFromJDO(Class arg0, String parentFolder, String documentId) throws DataAccessException {
							
			try {
 				//check if such document is already created in JDO				
				Collection jdo_result = 
						jdoDataAccessService.query(arg0, createQueryToJDOSearch(documentId, parentFolder), null);

				IDependentObject doc = null;						
				if (jdo_result.size()>0) {
						doc = (IDependentObject)jdo_result.iterator().next();
				}
				return doc;												
			} catch (Exception ex) {
				location.throwing(ex);
				throw new DataAccessException(ex);
			}
	}
	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#store(com.sap.caf.rt.bol.IDependentObject)
	 */
	public void store(IDependentObject arg0) throws DataAccessException {
		String method = JARM_REQUEST + ":store(IDependentObject)";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, new Object[] { arg0.getKey()}, CAFPublicLogger.LEVEL_MEDIUM);
		try {

				jdoDataAccessService.store(arg0);
		
				KMDataAccessHelper.getDocumentProxyService()
					.saveDocumentHeader((String)arg0.getProperty("documentId"), 
										(String)arg0.getProperty("parentFolder"), 
										(String)arg0.getProperty("title"), 
										(String)arg0.getProperty("description"), 
										(String)arg0.getProperty("link"),
										(Collection)arg0.getProperty("relatedObjects"));
										
				//store document content
				if (m_sDocContentName.equals(arg0.getObjectType())) {
					uploadDocumentContent(arg0);
				}
				
		} catch (Exception ex) {
				location.throwing(ex);
				throw new DataAccessException(ex);
		} finally {
			CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
		}
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#activate()
	 */
	public void activate() {
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#passivate()
	 */
	public void passivate() throws DataAccessException {
	}

	/**
	 * @see com.sap.caf.rt.bol.da.IDataAccessService#execute(java.lang.String, java.lang.Class, java.lang.Object, java.lang.Object)
	 */
	public Object execute(String arg0, Class arg1, Object arg2, Object arg3)
		throws DataAccessException {
		return jdoDataAccessService.execute(arg0, arg1, arg2, arg3);
	}

	/**
	 * Load collection of Document from KM
	 * 
	 * @param col	Collection of Document's with specified parentFolder 
	 * 				and documentId attributes.
	 * @return		Collection of Documents
	 * @throws DataAccessException
	 */	
	private Collection loadObjectsFromKM(Collection col) throws DataAccessException {
		Iterator i = col.iterator();
		Collection result = new ArrayList();
		while (i.hasNext()) {
			IDependentObject obj = loadObjectFromKM((IDependentObject)i.next());
			if (obj!=null)
			result.add(obj);
		}
		return result;
	}
	
	/**
	 * Load Document from KM
	 * 
	 * @param obj	Instance of Document with specified parentFolder 
	 * 				and documentId attributes 
	 */
	private IDependentObject loadObjectFromKM(IDependentObject obj) throws DataAccessException {
	String method = JARM_REQUEST + ":loadObjectFromKM(IDependentObject)";
	CAFPublicLogger.entering(null, JARM_REQUEST, method, location, new Object[] { obj.getKey()}, CAFPublicLogger.LEVEL_DETAIL);
		
	try {
			NodeHeader header = 
				KMDataAccessHelper.getDocumentProxyService()
					.readDocumentHeader((String)obj.getProperty("documentId"),
										(String)obj.getProperty("parentFolder"));

			//load document atributes
			obj.setProperty("title", header.getDisplayName());
			obj.setProperty("description",header.getDescription());
			obj.setProperty("createdBy", header.getCreatedBy());
			obj.setProperty("createdAt", header.getCreationdate().getTime());
			obj.setProperty("lastChangedBy", header.getLastmodifiedBy());
			obj.setProperty("lastChangedAt", header.getLastmodifieddate().getTime());
			obj.setProperty("link", header.getLink());
			obj.setProperty("relatedObjects", header.getRelationRids());
	
			//load document content if DocumentContent is requested
			if (m_sDocContentName.equals(obj.getObjectType())) {
				NodeContent data = KMDataAccessHelper
										.getDocumentProxyService() 
											.readDocumentContent((String)obj.getProperty("documentId"), 
																 (String)obj.getProperty("parentFolder"));
					
				obj.setProperty("contentEncoding", data.getContentEncoding());
				obj.setProperty("contentType", data.getContentType());
				obj.setProperty("contentLength", new Long(data.getContentLength()));				
				obj.setProperty("content", data.getContent());					
			}

			return obj;
		}
		catch (Exception ex) {
			location.throwing(ex);
			throw new DataAccessException(ex);
		} finally {
			CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_DETAIL);
		}
	}
	
	/**
	 * Uploads document content into KM
	 * 
	 * @param arg0	DocumentContent object Instance to upload
	 */
	private void uploadDocumentContent(IDependentObject arg0) throws KMNodeException, CAFFindException {
			KMDataAccessHelper.getDocumentProxyService()
				.saveDocumentContent((String)arg0.getProperty("documentId"),
									 (String)arg0.getProperty("parentFolder"),
									 ((Long)arg0.getProperty("contentLength")).longValue(),
									 (String)arg0.getProperty("contentType"),
									 (String)arg0.getProperty("contentEncoding"),
									 (byte[])arg0.getProperty("content"));																
	}
	
	/**
	 * Create object in JDO if it doesn't exist
	 * 
	 * @param arg0	Object to create
	 * @throws DataAccessException
	 */
	private void createInJDOIfNotExists(IDependentObject arg0) throws DataAccessException {
		IDependentObject obj=null;
		try {
			obj = (IDependentObject) jdoDataAccessService.findByPrimaryKey(arg0.getPK(), arg0.getClass());
			//TODO there should be catch for ObjectNotFound Exception
		} catch (DataAccessException ex) {
			location.throwing(ex);
		}
		
		if (obj==null) {
			jdoDataAccessService.create(arg0);
		}
	}
	
	/**
	 * Create object in JDO. 
	 * The method generates KEY for object.
	 * 
	 * @param arg0	Object to create
	 * @throws DataAccessException
	 */
	private void createInJDO(IDependentObject arg0) throws DataAccessException {
		arg0.setProperty("key", PrimaryKeyFactory.getInstance().getPrimaryKey());
		jdoDataAccessService.create(arg0);
	}
	
	/**
	 * Create IDependentObject Object with specified documentId and parentFolder attributes
	 * 
	 * @param arg0			Class to create
	 * @param documentId	Value of documentId attribute
	 * @param parentFolder	Value of parentFolder attribute
	 * @param key			Value of key attribute
	 */
	private IDependentObject createDependentObject(Class arg0, Object documentId, Object parentFolder, Object key) throws InstantiationException, IllegalAccessException {
		IDependentObject obj = (IDependentObject)arg0.newInstance();		
		obj.setProperty("documentId", (String)documentId );
		obj.setProperty("parentFolder", (String)parentFolder );
		obj.setProperty("key", (String)key );
		
		return obj;											
	}
	
	/**
	 * Create query filters to JDO search for document by 
	 * parentFolder and documentId
	 * 
	 * @param documentId		KM Document Id to search
	 * @param parentFolder		KM Folder RID to search in
	 * @return
	 */
	private static IntQueryFilter[] createQueryToJDOSearch(String documentId, String parentFolder) {
		IntQueryFilter fParentFolder = new IntQueryFilter(QueryFilter.createStringFilter(
							QueryFilter.ACTION_EXACT, parentFolder, "parentFolder", true));
		fParentFolder.setAttribute("parentFolder");					
		fParentFolder.condition = QueryFilter.CONDITION_EQ;
						
		IntQueryFilter fDocumentId = new IntQueryFilter(QueryFilter.createStringFilter(
							QueryFilter.ACTION_EXACT, documentId, "documentId", true));
		fDocumentId.setAttribute("documentId");
		fDocumentId.condition = QueryFilter.CONDITION_EQ;
		
		return new IntQueryFilter[] {fParentFolder, fDocumentId}; 
	}
	
	/**
	 * The method converts KMNodeHeader to IDependentObject
	 * 
	 * @param arg0		Class of IDependentObject to create
	 * @param header	KMNodeHeader
	 * @return			Created IDependentObject with attributes from KMNodeHeader 
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 */
	private IDependentObject createDependentObjectFromHeader(Class arg0, KMNodeHeader header) throws IllegalAccessException, InstantiationException {
		IDependentObject obj = (IDependentObject)arg0.newInstance();
		
		String rid = header.getRid();
		obj.setProperty("documentId", RidUtils.getName(rid));
		obj.setProperty("parentFolder", RidUtils.getPath(rid));
		 
		return obj;
	}
	
	public void setUserContext(UserContext userCtx) {this.userContext = userCtx;}
	
}